// Filename: v2.rs
// Version:	 0.4
// Date:	 18-11-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Data structures for file version 2.

use super::v0::FetchConfig as V0FetchConfig;
use crate::{component_config, CPUModel as NewModel, InstructionLatencies};

use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Clone, Default, Deserialize, Serialize)]
pub struct CPUModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// The ALU latency of each instruction class.
    pub execution_latencies: InstructionLatencies,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: V0FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: component_config::MemoryControllerConfig,
    /// Caches of the memory hierarchy.
    pub caches: Vec<component_config::CacheConfig>,
    /// Configuration for the decode stage of the front end.
    pub decode_config: component_config::DecodeConfig,
    /// How many cycles the dispatch unit needs to dispatch an instruction.
    pub dispatch_cycle_count: usize,
    /// Configuration of the register file.
    pub reg_file_config: component_config::RegisterFileConfig,
    /// Configuration of schedulers.
    pub schedulers: Vec<component_config::SchedulerConfig>,
    /// Configuration for each pipeline from front end to the last stage.
    pub pipelines: Vec<PipelineConfig>,
    /// Configuration of the load-store-unit.
    pub load_store_unit_config: component_config::LoadStoreUnitConfig,
}

/// The properties of a CPU pipeline.
///
/// This models the complete execution back-end of a CPU, including register file access, an ALU
/// and/or AGU and access to the Load-Store-Unit.
/// As there can be many different things happening in the back-end, a pipeline is highly
/// configurable.
#[derive(Clone, Deserialize, Serialize)]
pub struct PipelineConfig {
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// How many clock cycles are dedicated to reading register data.
    pub register_read_cycles: usize,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// Which types of ALU instructions the pipeline can support, if any.
    pub alu: Option<component_config::AluCapabilities>,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline can serve ALU instructions that also have a memory operand. If this is the
    /// case, there needs to be an AGU and the memory stage is set before the ALU stage.
    pub fused_memory_alu: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

impl From<CPUModel> for NewModel {
    fn from(old_model: CPUModel) -> Self {
        let CPUModel {
            max_vector_size,
            execution_latencies,
            fetch_config,
            memory_controller_config,
            caches,
            decode_config,
            dispatch_cycle_count,
            reg_file_config,
            schedulers,
            pipelines,
            load_store_unit_config,
        } = old_model;

        let pipelines = pipelines
            .into_iter()
            .map(component_config::PipelineConfig::from)
            .collect();

        NewModel {
            max_vector_size,
            execution_latencies,
            fetch_config: fetch_config.into(),
            memory_controller_config,
            caches,
            decode_config,
            reorder_buffer_config: component_config::ReorderBufferConfig {
                size: 128,
                retire_width: 1,
                branch_retire_width: 1,
            },
            dispatch_cycle_count,
            reg_file_config,
            schedulers,
            pipelines,
            load_store_unit_config,
        }
    }
}

impl From<PipelineConfig> for component_config::PipelineConfig {
    fn from(old: PipelineConfig) -> Self {
        let PipelineConfig {
            scheduler_id,
            register_read_cycles,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu: _,
            renaming,
        } = old;

        component_config::PipelineConfig {
            scheduler_id,
            register_read_cycles,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            renaming,
        }
    }
}
