// Filename: fetch.rs
// Version:	 0.5
// Date:	 01-12-2021 (DD-MM-YYYY)
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Configuration for the fetch unit.

use serde::{Deserialize, Serialize};

/// Properties of the fetch stage.
///
/// It is responsible for fetching program data from memory.
#[derive(Clone, Deserialize, Serialize)]
pub struct FetchConfig {
    /// The ID of the cache used to fetch instruction data. If no ID is given, connects directly
    /// to the memory controller.
    pub cache_connection: Option<usize>,
    /// Primary branch predictor. Predicts every branch with no additional latency.
    pub primary_predictor: BranchPredictor,
    /// Secondary branch predictor. Predicts every branch with settable latency. Its result has
    /// priority over the primary predictor.
    pub secondary_predictor: Option<(BranchPredictor, usize)>,
    /// If branch resolution in the front end is enabled.
    ///
    /// After loading the data from cache, the front end will attempt calculate the branch address
    /// if it isn't dependent on registers or memory. Unconditional jumps will cause a redirection,
    /// if not correctly predicted already.
    pub branch_resolution: bool,
}

/// Properties of a branch predictor.
#[derive(Clone, Deserialize, Serialize)]
pub enum BranchPredictor {
    /// No branch predictor. Make no prediction.
    None,
    /// Has a static probability to return the correct prediction.
    MagicProbability {
        /// Chance between 0 and 1 for correctly predicting a direct jump or call.
        direct_chance: f64,
        /// Chance between 0 and 1 for correctly predicting a conditional jump.
        conditional_chance: f64,
        /// Chance between 0 and 1 for correctly predicting an indirect jump or call target address.
        address_chance: f64,
        /// Chance between 0 and 1 for correctly predicting a return instruction.
        return_chance: f64,
    },
    /// Multi-part branch predictor.
    BranchTargetBuffer {
        /// The capacity of saved branches.
        entries: usize,
        /// If bi-modal tables are used to predict conditional branches.
        bimodal_prediction: bool,
        /// Limit of call depth. Unlimited if not set.
        return_address_stack_size: Option<usize>,
    },
    /// TAGE branch predictor. Doesn't provide addresses. Only predicts conditional branches.
    ///
    /// Has multiple tables. Each table uses double the history length of the last, starting at two.
    TaggedGeometric(Vec<TaggedGeometricTable>),
}

/// The parameters for a table of the TAGE branch predictor.
#[derive(Clone, Deserialize, Serialize)]
pub struct TaggedGeometricTable {
    /// How many entries for the given branch history length are saved.
    pub entries: usize,
    /// How many bits are used to tag each entry.
    ///
    /// A lower number means more potential collisions for branch addresses and/or histories.
    pub tag_size: usize,
}

impl Default for FetchConfig {
    fn default() -> Self {
        FetchConfig {
            cache_connection: None,
            primary_predictor: BranchPredictor::None,
            secondary_predictor: None,
            branch_resolution: false
        }
    }
}
