// Filename: v1.rs
// Version:	 0.1
// Date:	 24-09-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Data structures for file version 1.

use crate::{component_config, CPUModel as NewModel};

use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Default, Deserialize, Serialize)]
pub struct CPUModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// The ALU latency of each instruction class.
    pub execution_latencies: crate::InstructionLatencies,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: component_config::FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: component_config::MemoryControllerConfig,
    /// Caches of the memory hierarchy.
    pub caches: Vec<component_config::CacheConfig>,
    /// The amount of parallel decoders.
    pub decoder_count: usize,
    /// Configuration of the register file.
    pub reg_file_config: component_config::RegisterFileConfig,
    /// Configuration of schedulers.
    pub schedulers: Vec<component_config::SchedulerConfig>,
    /// Configuration for each pipeline from front end to the last stage.
    pub pipelines: Vec<super::v1::PipelineConfig>,
}

/// The properties of a CPU pipeline.
///
/// For more information, see [current `PipelineConfig`](component_config::PipelineConfig).
#[derive(Deserialize, Serialize)]
pub struct PipelineConfig {
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// Which types of ALU instructions the pipeline can support, if any.
    pub alu: Option<component_config::AluCapabilities>,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline can serve ALU instructions that also have a memory operand. If this is the
    /// case, there needs to be an AGU and the memory stage is set before the ALU stage.
    pub fused_memory_alu: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

impl From<CPUModel> for NewModel {
    fn from(old: CPUModel) -> Self {
        let CPUModel {
            max_vector_size,
            execution_latencies,
            fetch_config,
            memory_controller_config,
            caches,
            decoder_count,
            reg_file_config,
            schedulers,
            pipelines,
        } = old;

        let pipelines = pipelines
            .into_iter()
            .map(component_config::PipelineConfig::from)
            .collect();

        Self {
            max_vector_size,
            execution_latencies,
            fetch_config,
            memory_controller_config,
            caches,
            decode_config: component_config::DecodeConfig {
                width: decoder_count,
                pre_decode_buffer_size: 4,
            },
            dispatch_cycle_count: 0,
            reg_file_config,
            schedulers,
            pipelines,
            load_store_unit_config: component_config::LoadStoreUnitConfig {
                load_count: 16,
                store_count: 16,
                cache_connection: None,
            },
        }
    }
}

impl From<PipelineConfig> for component_config::PipelineConfig {
    fn from(old: PipelineConfig) -> Self {
        let PipelineConfig {
            scheduler_id,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        } = old;

        component_config::PipelineConfig {
            scheduler_id,
            register_read_cycles: 0,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        }
    }
}
