// Filename: v0.rs
// Version:	 0.1
// Date:	 24-09-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Data structures for file version 0.

use super::v1::CPUModel as V1Model;
use crate::{component_config, CPUModel as NewModel};

use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Deserialize, Serialize)]
pub struct CPUModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: component_config::FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: component_config::MemoryControllerConfig,
    /// Caches of the memory hierarchy.
    pub caches: Vec<component_config::CacheConfig>,
    /// Configuration for each pipeline from front end to the last stage.
    pub pipelines: Vec<PipelineConfig>,
}

/// The properties of a CPU pipeline.
///
/// For more information, see [current `PipelineConfig`](component_config::PipelineConfig).
#[derive(Deserialize, Serialize)]
pub struct PipelineConfig {
    /// The ALU latency of each instruction class.
    pub execution_latencies: InstructionLatencies,
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// If the pipeline can execute ALU instructions.
    pub alu: bool,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline can serve ALU instructions that also have a memory operand. If this is the
    /// case, there needs to be an AGU and the memory stage is set before the ALU stage.
    pub fused_memory_alu: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

/// Defines the amount of clock cycles each instruction type needs.
///
/// The fields of this struct are used to selectively overwrite default values. If a field is
/// `None`, the default value is used.
///
/// A value of zero means that the type isn't supported in a pipeline. By default, all types are
/// supported.
///
/// This got replaced by model-wide instruction latencies.
#[derive(Default, Deserialize, Serialize)]
pub struct InstructionLatencies {
    /// Simple addition.
    pub integer_add: Option<u16>,
    /// Typically takes multiple, but few clock cycles.
    pub integer_multiply: Option<u16>,
    /// Takes many clock cycles, usually dependent on the operand size.
    pub integer_divide: Option<u16>,
    /// Combination of add- and mul-operation.
    pub integer_multiply_add: Option<u16>,
    /// Completes fast, but needs a shifter.
    pub integer_shift: Option<u16>,
    /// Might take longer than floating point multiplication.
    pub float_add: Option<u16>,
    /// Usually as fast as integer multiplication.
    pub float_multiply: Option<u16>,
    /// Takes many clock cycles, usually dependent on the operand size.
    pub float_divide: Option<u16>,
    /// Might need multiple execution ports if a design has separate add- and mul-pipes.
    pub float_multiply_add: Option<u16>,
    /// Usually completes fast, but needs a branch unit.
    pub branch: Option<u16>,
}

impl From<CPUModel> for NewModel {
    fn from(old: CPUModel) -> Self {
        let CPUModel {
            max_vector_size,
            fetch_config,
            memory_controller_config,
            caches,
            pipelines,
        } = old;

        let pipelines = pipelines
            .into_iter()
            .map(super::v1::PipelineConfig::from)
            .collect();

        V1Model {
            max_vector_size,
            execution_latencies: Default::default(),
            fetch_config,
            decoder_count: 1,
            reg_file_config: Default::default(),
            memory_controller_config,
            caches,
            pipelines,
            schedulers: Vec::new(),
        }
        .into()
    }
}

impl From<PipelineConfig> for super::v1::PipelineConfig {
    fn from(other: PipelineConfig) -> Self {
        let PipelineConfig {
            execution_latencies,
            scheduler_id,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        } = other;

        let alu = if alu {
            Some(component_config::AluCapabilities {
                integer_add: execution_latencies.integer_add.unwrap_or(1) > 0,
                integer_mul: execution_latencies.integer_multiply.unwrap_or(1) > 0,
                integer_divide: execution_latencies.integer_divide.unwrap_or(1) > 0,
                integer_shift: execution_latencies.integer_shift.unwrap_or(1) > 0,
                float_add: execution_latencies.float_add.unwrap_or(1) > 0,
                float_mul: execution_latencies.float_multiply.unwrap_or(1) > 0,
                float_divide: execution_latencies.float_divide.unwrap_or(1) > 0,
                branch: execution_latencies.branch.unwrap_or(1) > 0,
            })
        } else {
            None
        };

        Self {
            scheduler_id,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        }
    }
}
