// Filename: pipeline.rs
// Version:	 0.4
// Date:	 15-09-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Configuration for pipelines, coming after the front end of fetch and decode.

use serde::{Deserialize, Serialize};

/// The properties of a CPU pipeline.
///
/// This models the complete execution back-end of a CPU, including register file access, an ALU
/// and/or AGU and access to the Load-Store-Unit.
/// As there can be many different things happening in the back-end, a pipeline is highly
/// configurable.
#[derive(Deserialize, Serialize)]
pub struct PipelineConfig {
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// Which types of ALU instructions the pipeline can support, if any.
    pub alu: Option<AluCapabilities>,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline can serve ALU instructions that also have a memory operand. If this is the
    /// case, there needs to be an AGU and the memory stage is set before the ALU stage.
    pub fused_memory_alu: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

/// A flag for each instruction class, defining if an ALU supports it.
#[derive(Deserialize, Serialize)]
pub struct AluCapabilities {
    /// Simple integer addition.
    pub integer_add: bool,
    /// Has multiplication support.
    pub integer_mul: bool,
    /// Has integer division support.
    pub integer_divide: bool,
    /// Provides a shift unit.
    pub integer_shift: bool,
    /// Support for floating point addition.
    pub float_add: bool,
    /// Support for floating point multiplication.
    pub float_mul: bool,
    /// Support for floating point division.
    pub float_divide: bool,
    /// Provides a branch unit.
    pub branch: bool,
}
