// Filename: legacy.rs
// Version:	 0.5
// Date:	 15-09-2021 (DD-MM-YYYY)
// Library:  gpcas_cpu_model
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Old compatible versions of data structures as well as upgrade-to-current functionality.

use crate::{component_config, CPUModel};

use crate::component_config::PipelineConfig;
use gpcas_base::file::OldGpcasFileStruct;
use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Deserialize, Serialize)]
pub struct CPUModelV0 {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: component_config::FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: component_config::MemoryControllerConfig,
    /// Caches of the memory hierarchy.
    pub caches: Vec<component_config::CacheConfig>,
    /// Configuration for each pipeline from front end to the last stage.
    pub pipelines: Vec<PipelineConfigV0>,
}

/// The properties of a CPU pipeline.
///
/// This models the complete execution back-end of a CPU, including register file access, an ALU
/// and/or AGU and access to the Load-Store-Unit.
/// As there can be many different things happening in the back-end, a pipeline is highly
/// configurable.
#[derive(Deserialize, Serialize)]
pub struct PipelineConfigV0 {
    /// The ALU latency of each instruction class.
    pub execution_latencies: InstructionLatenciesV0,
    /// The ID of the scheduler the pipeline uses, if any. Multiple pipelines with the same
    /// scheduler ID share a scheduler among them.
    pub scheduler_id: Option<usize>,
    /// If the pipeline can execute instructions exclusively using the general purpose register
    /// file.
    pub general_purpose: bool,
    /// If the pipeline can execute instructions using the vector register file.
    pub vector: bool,
    /// If the pipeline can execute ALU instructions.
    pub alu: bool,
    /// If the pipeline can serve memory read operands.
    pub memory_load: bool,
    /// If the pipeline can serve memory write operands.
    pub memory_store: bool,
    /// If the pipeline can serve ALU instructions that also have a memory operand. If this is the
    /// case, there needs to be an AGU and the memory stage is set before the ALU stage.
    pub fused_memory_alu: bool,
    /// If the pipeline makes use of register renaming. If it doesn't, an instruction can only be
    /// issued to this pipeline if every register operand is ready and the output register doesn't
    /// have an instruction already wanting to write to it.
    pub renaming: bool,
}

/// Defines the amount of clock cycles each instruction type needs.
///
/// The fields of this struct are used to selectively overwrite default values. If a field is
/// `None`, the default value is used.
///
/// A value of zero means that the type isn't supported in a pipeline. By default, all types are
/// supported.
#[derive(Default, Deserialize, Serialize)]
pub struct InstructionLatenciesV0 {
    /// Simple addition.
    pub integer_add: Option<u16>,
    /// Typically takes multiple, but few clock cycles.
    pub integer_multiply: Option<u16>,
    /// Takes many clock cycles, usually dependent on the operand size.
    pub integer_divide: Option<u16>,
    /// Combination of add- and mul-operation.
    pub integer_multiply_add: Option<u16>,
    /// Completes fast, but needs a shifter.
    pub integer_shift: Option<u16>,
    /// Might take longer than floating point multiplication.
    pub float_add: Option<u16>,
    /// Usually as fast as integer multiplication.
    pub float_multiply: Option<u16>,
    /// Takes many clock cycles, usually dependent on the operand size.
    pub float_divide: Option<u16>,
    /// Might need multiple execution ports if a design has separate add- and mul-pipes.
    pub float_multiply_add: Option<u16>,
    /// Usually completes fast, but needs a branch unit.
    pub branch: Option<u16>,
}

impl OldGpcasFileStruct<CPUModel> for CPUModelV0 {
    fn upgrade(self) -> CPUModel {
        let CPUModelV0 {
            max_vector_size,
            fetch_config,
            memory_controller_config,
            caches,
            pipelines,
        } = self;

        let pipelines = pipelines.into_iter().map(PipelineConfig::from).collect();

        CPUModel {
            max_vector_size,
            execution_latencies: Default::default(),
            fetch_config,
            decoder_count: 1,
            memory_controller_config,
            caches,
            pipelines,
        }
    }
}

impl From<PipelineConfigV0> for PipelineConfig {
    fn from(other: PipelineConfigV0) -> Self {
        let PipelineConfigV0 {
            execution_latencies,
            scheduler_id,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        } = other;

        let alu = if alu {
            Some(component_config::AluCapabilities {
                integer_add: execution_latencies.integer_add.unwrap_or(1) > 0,
                integer_mul: execution_latencies.integer_multiply.unwrap_or(1) > 0,
                integer_divide: execution_latencies.integer_divide.unwrap_or(1) > 0,
                integer_shift: execution_latencies.integer_shift.unwrap_or(1) > 0,
                float_add: execution_latencies.float_add.unwrap_or(1) > 0,
                float_mul: execution_latencies.float_multiply.unwrap_or(1) > 0,
                float_divide: execution_latencies.float_divide.unwrap_or(1) > 0,
                branch: execution_latencies.branch.unwrap_or(1) > 0,
            })
        } else {
            None
        };

        PipelineConfig {
            scheduler_id,
            general_purpose,
            vector,
            alu,
            memory_load,
            memory_store,
            fused_memory_alu,
            renaming,
        }
    }
}
