// Filename: lib.rs
// Version:	 0.7
// Date:	 27-07-2021 (DD-MM-YYYY)
//
// Copyright (c) 2021 Kai Rese
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this program. If not, see
// <https://www.gnu.org/licenses/>.

//! Defines the structure of an abstract CPU model and some helpful functions for working with it,
//! as well as the corresponding file structure.
//!
//! Any application working with the model should use the structs and definitions in this crate
//! to ensure compatibility.
#![warn(missing_docs)]

pub mod component_config;
mod legacy;

use gpcas_base::file::{DeserializeFunction, GpcasFileStruct};
use serde::{Deserialize, Serialize};

/// The configuration of an abstract CPU model.
#[derive(Default, Deserialize, Serialize)]
pub struct CPUModel {
    /// The maximum supported vector size of the model in bits.
    pub max_vector_size: usize,
    /// Configuration for the fetch stage of the front end.
    pub fetch_config: component_config::FetchConfig,
    /// Configuration for the memory controller.
    pub memory_controller_config: component_config::MemoryControllerConfig,
    /// Caches of the memory hierarchy.
    pub caches: Vec<component_config::CacheConfig>,
    /// Configuration for each pipeline from front end to the last stage.
    pub pipelines: Vec<component_config::PipelineConfig>,
}

impl CPUModel {
    /// Creates a new model without components and default global values.
    pub fn new() -> Self {
        CPUModel {
            max_vector_size: 128,
            fetch_config: component_config::FetchConfig::default(),
            memory_controller_config: component_config::MemoryControllerConfig::default(),
            caches: Vec::new(),
            pipelines: Vec::new(),
        }
    }
}

impl GpcasFileStruct for CPUModel {
    const FILE_IDENTIFIER: &'static str = "gpcas::cpu_model";
    const CURRENT_FILE_VERSION: usize = 0;
    const COMPATIBLE_VERSIONS: &'static [(usize, DeserializeFunction<CPUModel>)] = &[];
}
