//! Validates accounts structs.

use crate::*;
use vipers::{assert_keys_eq, invariant, unwrap_int, Validate};

impl<'info> Validate<'info> for CreateGovernor<'info> {
    fn validate(&self) -> ProgramResult {
        invariant!(
            self.smart_wallet.owners.contains(&self.governor.key()),
            SmartWalletNotFound
        );

        Ok(())
    }
}

impl<'info> Validate<'info> for CreateProposal<'info> {
    fn validate(&self) -> ProgramResult {
        Ok(())
    }
}

impl<'info> Validate<'info> for ActivateProposal<'info> {
    fn validate(&self) -> ProgramResult {
        invariant!(
            self.proposal.get_state(&self.governor)? == ProposalState::Draft,
            "proposal must be draft to be activated"
        );
        assert_keys_eq!(self.governor, self.proposal.governor);
        invariant!(
            unwrap_int!(
                (self.proposal.created_at as u64).checked_add(self.governor.params.voting_delay)
            ) < Clock::get()?.unix_timestamp as u64,
            VotingDelayNotMet
        );

        assert_keys_eq!(self.electorate, self.governor.electorate);

        Ok(())
    }
}

impl<'info> Validate<'info> for CancelProposal<'info> {
    fn validate(&self) -> ProgramResult {
        invariant!(
            self.proposal.get_state(&self.governor)? == ProposalState::Draft,
            "proposal must be draft to be cancelled"
        );
        assert_keys_eq!(
            self.proposer,
            self.proposal.proposer,
            "proposer should match recorded"
        );
        assert_keys_eq!(
            self.governor,
            self.proposal.governor,
            "proposal should be under the governor"
        );
        Ok(())
    }
}

impl<'info> Validate<'info> for QueueProposal<'info> {
    fn validate(&self) -> ProgramResult {
        assert_keys_eq!(self.governor, self.proposal.governor);
        invariant!(
            self.proposal.get_state(&self.governor)? == ProposalState::Succeeded,
            "proposal must be succeeded to be queued"
        );
        assert_keys_eq!(
            self.governor,
            self.proposal.governor,
            "proposal should be under the governor"
        );

        assert_keys_eq!(
            self.smart_wallet,
            self.governor.smart_wallet,
            "smart wallet should match"
        );
        Ok(())
    }
}

impl<'info> Validate<'info> for NewVote<'info> {
    fn validate(&self) -> ProgramResult {
        Ok(())
    }
}

impl<'info> Validate<'info> for SetVote<'info> {
    fn validate(&self) -> ProgramResult {
        assert_keys_eq!(self.governor.electorate, self.electorate);
        invariant!(
            self.proposal.get_state(&self.governor)? == ProposalState::Active,
            "proposal must be active to be voted on"
        );
        assert_keys_eq!(
            self.governor,
            self.proposal.governor,
            "proposal should be under the governor"
        );
        assert_keys_eq!(
            self.vote.proposal,
            self.proposal,
            "receipt proposal should match"
        );
        Ok(())
    }
}

impl<'info> Validate<'info> for CreateProposalMeta<'info> {
    fn validate(&self) -> ProgramResult {
        assert_keys_eq!(self.proposer, self.proposal.proposer);
        Ok(())
    }
}
