//! Handles proposal creation, lifecycle, voting, and execution.
#![deny(rustdoc::all)]
#![allow(rustdoc::missing_doc_code_examples)]

use anchor_lang::prelude::*;
use smart_wallet::SmartWallet;
use std::convert::TryInto;
use vipers::invariant;
use vipers::unwrap_int;
use vipers::Validate;

mod account_structs;
mod account_validators;
mod events;
pub mod proposal;
mod state;

use account_structs::*;

pub use events::*;
pub use proposal::*;
pub use state::*;

declare_id!("RLMVZzDAFw8aWZiwBzBmcddtcs2d3shtiuYEKJ3ptwj");

/// The [govern] program.
#[program]
pub mod govern {
    use super::*;

    /// Creates a [Governor].
    #[access_control(ctx.accounts.validate())]
    pub fn create_governor(
        ctx: Context<CreateGovernor>,
        bump: u8,
        electorate: Pubkey,
        params: GovernanceParameters,
    ) -> ProgramResult {
        msg!("Instruction: create_governor");

        invariant!(
            params.timelock_delay_seconds >= 0,
            "timelock delay must be at least 0 seconds"
        );

        let governor = &mut ctx.accounts.governor;
        governor.base = ctx.accounts.base.key();
        governor.bump = bump;

        governor.proposal_count = 0;
        governor.electorate = electorate;
        governor.smart_wallet = ctx.accounts.smart_wallet.key();

        governor.params = params;

        emit!(GovernorCreateEvent {
            governor: governor.key(),
            electorate,
            smart_wallet: ctx.accounts.smart_wallet.key(),
            parameters: params,
        });

        Ok(())
    }

    /// Creates a [Proposal].
    /// This may be called by anyone, since the [Proposal] does not do anything until
    /// it is activated in [activate_proposal].
    #[access_control(ctx.accounts.validate())]
    pub fn create_proposal(
        ctx: Context<CreateProposal>,
        bump: u8,
        instructions: Vec<ProposalInstruction>,
    ) -> ProgramResult {
        msg!("Instruction: create_proposal");

        let governor = &mut ctx.accounts.governor;

        let proposal = &mut ctx.accounts.proposal;
        proposal.governor = governor.key();
        proposal.index = governor.proposal_count;
        proposal.bump = bump;

        proposal.proposer = ctx.accounts.proposer.key();

        proposal.created_at = Clock::get()?.unix_timestamp;
        proposal.activated_at = -1;
        proposal.voting_ends_at = -1;

        proposal.queued_at = -1;
        proposal.queued_transaction = Pubkey::default();

        proposal.instructions = instructions.clone();

        governor.proposal_count += 1;

        emit!(ProposalCreateEvent {
            governor: governor.key(),
            proposal: proposal.key(),
            index: proposal.index,
            instructions,
        });

        Ok(())
    }

    /// Activates a proposal.
    /// Only the [Governor::electorate] may call this; that program
    /// may ensure that only certain types of users can activate proposals.
    #[access_control(ctx.accounts.validate())]
    pub fn activate_proposal(ctx: Context<ActivateProposal>) -> ProgramResult {
        msg!("Instruction: activate_proposal");

        let proposal = &mut ctx.accounts.proposal;
        let now = Clock::get()?.unix_timestamp;
        proposal.activated_at = now;
        proposal.voting_ends_at = unwrap_int!(ctx
            .accounts
            .governor
            .params
            .voting_period
            .try_into()
            .ok()
            .and_then(|v: i64| now.checked_add(v)));

        emit!(ProposalActivateEvent {
            governor: proposal.governor.key(),
            proposal: proposal.key(),
        });

        Ok(())
    }

    /// Cancels a proposal.
    /// This is only callable by the creator of the proposal.
    #[access_control(ctx.accounts.validate())]
    pub fn cancel_proposal(ctx: Context<CancelProposal>) -> ProgramResult {
        msg!("Instruction: cancel_proposal");

        let proposal = &mut ctx.accounts.proposal;
        proposal.canceled = true;

        emit!(ProposalCancelEvent {
            governor: proposal.governor.key(),
            proposal: proposal.key(),
        });

        Ok(())
    }

    /// Queues a proposal for execution by the [SmartWallet].
    #[access_control(ctx.accounts.validate())]
    pub fn queue_proposal(ctx: Context<QueueProposal>, tx_bump: u8) -> ProgramResult {
        msg!("Instruction: queue_proposal");

        ctx.accounts.queue_transaction(tx_bump)?;

        emit!(ProposalQueueEvent {
            governor: ctx.accounts.proposal.governor.key(),
            proposal: ctx.accounts.proposal.key(),
            transaction: ctx.accounts.transaction.key(),
        });

        Ok(())
    }

    /// Creates a new [Vote]. Anyone can call this.
    #[access_control(ctx.accounts.validate())]
    pub fn new_vote(ctx: Context<NewVote>, bump: u8, voter: Pubkey) -> ProgramResult {
        msg!("Instruction: new_vote");

        let vote = &mut ctx.accounts.vote;
        vote.proposal = ctx.accounts.proposal.key();
        vote.voter = voter;
        vote.bump = bump;

        vote.side = VoteSide::Pending.into();
        vote.weight = 0;

        Ok(())
    }

    /// Sets a [Vote] weight and side.
    /// This may only be called by the [Governor::electorate].
    #[access_control(ctx.accounts.validate())]
    pub fn set_vote(ctx: Context<SetVote>, side: u8, weight: u64) -> ProgramResult {
        msg!("Instruction: set_vote");

        let vote = &ctx.accounts.vote;

        let proposal = &mut ctx.accounts.proposal;
        proposal.subtract_vote_weight(vote.side.try_into()?, vote.weight)?;
        proposal.add_vote_weight(side.try_into()?, weight)?;

        let vote = &mut ctx.accounts.vote;
        vote.side = side;
        vote.weight = weight;

        emit!(VoteSetEvent {
            governor: proposal.governor.key(),
            proposal: proposal.key(),
            voter: vote.voter,
            vote: vote.key(),
            side,
            weight,
        });

        Ok(())
    }
}

/// Errors.
#[error]
pub enum ErrorCode {
    #[msg("Activator is not whitelisted.")]
    ActivatorNotWhitelisted,
    #[msg("Invalid vote side.")]
    InvalidVoteSide,
    #[msg("Invalid proposal state.")]
    InvalidProposalState,
    #[msg("Invalid transaction account.")]
    InvalidTransactionAccount,
    #[msg("The owner of the smart wallet doesn't match with current.")]
    SmartWalletNotFound,
    #[msg("The proposal cannot be activated since it has not yet passed the voting delay.")]
    VotingDelayNotMet,
    #[msg("Only drafts can be cancelled.")]
    ProposalNotDraft,
    #[msg("The proposal must be activated.")]
    ProposalNotActivated,
}
