use clap::{Parser, Subcommand};
use fs_extra::dir::{copy, CopyOptions};
use git2::Repository;
use git_url_parse::GitUrl;
use std::{
  env,
  fs::{create_dir, read_to_string, File},
  io::Write,
  path::{Path, PathBuf},
};
use toml_edit::{value, Document};

const LOCAL_DIR: &str = ".got_modules";
const ADDONS_DIR: &str = "addons";
const GOTPM_FILE: &str = "gotpm.toml";

#[derive(Parser, Debug)]
#[clap(author, version, about, long_about = None)]
struct Cli {
  #[clap(subcommand)]
  command: Option<Commands>,
}

#[derive(Subcommand, Debug)]
enum Commands {
  Init,
  Install,
  Add {
    url: Option<String>,
    #[clap(short, long, default_value_t = String::from("main"))]
    reference: String,
  },
}

fn exit(err: &str) {
  eprintln!("Error: {}", err);
  std::process::exit(1);
}

fn find_package(current_dir: PathBuf, file_name: &str) -> Option<PathBuf> {
  let mut path: PathBuf = current_dir.clone();
  let file = Path::new(file_name);

  loop {
    path.push(file);

    if path.is_file() {
      break Some(path);
    }

    if !(path.pop() && path.pop()) {
      // remove file && remove parent
      break None;
    }
  }
}

fn parse_git_url(url: &str) -> GitUrl {
  let parsed = GitUrl::parse(url);
  let parsed = parsed.unwrap();

  if parsed.host.is_none() && !parsed.git_suffix {
    eprintln!("Error: {}", "Not a valid git URL!");
    std::process::exit(1);
  }

  return parsed;
}

fn install_package(project_dir: PathBuf, name: &str, url: &str, reference: &str) {
  let folder = format!("{}/{}", LOCAL_DIR, name);
  match Repository::clone(url, &folder) {
    Ok(repo) => {
      let (object, reference) = repo
        .revparse_ext(reference)
        .expect("Object not found");
      match reference {
        // gref is an actual reference like branches or tags
        Some(gref) => repo.set_head(gref.name().unwrap()),
        // this is a commit, not a reference
        None => repo.set_head_detached(object.id()),
      }
      .expect("Failed to set HEAD");
    }
    Err(e) => {
      println!("{:?}", e);
    }
  };

  let mut addons_dir = project_dir.clone();
  addons_dir.push(ADDONS_DIR);

  if !addons_dir.is_dir() {
    create_dir(&addons_dir);
  }

  let repo_addons = Path::new(&project_dir)
    .join(&folder)
    .join(ADDONS_DIR);

  let repo_source = if repo_addons.is_dir() {
    repo_addons
  } else {
    repo_addons.parent().unwrap().to_path_buf()
  };
  println!("@{:?}", repo_source);

  let mut copy_options = CopyOptions::new();
  copy_options.content_only = true;
  match copy(repo_source, addons_dir.as_path(), &copy_options) {
    Ok(r) => {
      println!("Ok: {:?}", r);
    }
    Err(err) => {
      println!("Err: {:?}", err);
    }
  }
  println!("{:?}", addons_dir.is_dir());
}

fn main() {
  let cli = Cli::parse();
  let process_dir = env::current_dir().unwrap();

  match &cli.command {
    Some(Commands::Init) => {
      let mut path = process_dir.clone();
      path.push(GOTPM_FILE);
      if path.is_file() {
        exit("You already have a gotpm.toml file in this directory!");
      }

      let template = r#"[dependencies]"#;
      // better organize this
      let mut doc = template.parse::<Document>();
      let file_name = path.parent().unwrap().join("gotpm.toml");
      let mut file = File::create(file_name).unwrap();
      write!(file, "{}", doc.unwrap().to_string());
    }
    Some(Commands::Install) => {
      let got_file = find_package(process_dir.clone(), GOTPM_FILE);
      if got_file.is_none() {
        exit("This is not a gotpm project! Consider running the init command first.");
      }
      let got_file = got_file.unwrap();
      let got_contents = read_to_string(&got_file).expect("Something went wrong reading the file");
      let got_toml = got_contents.parse::<Document>();

      match got_toml {
        Ok(got) => {
          let values = got["dependencies"]
            .as_table()
            .unwrap()
            .get_values();
          for library in values.iter() {
            let name = library.0[0].get();
            let table = library.1.as_inline_table().unwrap();
            let url = table.get("url").unwrap().as_str().unwrap();
            let reference = table.get("ref").unwrap().as_str().unwrap();
            install_package(process_dir.clone(), name, &url, &reference);
          }
        }
        Err(err) => {
          eprintln!("{:?}", err);
          exit("There is an error on your gotpm file");
        }
      }
    }
    Some(Commands::Add { url, reference }) => {
      let got_file = find_package(process_dir.clone(), GOTPM_FILE);
      if got_file.is_none() {
        exit("This is not a gotpm project! Consider running the init command first.");
      }
      let got_file = got_file.unwrap();
      let got_contents = read_to_string(&got_file).expect("Something went wrong reading the file");
      let got_toml = got_contents.parse::<Document>();
      if got_toml.is_err() {
        eprintln!("{:?}", got_toml);
        exit("There is an error on your gotpm file");
      }

      if let Some(url) = url.as_deref() {
        let parsed = parse_git_url(url);

        let mut got_toml = got_toml.unwrap();
        got_toml["dependencies"][&parsed.name]["url"] = value(url);
        got_toml["dependencies"][&parsed.name]["ref"] = value(reference);

        let mut file = File::create(&got_file).unwrap();
        let _res = file.write_all(got_toml.to_string().as_bytes());

        install_package(process_dir.clone(), &parsed.name, url, reference);
      }
    }
    None => {}
  }
}
