// [[file:../dimer.note::782181ce][782181ce]]
use super::*;
// 782181ce ends here

// [[file:../dimer.note::df98a463][df98a463]]
/// Optimized results in DIMER algorithm
pub struct DimerOutput {
    /// DIMER energy, which is equal to potential energy when at dimer center.
    pub total_energy: f64,
    /// The effective force felt at dimer center (modified force for DIMER translation)
    pub effective_force: Vec<f64>,
    /// The optimized lowest curvature
    pub curvature: f64,
}

/// Main entry point for DIMER algorithm.
impl Dimer {
    /// Optimize `orientation` into the lowest curvature mode in Dimer
    /// algorithm. `pot` is a potential surface walker for evaluation of energy
    /// and force required in optimization. The dimer center is set as current
    /// position of potential `pot`. `pot` will be reset to its starting
    /// position on done. On input, `orientation` is the starting orientation of
    /// the dimer. On output, `orientation` is the optimized dimer orientation
    /// with lowest curvature.
    pub fn optimize_rotation(&mut self, orientation: &mut [f64], pot: &mut impl EvaluateDimer) -> Result<DimerOutput> {
        let center = pot.position().to_vector();
        assert_eq!(center.len(), orientation.len());
        let mut tau = orientation.to_vector().normalize();

        let rotation = self.next_rotation_step(&center, &mut tau, self.vars.max_num_rot, pot)?;
        let mut raw_dimer = rotation.raw_dimer;
        let c_min = rotation.curvature_min;
        let effective_force = self
            .next_translation_step(&tau, &mut raw_dimer, c_min)
            .as_slice()
            .to_vec();
        orientation.clone_from_slice(tau.as_slice());
        // reset potential to starting point
        pot.set_position(center.as_slice());

        Ok(DimerOutput {
            effective_force,
            curvature: c_min,
            total_energy: rotation.energy,
        })
    }
}
// df98a463 ends here
