# Creating A Load Test

[Cargo](https://doc.rust-lang.org/cargo/) is the Rust package manager. To create a new load test, use Cargo to create a new application (you can name your application anything, we've generically selected `loadtest`):

```bash
$ cargo new loadtest
     Created binary (application) `loadtest` package
$ cd loadtest/
```

This creates a new directory named `loadtest/` containing `loadtest/Cargo.toml` and `loadtest/src/main.rs`. Edit `Cargo.toml` and add Goose and [`Tokio`](https://tokio.rs/) under the dependencies heading:

```toml
[dependencies]
goose = "^0.15"
tokio = "^1.12"
```

At this point it's possible to compile all dependencies, though the resulting binary only displays "Hello, world!":

```bash
$ cargo run
    Updating crates.io index
  Downloaded goose v0.15.0
      ...
   Compiling goose v0.15.0
   Compiling loadtest v0.1.0 (/home/jandrews/devel/rust/loadtest)
    Finished dev [unoptimized + debuginfo] target(s) in 52.97s
     Running `target/debug/loadtest`
Hello, world!
```

To create an actual load test, you first have to add the following boilerplate to the top of `src/main.rs` to make Goose's functionality available to your code:

```rust,ignore
use goose::prelude::*;
```

> **Note:** Using the above prelude automatically adds the following `use` statements necessary when writing a load test, so you don't need to manually add all of them:
>
> ```rust
> use crate::config::{GooseDefault, GooseDefaultType};
> use crate::goose::{
>     GooseMethod, GooseRequest, GooseTask, GooseTaskError, GooseTaskFunction, GooseTaskResult,
>     GooseTaskSet, GooseUser,
> };
> use crate::metrics::{GooseCoordinatedOmissionMitigation, GooseMetrics};
> use crate::{task, taskset, GooseAttack, GooseError, GooseScheduler};
> ```

Then create a new load testing function. For our example we're simply going to load the front page of the website we're load-testing. Goose passes all load testing functions a mutable pointer to a GooseUser object, which is used to track metrics and make web requests. Thanks to the Reqwest library, the Goose client manages things like cookies, headers, and sessions for you. Load testing functions must be declared async, ensuring that your simulated users don't become CPU-locked.

In load test functions you typically do not set the host, and instead configure the host at run time, so you can easily run your load test against different environments without recompiling. Relative paths (not starting with a "/") should be used.

The following `loadtest_index` function simply loads the front page of our web page:

```rust,ignore
use goose::prelude::*;

async fn loadtest_index(user: &mut GooseUser) -> GooseTaskResult {
    let _goose_metrics = user.get("").await?;

    Ok(())
}
```

The function is declared `async` so that we don't block a CPU-core while loading web pages. All Goose load test functions are passed in a mutable reference to a `GooseUser` object, and return a `GooseTaskResult` which is either an empty `Ok(())` on success, or a `GooseTaskError` on failure. We use the `GooseUser` object to make requests, in this case we make a `GET` request for the front page, specified with an empty path `""`. The `.await` frees up the CPU-core while we wait for the web page to respond, and the trailing `?` unwraps the response, returning any unexpected errors that may be generated by this request.

When the GET request completes, Goose returns metrics which we store in the  `_goose_metrics` variable. The variable is prefixed with an underscore (`_`) to tell the compiler we are intentionally not using the results. Finally, after making a single successful request, we return `Ok(())` to let Goose know this task function completed successfully.

Now we have to tell Goose about our new task function. Edit the `main()` function, setting a return type and replacing the hello world text as follows:

```rust,ignore
#[tokio::main]
async fn main() -> Result<(), GooseError> {
    GooseAttack::initialize()?
        .register_taskset(taskset!("LoadtestTasks")
            .register_task(task!(loadtest_index))
        )
        .execute()
        .await?
        .print();

    Ok(())
}
```

The `#[tokio::main]` at the beginning of this example is a Tokio macro necessary because Goose is an asynchronous library, allowing (and requiring) us to declare the `main()` function of our load test application as `async`.

If you're new to Rust, `main()`'s return type of `Result<(), GooseError>` may look strange. It essentially says that `main` will return nothing (`()`) on success, and will return a `GooseError` on failure. This is helpful as several of `GooseAttack`'s methods can fail, returning an error. In our example, `initialize()` and `execute()` each may fail. The `?` that follows the method's name tells our program to exit and return an error on failure, otherwise continue on. Note that the `.execute()` method is asynchronous, so it must be followed with `.await`, and as it can return an error it alsos has a `?`. The `print()` method consumes the `GooseMetrics` object returned by `GooseAttack.execute()` and prints a summary if metrics are enabled. The final line, `Ok(())` returns the empty result expected on success.

And that's it, you've created your first load test! Read on to see how to run it and what it does.
