use crate::api::{PullRequest, ReceivedMessage};
use crate::client::Client;
use crate::message::Message;
use anyhow::Result;
use std::collections::VecDeque;

pub struct Subscription {
    pub(crate) name: String,
    pub(crate) client: Client,
    pub(crate) buffer: VecDeque<ReceivedMessage>,
}

impl Subscription {
    pub(crate) fn new(client: Client, name: impl Into<String>) -> Self {
        Self {
            client,
            name: name.into(),
            buffer: VecDeque::new(),
        }
    }
    async fn pull(&mut self, max_messages: i32) -> Result<Vec<ReceivedMessage>> {
        let mut pr = PullRequest::default();
        pr.subscription = self.name.clone();
        pr.max_messages = max_messages;
        let req = self.client.insert_authz_token(pr).await?;
        let response = self.client.subscriber.pull(req).await?.into_inner();
        Ok(response.received_messages)
    }
    pub async fn receive(&mut self) -> Option<Message> {
        loop {
            if let Some(handle) = self.buffer.pop_front() {
                let pubsub_message = handle.message.unwrap();
                let timestamp = pubsub_message.publish_time.unwrap();
                let message = Message {
                    client: self.client.clone(),
                    subscription_name: self.name.clone(),
                    data: pubsub_message.data,
                    message_id: pubsub_message.message_id,
                    ack_id: handle.ack_id,
                    attributes: pubsub_message.attributes,
                    publish_time: chrono::NaiveDateTime::from_timestamp(
                        timestamp.seconds,
                        timestamp.nanos as u32,
                    ),
                };
                break Some(message);
            } else {
                if let Ok(messages) = self.pull(50).await {
                    if messages.is_empty() {
                        break None;
                    }
                    self.buffer.extend(messages)
                }
            }
        }
    }
}
