use crate::api::{publisher_client::PublisherClient, subscriber_client::SubscriberClient};
use crate::certs::get_pem;
use crate::subscription::Subscription;
use crate::topic::Topic;
use anyhow::Result;
use google_auth::TokenManager;
use tonic::transport::{Certificate, Channel, ClientTlsConfig};

#[derive(Clone)]
pub struct Client {
    pub(crate) project_id: String,
    pub(crate) publisher: PublisherClient<Channel>,
    pub(crate) subscriber: SubscriberClient<Channel>,
    pub(crate) token_manager: TokenManager,
}

impl Client {
    pub(crate) const DOMAIN_NAME: &'static str = "pubsub.googleapis.com";
    pub(crate) const ENDPOINT: &'static str = "https://pubsub.googleapis.com";
    pub(crate) const SCOPES: [&'static str; 2] = [
        "https://www.googleapis.com/auth/cloud-platform",
        "https://www.googleapis.com/auth/pubsub",
    ];
    pub async fn new(project_id: &str) -> Result<Self> {
        let pem = get_pem();
        let tls_config = ClientTlsConfig::new()
            .ca_certificate(Certificate::from_pem(&pem))
            .domain_name(Self::DOMAIN_NAME);
        let channel = Channel::from_static(Self::ENDPOINT)
            .tls_config(tls_config)?
            .connect()
            .await?;
        let subscriber = SubscriberClient::new(channel.clone());
        let publisher = PublisherClient::new(channel);
        let token_manager = TokenManager::new(&Self::SCOPES).await?;
        Ok(Self {
            project_id: project_id.to_string(),
            subscriber,
            publisher,
            token_manager,
        })
    }
    pub fn subscription(&self, subscription_name: &str) -> Subscription {
        let name = format!(
            "projects/{}/subscriptions/{}",
            &self.project_id, subscription_name
        );
        Subscription::new(self.clone(), name)
    }
    pub fn topic(&self, topic_name: &str) -> Topic {
        let name = format!("projects/{}/topics/{}", self.project_id, topic_name);
        Topic::new(self.clone(), name)
    }
}
