//! Contains the `Country` enum and its associated traits. It specifies a
//! country or a nation. See also [ISO 3166 country
//! codes](https://en.wikipedia.org/wiki/ISO_3166-1).

use crate::geocoding::error::Error;
use serde::{Deserialize, Serialize, Deserializer};
use std::convert::TryFrom;

/// Country is the national political entity and is typically the highest order
/// type returned by the Geocoder.
///
/// The codes match a country name or a two letter [ISO
/// 3166-1](https://en.wikipedia.org/wiki/ISO_3166-1) country code. The API
/// follows the ISO standard for defining countries, and the filtering works
/// best when using the corresponding ISO code of the country.
///
/// **Note: If you receive unexpected results with a country code, verify that
/// you are using a code which includes the countries, dependent territories,
/// and special areas of geographical interest you intend. You can find code
/// information at [Wikipedia: List of ISO 3166 country
/// codes](https://en.wikipedia.org/wiki/List_of_ISO_3166_country_codes) or the
/// [ISO Online Browsing Platform](https://www.iso.org/obp/ui/#search).**

#[derive(Clone, Debug, Eq, Hash, Ord, PartialEq, PartialOrd, Serialize)]
pub enum Country {
    Afghanistan,
    AlandIslands,
    Albania,
    Algeria,
    AmericanSamoa,
    Andorra,
    Angola,
    Anguilla,
    Antarctica,
    AntiguaAndBarbuda,
    Argentina,
    Armenia,
    Aruba,
    Australia,
    Austria,
    Azerbaijan,
    Bahamas,
    Bahrain,
    Bangladesh,
    Barbados,
    Belarus,
    Belgium,
    Belize,
    Benin,
    Bermuda,
    Bhutan,
    Bolivia,
    BonaireSintEustatiusAndSaba,
    BosniaAndHerzegovina,
    Botswana,
    BouvetIsland,
    Brazil,
    BritishIndianOceanTerritory,
    BritishVirginIslands,
    Brunei,
    Bulgaria,
    BurkinaFaso,
    Burundi,
    CaboVerde,
    Cambodia,
    Cameroon,
    Canada,
    CaymanIslands,
    CentralAfricanRepublic,
    Chad,
    Chile,
    China,
    ChristmasIsland,
    CocosKeelingIslands,
    Colombia,
    Comoros,
    CongoBrazzaville,
    CongoKinshasa,
    CookIslands,
    CostaRica,
    CotedIvoire,
    Croatia,
    Cuba,
    Curacao,
    Cyprus,
    Czechia,
    Denmark,
    Djibouti,
    Dominica,
    DominicanRepublic,
    Ecuador,
    Egypt,
    ElSalvador,
    EquatorialGuinea,
    Eritrea,
    Estonia,
    Eswatini,
    Ethiopia,
    FalklandIslands,
    FaroeIslands,
    Fiji,
    Finland,
    France,
    FrenchGuiana,
    FrenchPolynesia,
    FrenchSouthernTerritories,
    Gabon,
    Gambia,
    Georgia,
    Germany,
    Ghana,
    Gibraltar,
    Greece,
    Greenland,
    Grenada,
    Guadeloupe,
    Guam,
    Guatemala,
    Guernsey,
    Guinea,
    GuineaBissau,
    Guyana,
    Haiti,
    HeardAndMcDonaldIslands,
    HolySee,
    Honduras,
    HongKong,
    Hungary,
    Iceland,
    India,
    Indonesia,
    Iran,
    Iraq,
    Ireland,
    IsleOfMan,
    Israel,
    Italy,
    Jamaica,
    Japan,
    Jersey,
    Jordan,
    Kazakhstan,
    Kenya,
    Kiribati,
    Kuwait,
    Kyrgyzstan,
    Laos,
    Latvia,
    Lebanon,
    Lesotho,
    Liberia,
    Libya,
    Liechtenstein,
    Lithuania,
    Luxembourg,
    Macao,
    Madagascar,
    Malawi,
    Malaysia,
    Maldives,
    Mali,
    Malta,
    MarshallIslands,
    Martinique,
    Mauritania,
    Mauritius,
    Mayotte,
    Mexico,
    Micronesia,
    Moldova,
    Monaco,
    Mongolia,
    Montenegro,
    Montserrat,
    Morocco,
    Mozambique,
    Myanmar,
    Namibia,
    Nauru,
    Nepal,
    Netherlands,
    NewCaledonia,
    NewZealand,
    Nicaragua,
    Niger,
    Nigeria,
    Niue,
    NorfolkIsland,
    NorthernMarianaIslands,
    NorthKorea,
    NorthMacedonia,
    Norway,
    Oman,
    Pakistan,
    Palau,
    Palestine,
    Panama,
    PapuaNewGuinea,
    Paraguay,
    Peru,
    Philippines,
    Pitcairn,
    Poland,
    Portugal,
    PuertoRico,
    Qatar,
    Reunion,
    Romania,
    Russia,
    Rwanda,
    Samoa,
    SanMarino,
    SaoTomeAndPrincipe,
    SaudiArabia,
    Senegal,
    Serbia,
    Seychelles,
    SierraLeone,
    Singapore,
    SintMaarten,
    Slovakia,
    Slovenia,
    SolomonIslands,
    Somalia,
    SouthAfrica,
    SouthGeorgiaAndSouthSandwichIslands,
    SouthKorea,
    SouthSudan,
    Spain,
    SriLanka,
    StBarthelemy,
    StHelena,
    StKittsAndNevis,
    StLucia,
    StMartin,
    StPierreAndMiquelon,
    StVincentAndGrenadines,
    Sudan,
    Suriname,
    SvalbardAndJanMayen,
    Sweden,
    Switzerland,
    Syria,
    Taiwan,
    Tajikistan,
    Tanzania,
    Thailand,
    TimorLeste,
    Togo,
    Tokelau,
    Tonga,
    TrinidadAndTobago,
    Tunisia,
    Turkey,
    Turkmenistan,
    TurksAndCaicosIslands,
    Tuvalu,
    Uganda,
    Ukraine,
    UnitedArabEmirates,
    UnitedKingdom,
    UnitedStates,
    UnitedStatesMinorOutlyingIslands,
    Uruguay,
    USVirginIslands,
    Uzbekistan,
    Vanuatu,
    Venezuela,
    Vietnam,
    WallisAndFutuna,
    WesternSahara,
    Yemen,
    Zambia,
    Zimbabwe,
} // enum

impl<'de> Deserialize<'de> for Country {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        use serde::de::Error;

        let s = String::deserialize(deserializer)?;
        let try_country = Country::try_from(s.as_str());
        match try_country {
            Ok(country) => Ok(country),
            Err(err) => Err(Error::custom(err.to_string()))
        }
    }
}

impl std::convert::From<&Country> for String {
    /// Converts a `Country` enum to a `String` that contains a [ISO 3166-1
    /// Alpha-2](https://en.wikipedia.org/wiki/List_of_ISO_3166_country_codes)
    /// country code.
    fn from(country: &Country) -> String {
        match country {
            Country::Afghanistan => String::from("AF"),
            Country::AlandIslands => String::from("AX"),
            Country::Albania => String::from("AL"),
            Country::Algeria => String::from("DZ"),
            Country::AmericanSamoa => String::from("AS"),
            Country::Andorra => String::from("AD"),
            Country::Angola => String::from("AO"),
            Country::Anguilla => String::from("AI"),
            Country::Antarctica => String::from("AQ"),
            Country::AntiguaAndBarbuda => String::from("AG"),
            Country::Argentina => String::from("AR"),
            Country::Armenia => String::from("AM"),
            Country::Aruba => String::from("AW"),
            Country::Australia => String::from("AU"),
            Country::Austria => String::from("AT"),
            Country::Azerbaijan => String::from("AZ"),
            Country::Bahamas => String::from("BS"),
            Country::Bahrain => String::from("BH"),
            Country::Bangladesh => String::from("BD"),
            Country::Barbados => String::from("BB"),
            Country::Belarus => String::from("BY"),
            Country::Belgium => String::from("BE"),
            Country::Belize => String::from("BZ"),
            Country::Benin => String::from("BJ"),
            Country::Bermuda => String::from("BM"),
            Country::Bhutan => String::from("BT"),
            Country::Bolivia => String::from("BO"),
            Country::BonaireSintEustatiusAndSaba => String::from("BQ"),
            Country::BosniaAndHerzegovina => String::from("BA"),
            Country::Botswana => String::from("BW"),
            Country::BouvetIsland => String::from("BV"),
            Country::Brazil => String::from("BR"),
            Country::BritishIndianOceanTerritory => String::from("IO"),
            Country::BritishVirginIslands => String::from("VG"),
            Country::Brunei => String::from("BN"),
            Country::Bulgaria => String::from("BG"),
            Country::BurkinaFaso => String::from("BF"),
            Country::Burundi => String::from("BI"),
            Country::CaboVerde => String::from("CV"),
            Country::Cambodia => String::from("KH"),
            Country::Cameroon => String::from("CM"),
            Country::Canada => String::from("CA"),
            Country::CaymanIslands => String::from("KY"),
            Country::CentralAfricanRepublic => String::from("CF"),
            Country::Chad => String::from("TD"),
            Country::Chile => String::from("CL"),
            Country::China => String::from("CN"),
            Country::ChristmasIsland => String::from("CX"),
            Country::CocosKeelingIslands => String::from("CC"),
            Country::Colombia => String::from("CO"),
            Country::Comoros => String::from("KM"),
            Country::CongoBrazzaville => String::from("CG"),
            Country::CongoKinshasa => String::from("CD"),
            Country::CookIslands => String::from("CK"),
            Country::CostaRica => String::from("CR"),
            Country::CotedIvoire => String::from("CI"),
            Country::Croatia => String::from("HR"),
            Country::Cuba => String::from("CU"),
            Country::Curacao => String::from("CW"),
            Country::Cyprus => String::from("CY"),
            Country::Czechia => String::from("CZ"),
            Country::Denmark => String::from("DK"),
            Country::Djibouti => String::from("DJ"),
            Country::Dominica => String::from("DM"),
            Country::DominicanRepublic => String::from("DO"),
            Country::Ecuador => String::from("EC"),
            Country::Egypt => String::from("EG"),
            Country::ElSalvador => String::from("SV"),
            Country::EquatorialGuinea => String::from("GQ"),
            Country::Eritrea => String::from("ER"),
            Country::Estonia => String::from("EE"),
            Country::Eswatini => String::from("SZ"),
            Country::Ethiopia => String::from("ET"),
            Country::FalklandIslands => String::from("FK"),
            Country::FaroeIslands => String::from("FO"),
            Country::Fiji => String::from("FJ"),
            Country::Finland => String::from("FI"),
            Country::France => String::from("FR"),
            Country::FrenchGuiana => String::from("GF"),
            Country::FrenchPolynesia => String::from("PF"),
            Country::FrenchSouthernTerritories => String::from("TF"),
            Country::Gabon => String::from("GA"),
            Country::Gambia => String::from("GM"),
            Country::Georgia => String::from("GE"),
            Country::Germany => String::from("DE"),
            Country::Ghana => String::from("GH"),
            Country::Gibraltar => String::from("GI"),
            Country::Greece => String::from("GR"),
            Country::Greenland => String::from("GL"),
            Country::Grenada => String::from("GD"),
            Country::Guadeloupe => String::from("GP"),
            Country::Guam => String::from("GU"),
            Country::Guatemala => String::from("GT"),
            Country::Guernsey => String::from("GG"),
            Country::Guinea => String::from("GN"),
            Country::GuineaBissau => String::from("GW"),
            Country::Guyana => String::from("GY"),
            Country::Haiti => String::from("HT"),
            Country::HeardAndMcDonaldIslands => String::from("HM"),
            Country::HolySee => String::from("VA"),
            Country::Honduras => String::from("HN"),
            Country::HongKong => String::from("HK"),
            Country::Hungary => String::from("HU"),
            Country::Iceland => String::from("IS"),
            Country::India => String::from("IN"),
            Country::Indonesia => String::from("ID"),
            Country::Iran => String::from("IR"),
            Country::Iraq => String::from("IQ"),
            Country::Ireland => String::from("IE"),
            Country::IsleOfMan => String::from("IM"),
            Country::Israel => String::from("IL"),
            Country::Italy => String::from("IT"),
            Country::Jamaica => String::from("JM"),
            Country::Japan => String::from("JP"),
            Country::Jersey => String::from("JE"),
            Country::Jordan => String::from("JO"),
            Country::Kazakhstan => String::from("KZ"),
            Country::Kenya => String::from("KE"),
            Country::Kiribati => String::from("KI"),
            Country::Kuwait => String::from("KW"),
            Country::Kyrgyzstan => String::from("KG"),
            Country::Laos => String::from("LA"),
            Country::Latvia => String::from("LV"),
            Country::Lebanon => String::from("LB"),
            Country::Lesotho => String::from("LS"),
            Country::Liberia => String::from("LR"),
            Country::Libya => String::from("LY"),
            Country::Liechtenstein => String::from("LI"),
            Country::Lithuania => String::from("LT"),
            Country::Luxembourg => String::from("LU"),
            Country::Macao => String::from("MO"),
            Country::Madagascar => String::from("MG"),
            Country::Malawi => String::from("MW"),
            Country::Malaysia => String::from("MY"),
            Country::Maldives => String::from("MV"),
            Country::Mali => String::from("ML"),
            Country::Malta => String::from("MT"),
            Country::MarshallIslands => String::from("MH"),
            Country::Martinique => String::from("MQ"),
            Country::Mauritania => String::from("MR"),
            Country::Mauritius => String::from("MU"),
            Country::Mayotte => String::from("YT"),
            Country::Mexico => String::from("MX"),
            Country::Micronesia => String::from("FM"),
            Country::Moldova => String::from("MD"),
            Country::Monaco => String::from("MC"),
            Country::Mongolia => String::from("MN"),
            Country::Montenegro => String::from("ME"),
            Country::Montserrat => String::from("MS"),
            Country::Morocco => String::from("MA"),
            Country::Mozambique => String::from("MZ"),
            Country::Myanmar => String::from("MM"),
            Country::Namibia => String::from("NA"),
            Country::Nauru => String::from("NR"),
            Country::Nepal => String::from("NP"),
            Country::Netherlands => String::from("NL"),
            Country::NewCaledonia => String::from("NC"),
            Country::NewZealand => String::from("NZ"),
            Country::Nicaragua => String::from("NI"),
            Country::Niger => String::from("NE"),
            Country::Nigeria => String::from("NG"),
            Country::Niue => String::from("NU"),
            Country::NorfolkIsland => String::from("NF"),
            Country::NorthernMarianaIslands => String::from("MP"),
            Country::NorthKorea => String::from("KP"),
            Country::NorthMacedonia => String::from("MK"),
            Country::Norway => String::from("NO"),
            Country::Oman => String::from("OM"),
            Country::Pakistan => String::from("PK"),
            Country::Palau => String::from("PW"),
            Country::Palestine => String::from("PS"),
            Country::Panama => String::from("PA"),
            Country::PapuaNewGuinea => String::from("PG"),
            Country::Paraguay => String::from("PY"),
            Country::Peru => String::from("PE"),
            Country::Philippines => String::from("PH"),
            Country::Pitcairn => String::from("PN"),
            Country::Poland => String::from("PL"),
            Country::Portugal => String::from("PT"),
            Country::PuertoRico => String::from("PR"),
            Country::Qatar => String::from("QA"),
            Country::Reunion => String::from("RE"),
            Country::Romania => String::from("RO"),
            Country::Russia => String::from("RU"),
            Country::Rwanda => String::from("RW"),
            Country::Samoa => String::from("WS"),
            Country::SanMarino => String::from("SM"),
            Country::SaoTomeAndPrincipe => String::from("ST"),
            Country::SaudiArabia => String::from("SA"),
            Country::Senegal => String::from("SN"),
            Country::Serbia => String::from("RS"),
            Country::Seychelles => String::from("SC"),
            Country::SierraLeone => String::from("SL"),
            Country::Singapore => String::from("SG"),
            Country::SintMaarten => String::from("SX"),
            Country::Slovakia => String::from("SK"),
            Country::Slovenia => String::from("SI"),
            Country::SolomonIslands => String::from("SB"),
            Country::Somalia => String::from("SO"),
            Country::SouthAfrica => String::from("ZA"),
            Country::SouthGeorgiaAndSouthSandwichIslands => String::from("GS"),
            Country::SouthKorea => String::from("KR"),
            Country::SouthSudan => String::from("SS"),
            Country::Spain => String::from("ES"),
            Country::SriLanka => String::from("LK"),
            Country::StBarthelemy => String::from("BL"),
            Country::StHelena => String::from("SH"),
            Country::StKittsAndNevis => String::from("KN"),
            Country::StLucia => String::from("LC"),
            Country::StMartin => String::from("MF"),
            Country::StPierreAndMiquelon => String::from("PM"),
            Country::StVincentAndGrenadines => String::from("VC"),
            Country::Sudan => String::from("SD"),
            Country::Suriname => String::from("SR"),
            Country::SvalbardAndJanMayen => String::from("SJ"),
            Country::Sweden => String::from("SE"),
            Country::Switzerland => String::from("CH"),
            Country::Syria => String::from("SY"),
            Country::Taiwan => String::from("TW"),
            Country::Tajikistan => String::from("TJ"),
            Country::Tanzania => String::from("TZ"),
            Country::Thailand => String::from("TH"),
            Country::TimorLeste => String::from("TL"),
            Country::Togo => String::from("TG"),
            Country::Tokelau => String::from("TK"),
            Country::Tonga => String::from("TO"),
            Country::TrinidadAndTobago => String::from("TT"),
            Country::Tunisia => String::from("TN"),
            Country::Turkey => String::from("TR"),
            Country::Turkmenistan => String::from("TM"),
            Country::TurksAndCaicosIslands => String::from("TC"),
            Country::Tuvalu => String::from("TV"),
            Country::Uganda => String::from("UG"),
            Country::Ukraine => String::from("UA"),
            Country::UnitedArabEmirates => String::from("AE"),
            Country::UnitedKingdom => String::from("GB"),
            Country::UnitedStates => String::from("US"),
            Country::UnitedStatesMinorOutlyingIslands => String::from("UM"),
            Country::Uruguay => String::from("UY"),
            Country::USVirginIslands => String::from("VI"),
            Country::Uzbekistan => String::from("UZ"),
            Country::Vanuatu => String::from("VU"),
            Country::Venezuela => String::from("VE"),
            Country::Vietnam => String::from("VN"),
            Country::WallisAndFutuna => String::from("WF"),
            Country::WesternSahara => String::from("EH"),
            Country::Yemen => String::from("YE"),
            Country::Zambia => String::from("ZM"),
            Country::Zimbabwe => String::from("ZW"),
        } // match
    } // fn
} // enum

impl std::convert::TryFrom<&str> for Country {
    // Error definitions are contained in the
    // `google_maps\src\geocoding\error.rs` module.
    type Error = crate::geocoding::error::Error;

    /// Gets a `Country` enum from a `String` that contains a valid [ISO 3166-1
    /// Alpha-2](https://en.wikipedia.org/wiki/List_of_ISO_3166_country_codes)
    /// country code.
    fn try_from(country: &str) -> Result<Country, Error> {
        match country {
            "AF" => Ok(Country::Afghanistan),
            "AX" => Ok(Country::AlandIslands),
            "AL" => Ok(Country::Albania),
            "DZ" => Ok(Country::Algeria),
            "AS" => Ok(Country::AmericanSamoa),
            "AD" => Ok(Country::Andorra),
            "AO" => Ok(Country::Angola),
            "AI" => Ok(Country::Anguilla),
            "AQ" => Ok(Country::Antarctica),
            "AG" => Ok(Country::AntiguaAndBarbuda),
            "AR" => Ok(Country::Argentina),
            "AM" => Ok(Country::Armenia),
            "AW" => Ok(Country::Aruba),
            "AU" => Ok(Country::Australia),
            "AT" => Ok(Country::Austria),
            "AZ" => Ok(Country::Azerbaijan),
            "BS" => Ok(Country::Bahamas),
            "BH" => Ok(Country::Bahrain),
            "BD" => Ok(Country::Bangladesh),
            "BB" => Ok(Country::Barbados),
            "BY" => Ok(Country::Belarus),
            "BE" => Ok(Country::Belgium),
            "BZ" => Ok(Country::Belize),
            "BJ" => Ok(Country::Benin),
            "BM" => Ok(Country::Bermuda),
            "BT" => Ok(Country::Bhutan),
            "BO" => Ok(Country::Bolivia),
            "BQ" => Ok(Country::BonaireSintEustatiusAndSaba),
            "BA" => Ok(Country::BosniaAndHerzegovina),
            "BW" => Ok(Country::Botswana),
            "BV" => Ok(Country::BouvetIsland),
            "BR" => Ok(Country::Brazil),
            "IO" => Ok(Country::BritishIndianOceanTerritory),
            "VG" => Ok(Country::BritishVirginIslands),
            "BN" => Ok(Country::Brunei),
            "BG" => Ok(Country::Bulgaria),
            "BF" => Ok(Country::BurkinaFaso),
            "BI" => Ok(Country::Burundi),
            "CV" => Ok(Country::CaboVerde),
            "KH" => Ok(Country::Cambodia),
            "CM" => Ok(Country::Cameroon),
            "CA" => Ok(Country::Canada),
            "KY" => Ok(Country::CaymanIslands),
            "CF" => Ok(Country::CentralAfricanRepublic),
            "TD" => Ok(Country::Chad),
            "CL" => Ok(Country::Chile),
            "CN" => Ok(Country::China),
            "CX" => Ok(Country::ChristmasIsland),
            "CC" => Ok(Country::CocosKeelingIslands),
            "CO" => Ok(Country::Colombia),
            "KM" => Ok(Country::Comoros),
            "CG" => Ok(Country::CongoBrazzaville),
            "CD" => Ok(Country::CongoKinshasa),
            "CK" => Ok(Country::CookIslands),
            "CR" => Ok(Country::CostaRica),
            "CI" => Ok(Country::CotedIvoire),
            "HR" => Ok(Country::Croatia),
            "CU" => Ok(Country::Cuba),
            "CW" => Ok(Country::Curacao),
            "CY" => Ok(Country::Cyprus),
            "CZ" => Ok(Country::Czechia),
            "DK" => Ok(Country::Denmark),
            "DJ" => Ok(Country::Djibouti),
            "DM" => Ok(Country::Dominica),
            "DO" => Ok(Country::DominicanRepublic),
            "EC" => Ok(Country::Ecuador),
            "EG" => Ok(Country::Egypt),
            "SV" => Ok(Country::ElSalvador),
            "GQ" => Ok(Country::EquatorialGuinea),
            "ER" => Ok(Country::Eritrea),
            "EE" => Ok(Country::Estonia),
            "SZ" => Ok(Country::Eswatini),
            "ET" => Ok(Country::Ethiopia),
            "FK" => Ok(Country::FalklandIslands),
            "FO" => Ok(Country::FaroeIslands),
            "FJ" => Ok(Country::Fiji),
            "FI" => Ok(Country::Finland),
            "FR" => Ok(Country::France),
            "GF" => Ok(Country::FrenchGuiana),
            "PF" => Ok(Country::FrenchPolynesia),
            "TF" => Ok(Country::FrenchSouthernTerritories),
            "GA" => Ok(Country::Gabon),
            "GM" => Ok(Country::Gambia),
            "GE" => Ok(Country::Georgia),
            "DE" => Ok(Country::Germany),
            "GH" => Ok(Country::Ghana),
            "GI" => Ok(Country::Gibraltar),
            "GR" => Ok(Country::Greece),
            "GL" => Ok(Country::Greenland),
            "GD" => Ok(Country::Grenada),
            "GP" => Ok(Country::Guadeloupe),
            "GU" => Ok(Country::Guam),
            "GT" => Ok(Country::Guatemala),
            "GG" => Ok(Country::Guernsey),
            "GN" => Ok(Country::Guinea),
            "GW" => Ok(Country::GuineaBissau),
            "GY" => Ok(Country::Guyana),
            "HT" => Ok(Country::Haiti),
            "HM" => Ok(Country::HeardAndMcDonaldIslands),
            "VA" => Ok(Country::HolySee),
            "HN" => Ok(Country::Honduras),
            "HK" => Ok(Country::HongKong),
            "HU" => Ok(Country::Hungary),
            "IS" => Ok(Country::Iceland),
            "IN" => Ok(Country::India),
            "ID" => Ok(Country::Indonesia),
            "IR" => Ok(Country::Iran),
            "IQ" => Ok(Country::Iraq),
            "IE" => Ok(Country::Ireland),
            "IM" => Ok(Country::IsleOfMan),
            "IL" => Ok(Country::Israel),
            "IT" => Ok(Country::Italy),
            "JM" => Ok(Country::Jamaica),
            "JP" => Ok(Country::Japan),
            "JE" => Ok(Country::Jersey),
            "JO" => Ok(Country::Jordan),
            "KZ" => Ok(Country::Kazakhstan),
            "KE" => Ok(Country::Kenya),
            "KI" => Ok(Country::Kiribati),
            "KW" => Ok(Country::Kuwait),
            "KG" => Ok(Country::Kyrgyzstan),
            "LA" => Ok(Country::Laos),
            "LV" => Ok(Country::Latvia),
            "LB" => Ok(Country::Lebanon),
            "LS" => Ok(Country::Lesotho),
            "LR" => Ok(Country::Liberia),
            "LY" => Ok(Country::Libya),
            "LI" => Ok(Country::Liechtenstein),
            "LT" => Ok(Country::Lithuania),
            "LU" => Ok(Country::Luxembourg),
            "MO" => Ok(Country::Macao),
            "MG" => Ok(Country::Madagascar),
            "MW" => Ok(Country::Malawi),
            "MY" => Ok(Country::Malaysia),
            "MV" => Ok(Country::Maldives),
            "ML" => Ok(Country::Mali),
            "MT" => Ok(Country::Malta),
            "MH" => Ok(Country::MarshallIslands),
            "MQ" => Ok(Country::Martinique),
            "MR" => Ok(Country::Mauritania),
            "MU" => Ok(Country::Mauritius),
            "YT" => Ok(Country::Mayotte),
            "MX" => Ok(Country::Mexico),
            "FM" => Ok(Country::Micronesia),
            "MD" => Ok(Country::Moldova),
            "MC" => Ok(Country::Monaco),
            "MN" => Ok(Country::Mongolia),
            "ME" => Ok(Country::Montenegro),
            "MS" => Ok(Country::Montserrat),
            "MA" => Ok(Country::Morocco),
            "MZ" => Ok(Country::Mozambique),
            "MM" => Ok(Country::Myanmar),
            "NA" => Ok(Country::Namibia),
            "NR" => Ok(Country::Nauru),
            "NP" => Ok(Country::Nepal),
            "NL" => Ok(Country::Netherlands),
            "NC" => Ok(Country::NewCaledonia),
            "NZ" => Ok(Country::NewZealand),
            "NI" => Ok(Country::Nicaragua),
            "NE" => Ok(Country::Niger),
            "NG" => Ok(Country::Nigeria),
            "NU" => Ok(Country::Niue),
            "NF" => Ok(Country::NorfolkIsland),
            "MP" => Ok(Country::NorthernMarianaIslands),
            "KP" => Ok(Country::NorthKorea),
            "MK" => Ok(Country::NorthMacedonia),
            "NO" => Ok(Country::Norway),
            "OM" => Ok(Country::Oman),
            "PK" => Ok(Country::Pakistan),
            "PW" => Ok(Country::Palau),
            "PS" => Ok(Country::Palestine),
            "PA" => Ok(Country::Panama),
            "PG" => Ok(Country::PapuaNewGuinea),
            "PY" => Ok(Country::Paraguay),
            "PE" => Ok(Country::Peru),
            "PH" => Ok(Country::Philippines),
            "PN" => Ok(Country::Pitcairn),
            "PL" => Ok(Country::Poland),
            "PT" => Ok(Country::Portugal),
            "PR" => Ok(Country::PuertoRico),
            "QA" => Ok(Country::Qatar),
            "RE" => Ok(Country::Reunion),
            "RO" => Ok(Country::Romania),
            "RU" => Ok(Country::Russia),
            "RW" => Ok(Country::Rwanda),
            "WS" => Ok(Country::Samoa),
            "SM" => Ok(Country::SanMarino),
            "ST" => Ok(Country::SaoTomeAndPrincipe),
            "SA" => Ok(Country::SaudiArabia),
            "SN" => Ok(Country::Senegal),
            "RS" => Ok(Country::Serbia),
            "SC" => Ok(Country::Seychelles),
            "SL" => Ok(Country::SierraLeone),
            "SG" => Ok(Country::Singapore),
            "SX" => Ok(Country::SintMaarten),
            "SK" => Ok(Country::Slovakia),
            "SI" => Ok(Country::Slovenia),
            "SB" => Ok(Country::SolomonIslands),
            "SO" => Ok(Country::Somalia),
            "ZA" => Ok(Country::SouthAfrica),
            "GS" => Ok(Country::SouthGeorgiaAndSouthSandwichIslands),
            "KR" => Ok(Country::SouthKorea),
            "SS" => Ok(Country::SouthSudan),
            "ES" => Ok(Country::Spain),
            "LK" => Ok(Country::SriLanka),
            "BL" => Ok(Country::StBarthelemy),
            "SH" => Ok(Country::StHelena),
            "KN" => Ok(Country::StKittsAndNevis),
            "LC" => Ok(Country::StLucia),
            "MF" => Ok(Country::StMartin),
            "PM" => Ok(Country::StPierreAndMiquelon),
            "VC" => Ok(Country::StVincentAndGrenadines),
            "SD" => Ok(Country::Sudan),
            "SR" => Ok(Country::Suriname),
            "SJ" => Ok(Country::SvalbardAndJanMayen),
            "SE" => Ok(Country::Sweden),
            "CH" => Ok(Country::Switzerland),
            "SY" => Ok(Country::Syria),
            "TW" => Ok(Country::Taiwan),
            "TJ" => Ok(Country::Tajikistan),
            "TZ" => Ok(Country::Tanzania),
            "TH" => Ok(Country::Thailand),
            "TL" => Ok(Country::TimorLeste),
            "TG" => Ok(Country::Togo),
            "TK" => Ok(Country::Tokelau),
            "TO" => Ok(Country::Tonga),
            "TT" => Ok(Country::TrinidadAndTobago),
            "TN" => Ok(Country::Tunisia),
            "TR" => Ok(Country::Turkey),
            "TM" => Ok(Country::Turkmenistan),
            "TC" => Ok(Country::TurksAndCaicosIslands),
            "TV" => Ok(Country::Tuvalu),
            "UG" => Ok(Country::Uganda),
            "UA" => Ok(Country::Ukraine),
            "AE" => Ok(Country::UnitedArabEmirates),
            "GB" => Ok(Country::UnitedKingdom),
            "US" => Ok(Country::UnitedStates),
            "UM" => Ok(Country::UnitedStatesMinorOutlyingIslands),
            "UY" => Ok(Country::Uruguay),
            "VI" => Ok(Country::USVirginIslands),
            "UZ" => Ok(Country::Uzbekistan),
            "VU" => Ok(Country::Vanuatu),
            "VE" => Ok(Country::Venezuela),
            "VN" => Ok(Country::Vietnam),
            "WF" => Ok(Country::WallisAndFutuna),
            "EH" => Ok(Country::WesternSahara),
            "YE" => Ok(Country::Yemen),
            "ZM" => Ok(Country::Zambia),
            "ZW" => Ok(Country::Zimbabwe),
            _ => Err(Error::InvalidCountryCode(country.to_string())),
        } // match
    } // fn
} // impl

impl std::fmt::Display for Country {
    /// Formats a `Country` enum into a string that is presentable to the end
    /// user.
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Country::Afghanistan => write!(f, "Afghanistan"),
            Country::AlandIslands => write!(f, "Åland Islands"),
            Country::Albania => write!(f, "Albania"),
            Country::Algeria => write!(f, "Algeria"),
            Country::AmericanSamoa => write!(f, "American Samoa"),
            Country::Andorra => write!(f, "Andorra"),
            Country::Angola => write!(f, "Angola"),
            Country::Anguilla => write!(f, "Anguilla"),
            Country::Antarctica => write!(f, "Antarctica"),
            Country::AntiguaAndBarbuda => write!(f, "Antigua and Barbuda"),
            Country::Argentina => write!(f, "Argentina"),
            Country::Armenia => write!(f, "Armenia"),
            Country::Aruba => write!(f, "Aruba"),
            Country::Australia => write!(f, "Australia"),
            Country::Austria => write!(f, "Austria"),
            Country::Azerbaijan => write!(f, "Azerbaijan"),
            Country::Bahamas => write!(f, "Bahamas"),
            Country::Bahrain => write!(f, "Bahrain"),
            Country::Bangladesh => write!(f, "Bangladesh"),
            Country::Barbados => write!(f, "Barbados"),
            Country::Belarus => write!(f, "Belarus"),
            Country::Belgium => write!(f, "Belgium"),
            Country::Belize => write!(f, "Belize"),
            Country::Benin => write!(f, "Benin"),
            Country::Bermuda => write!(f, "Bermuda"),
            Country::Bhutan => write!(f, "Bhutan"),
            Country::Bolivia => write!(f, "Bolivia"),
            Country::BonaireSintEustatiusAndSaba => write!(f, "Bonaire, Sint Eustatius and Saba"),
            Country::BosniaAndHerzegovina => write!(f, "Bosnia and Herzegovina"),
            Country::Botswana => write!(f, "Botswana"),
            Country::BouvetIsland => write!(f, "Bouvet Island"),
            Country::Brazil => write!(f, "Brazil"),
            Country::BritishIndianOceanTerritory => write!(f, "British Indian Ocean Territory"),
            Country::BritishVirginIslands => write!(f, "British Virgin Islands"),
            Country::Brunei => write!(f, "Brunei"),
            Country::Bulgaria => write!(f, "Bulgaria"),
            Country::BurkinaFaso => write!(f, "Burkina Faso"),
            Country::Burundi => write!(f, "Burundi"),
            Country::CaboVerde => write!(f, "Cabo Verde"),
            Country::Cambodia => write!(f, "Cambodia"),
            Country::Cameroon => write!(f, "Cameroon"),
            Country::Canada => write!(f, "Canada"),
            Country::CaymanIslands => write!(f, "Cayman Islands"),
            Country::CentralAfricanRepublic => write!(f, "Central African Republic"),
            Country::Chad => write!(f, "Chad"),
            Country::Chile => write!(f, "Chile"),
            Country::China => write!(f, "China"),
            Country::ChristmasIsland => write!(f, "Christmas Island"),
            Country::CocosKeelingIslands => write!(f, "Cocos (Keeling) Islands"),
            Country::Colombia => write!(f, "Colombia"),
            Country::Comoros => write!(f, "Comoros"),
            Country::CongoBrazzaville => write!(f, "Congo"),
            Country::CongoKinshasa => write!(f, "Democratic Republic of the Congo"),
            Country::CookIslands => write!(f, "Cook Islands"),
            Country::CostaRica => write!(f, "Costa Rica"),
            Country::CotedIvoire => write!(f, "Côte d'Ivoire"),
            Country::Croatia => write!(f, "Croatia"),
            Country::Cuba => write!(f, "Cuba"),
            Country::Curacao => write!(f, "Curaçao"),
            Country::Cyprus => write!(f, "Cyprus"),
            Country::Czechia => write!(f, "Czechia"),
            Country::Denmark => write!(f, "Denmark"),
            Country::Djibouti => write!(f, "Djibouti"),
            Country::Dominica => write!(f, "Dominica"),
            Country::DominicanRepublic => write!(f, "Dominican Republic"),
            Country::Ecuador => write!(f, "Ecuador"),
            Country::Egypt => write!(f, "Egypt"),
            Country::ElSalvador => write!(f, "El Salvador"),
            Country::EquatorialGuinea => write!(f, "Equatorial Guinea"),
            Country::Eritrea => write!(f, "Eritrea"),
            Country::Estonia => write!(f, "Estonia"),
            Country::Eswatini => write!(f, "Eswatini"),
            Country::Ethiopia => write!(f, "Ethiopia"),
            Country::FalklandIslands => write!(f, "Falkland Islands"),
            Country::FaroeIslands => write!(f, "Faroe Islands"),
            Country::Fiji => write!(f, "Fiji"),
            Country::Finland => write!(f, "Finland"),
            Country::France => write!(f, "France"),
            Country::FrenchGuiana => write!(f, "French Guiana"),
            Country::FrenchPolynesia => write!(f, "French Polynesia"),
            Country::FrenchSouthernTerritories => write!(f, "French Southern Territories"),
            Country::Gabon => write!(f, "Gabon"),
            Country::Gambia => write!(f, "Gambia"),
            Country::Georgia => write!(f, "Georgia"),
            Country::Germany => write!(f, "Germany"),
            Country::Ghana => write!(f, "Ghana"),
            Country::Gibraltar => write!(f, "Gibraltar"),
            Country::Greece => write!(f, "Greece"),
            Country::Greenland => write!(f, "Greenland"),
            Country::Grenada => write!(f, "Grenada"),
            Country::Guadeloupe => write!(f, "Guadeloupe"),
            Country::Guam => write!(f, "Guam"),
            Country::Guatemala => write!(f, "Guatemala"),
            Country::Guernsey => write!(f, "Guernsey"),
            Country::Guinea => write!(f, "Guinea"),
            Country::GuineaBissau => write!(f, "Guinea-Bissau"),
            Country::Guyana => write!(f, "Guyana"),
            Country::Haiti => write!(f, "Haiti"),
            Country::HeardAndMcDonaldIslands => write!(f, "Heard Island and McDonald Islands"),
            Country::HolySee => write!(f, "Holy See"),
            Country::Honduras => write!(f, "Honduras"),
            Country::HongKong => write!(f, "Hong Kong"),
            Country::Hungary => write!(f, "Hungary"),
            Country::Iceland => write!(f, "Iceland"),
            Country::India => write!(f, "India"),
            Country::Indonesia => write!(f, "Indonesia"),
            Country::Iran => write!(f, "Iran"),
            Country::Iraq => write!(f, "Iraq"),
            Country::Ireland => write!(f, "Ireland"),
            Country::IsleOfMan => write!(f, "Isle of Man"),
            Country::Israel => write!(f, "Israel"),
            Country::Italy => write!(f, "Italy"),
            Country::Jamaica => write!(f, "Jamaica"),
            Country::Japan => write!(f, "Japan"),
            Country::Jersey => write!(f, "Jersey"),
            Country::Jordan => write!(f, "Jordan"),
            Country::Kazakhstan => write!(f, "Kazakhstan"),
            Country::Kenya => write!(f, "Kenya"),
            Country::Kiribati => write!(f, "Kiribati"),
            Country::Kuwait => write!(f, "Kuwait"),
            Country::Kyrgyzstan => write!(f, "Kyrgyzstan"),
            Country::Laos => write!(f, "Laos"),
            Country::Latvia => write!(f, "Latvia"),
            Country::Lebanon => write!(f, "Lebanon"),
            Country::Lesotho => write!(f, "Lesotho"),
            Country::Liberia => write!(f, "Liberia"),
            Country::Libya => write!(f, "Libya"),
            Country::Liechtenstein => write!(f, "Liechtenstein"),
            Country::Lithuania => write!(f, "Lithuania"),
            Country::Luxembourg => write!(f, "Luxembourg"),
            Country::Macao => write!(f, "Macao"),
            Country::Madagascar => write!(f, "Madagascar"),
            Country::Malawi => write!(f, "Malawi"),
            Country::Malaysia => write!(f, "Malaysia"),
            Country::Maldives => write!(f, "Maldives"),
            Country::Mali => write!(f, "Mali"),
            Country::Malta => write!(f, "Malta"),
            Country::MarshallIslands => write!(f, "Marshall Islands"),
            Country::Martinique => write!(f, "Martinique"),
            Country::Mauritania => write!(f, "Mauritania"),
            Country::Mauritius => write!(f, "Mauritius"),
            Country::Mayotte => write!(f, "Mayotte"),
            Country::Mexico => write!(f, "Mexico"),
            Country::Micronesia => write!(f, "Micronesia"),
            Country::Moldova => write!(f, "Moldova"),
            Country::Monaco => write!(f, "Monaco"),
            Country::Mongolia => write!(f, "Mongolia"),
            Country::Montenegro => write!(f, "Montenegro"),
            Country::Montserrat => write!(f, "Montserrat"),
            Country::Morocco => write!(f, "Morocco"),
            Country::Mozambique => write!(f, "Mozambique"),
            Country::Myanmar => write!(f, "Myanmar"),
            Country::Namibia => write!(f, "Namibia"),
            Country::Nauru => write!(f, "Nauru"),
            Country::Nepal => write!(f, "Nepal"),
            Country::Netherlands => write!(f, "Netherlands"),
            Country::NewCaledonia => write!(f, "New Caledonia"),
            Country::NewZealand => write!(f, "New Zealand"),
            Country::Nicaragua => write!(f, "Nicaragua"),
            Country::Niger => write!(f, "Niger"),
            Country::Nigeria => write!(f, "Nigeria"),
            Country::Niue => write!(f, "Niue"),
            Country::NorfolkIsland => write!(f, "Norfolk Island"),
            Country::NorthernMarianaIslands => write!(f, "Northern Mariana Islands"),
            Country::NorthKorea => write!(f, "North Korea"),
            Country::NorthMacedonia => write!(f, "North Macedonia"),
            Country::Norway => write!(f, "Norway"),
            Country::Oman => write!(f, "Oman"),
            Country::Pakistan => write!(f, "Pakistan"),
            Country::Palau => write!(f, "Palau"),
            Country::Palestine => write!(f, "Palestine"),
            Country::Panama => write!(f, "Panama"),
            Country::PapuaNewGuinea => write!(f, "Papua New Guinea"),
            Country::Paraguay => write!(f, "Paraguay"),
            Country::Peru => write!(f, "Peru"),
            Country::Philippines => write!(f, "Philippines"),
            Country::Pitcairn => write!(f, "Pitcairn"),
            Country::Poland => write!(f, "Poland"),
            Country::Portugal => write!(f, "Portugal"),
            Country::PuertoRico => write!(f, "Puerto Rico"),
            Country::Qatar => write!(f, "Qatar"),
            Country::Reunion => write!(f, "Réunion"),
            Country::Romania => write!(f, "Romania"),
            Country::Russia => write!(f, "Russian Federation"),
            Country::Rwanda => write!(f, "Rwanda"),
            Country::Samoa => write!(f, "Samoa"),
            Country::SanMarino => write!(f, "San Marino"),
            Country::SaoTomeAndPrincipe => write!(f, "Sao Tome and Principe"),
            Country::SaudiArabia => write!(f, "Saudi Arabia"),
            Country::Senegal => write!(f, "Senegal"),
            Country::Serbia => write!(f, "Serbia"),
            Country::Seychelles => write!(f, "Seychelles"),
            Country::SierraLeone => write!(f, "Sierra Leone"),
            Country::Singapore => write!(f, "Singapore"),
            Country::SintMaarten => write!(f, "Sint Maarten"),
            Country::Slovakia => write!(f, "Slovakia"),
            Country::Slovenia => write!(f, "Slovenia"),
            Country::SolomonIslands => write!(f, "Solomon Islands"),
            Country::Somalia => write!(f, "Somalia"),
            Country::SouthAfrica => write!(f, "SouthA frica"),
            Country::SouthGeorgiaAndSouthSandwichIslands => write!(f, "South Georgia and the South Sandwich Islands"),
            Country::SouthKorea => write!(f, "South Korea"),
            Country::SouthSudan => write!(f, "South Sudan"),
            Country::Spain => write!(f, "Spain"),
            Country::SriLanka => write!(f, "Sri Lanka"),
            Country::StBarthelemy => write!(f, "Saint Barthélemy"),
            Country::StHelena => write!(f, "Saint Helena, Ascension and Tristan da Cunha"),
            Country::StKittsAndNevis => write!(f, "Saint Kitts and Nevis"),
            Country::StLucia => write!(f, "Saint Lucia"),
            Country::StMartin => write!(f, "Saint Martin"),
            Country::StPierreAndMiquelon => write!(f, "Saint Pierre and Miquelon"),
            Country::StVincentAndGrenadines => write!(f, "Saint Vincent and the Grenadines"),
            Country::Sudan => write!(f, "Sudan"),
            Country::Suriname => write!(f, "Suriname"),
            Country::SvalbardAndJanMayen => write!(f, "Svalbard and Jan Mayen"),
            Country::Sweden => write!(f, "Sweden"),
            Country::Switzerland => write!(f, "Switzerland"),
            Country::Syria => write!(f, "Syria"),
            Country::Taiwan => write!(f, "Taiwan, Province of China"),
            Country::Tajikistan => write!(f, "Tajikistan"),
            Country::Tanzania => write!(f, "Tanzania"),
            Country::Thailand => write!(f, "Thailand"),
            Country::TimorLeste => write!(f, "Timor-Leste"),
            Country::Togo => write!(f, "Togo"),
            Country::Tokelau => write!(f, "Tokelau"),
            Country::Tonga => write!(f, "Tonga"),
            Country::TrinidadAndTobago => write!(f, "Trinidad and Tobago"),
            Country::Tunisia => write!(f, "Tunisia"),
            Country::Turkey => write!(f, "Turkey"),
            Country::Turkmenistan => write!(f, "Turkmenistan"),
            Country::TurksAndCaicosIslands => write!(f, "Turks and Caicos Islands"),
            Country::Tuvalu => write!(f, "Tuvalu"),
            Country::Uganda => write!(f, "Uganda"),
            Country::Ukraine => write!(f, "Ukraine"),
            Country::UnitedArabEmirates => write!(f, "United Arab Emirates"),
            Country::UnitedKingdom => write!(f, "United Kingdom"),
            Country::UnitedStates => write!(f, "United States"),
            Country::UnitedStatesMinorOutlyingIslands => write!(f, "United States Minor Outlying Islands"),
            Country::Uruguay => write!(f, "Uruguay"),
            Country::USVirginIslands => write!(f, "US Virgin ISlands"),
            Country::Uzbekistan => write!(f, "Uzbekistan"),
            Country::Vanuatu => write!(f, "Vanuatu"),
            Country::Venezuela => write!(f, "Venezuela"),
            Country::Vietnam => write!(f, "Viet Nam"),
            Country::WallisAndFutuna => write!(f, "Wallis and Futuna"),
            Country::WesternSahara => write!(f, "Western Sahara"),
            Country::Yemen => write!(f, "Yemen"),
            Country::Zambia => write!(f, "Zambia"),
            Country::Zimbabwe => write!(f, "Zimbabwe"),
        } // match
    } // fn
} // impl