//! Contains the `Region` enum and its associated traits. It is used to specify
//! a country in which Google Maps has been officially launched. Google's
//! _regions_ uses ccTLDs rather than ISO 3166 country codes. _This is not a
//! comprehensive list of countries, it is a list of countries that Google Maps
//! supports._

use crate::error::Error;
use serde::{Deserialize, Serialize};

/// Specifies the region bias.
///
/// [Region Biasing](https://developers.google.com/maps/documentation/directions/intro#RegionBiasing)
/// =================================================================================================
///
/// You can set the Directions service to return results from a specific region
/// by using the `region` parameter. This parameter takes a
/// [ccTLD](https://en.wikipedia.org/wiki/CcTLD) (country code top-level domain)
/// argument specifying the region bias. Most ccTLD codes are identical to ISO
/// 3166-1 codes, with some notable exceptions. For example, the United
/// Kingdom's ccTLD is "uk" (`.co.uk`) while its ISO 3166-1 code is "gb"
/// (technically for the entity of "The United Kingdom of Great Britain and
/// Northern Ireland").
///
/// You may utilize any domain in which the main Google Maps application has
/// launched driving directions.
///
/// For example, a directions request for "Toledo" to "Madrid" returns
/// appropriate results when `region` is set to `es` and "Toledo" is then
/// interpreted as the Spanish city:
///
/// ```
/// https://maps.googleapis.com/maps/api/directions/json?
/// origin=Toledo&destination=Madrid
/// &region=es
/// &key=YOUR_API_KEY
///
/// {
///     "status": "OK",
///     "routes": [ {
///         "summary": "AP-41",
///         "legs": [ {
///             ...
///         } ],
///         "copyrights": "Map data ©2010 Europa Technologies, Tele Atlas",
///         "warnings": [ ],
///         "waypoint_order": [ ]
///     } ]
/// }
/// ```
///
/// A directions request for "Toledo" to "Madrid" sent without a `region`
/// parameter does not return results, because "Toledo" is interpreted as the
/// city in Ohio and not Spain:
///
/// ```
/// https://maps.googleapis.com/maps/api/directions/json?
/// origin=Toledo&destination=Madrid
/// &key=YOUR_API_KEY
///
/// {
///     "status": "ZERO_RESULTS",
///     "routes": [ ]
/// }
/// ```
///
/// [Map Coverage Details](https://developers.google.com/maps/coverage)
/// ===================================================================
///
/// The Google Maps team is constantly working to improve our international
/// coverage. Some data, such as public transit routes, are available at a city
/// level and do not appear in this list. Please note that data coverage can
/// change if licensing agreements with the data providers change.

#[derive(Clone, Debug, Eq, Hash, Ord, PartialEq, PartialOrd, Serialize, Deserialize)]
pub enum Region {
    AlandIslands,
    Afghanistan,
    Albania,
    Algeria,
    AmericanSamoa,
    Andorra,
    Angola,
    Anguilla,
    Antarctica,
    AntiguaAndBarbuda,
    Argentina,
    Armenia,
    Aruba,
    AscensionIsland,
    Australia,
    Austria,
    Azerbaijan,
    Bahamas,
    Bahrain,
    Bangladesh,
    Barbados,
    Belarus,
    Belgium,
    Belize,
    Benin,
    Bermuda,
    Bhutan,
    Bolivia,
    BosniaAndHerzegovina,
    Botswana,
    BouvetIsland,
    Brazil,
    BritishIndianOceanTerritory,
    BritishVirginIslands,
    Brunei,
    Bulgaria,
    BurkinaFaso,
    Burundi,
    Cambodia,
    Cameroon,
    Canada,
    CanaryIslands,
    CapeVerde,
    CaribbeanNetherlands,
    CaymanIslands,
    CentralAfricanRepublic,
    CeutaAndMelilla,
    Chad,
    Chile,
    China,
    ChristmasIsland,
    ClippertonIsland,
    CocosKeelingIslands,
    Colombia,
    Comoros,
    CongoBrazzaville,
    CongoKinshasa,
    CookIslands,
    CostaRica,
    CotedIvoire,
    Croatia,
    Cuba,
    Curacao,
    Cyprus,
    Czechia,
    Denmark,
    DiegoGarcia,
    Djibouti,
    Dominica,
    DominicanRepublic,
    Ecuador,
    Egypt,
    ElSalvador,
    EquatorialGuinea,
    Eritrea,
    Estonia,
    Eswatini,
    Ethiopia,
    FalklandIslands,
    FaroeIslands,
    Fiji,
    Finland,
    France,
    FrenchGuiana,
    FrenchPolynesia,
    FrenchSouthernTerritories,
    Gabon,
    Gambia,
    Georgia,
    Germany,
    Ghana,
    Gibraltar,
    Greece,
    Greenland,
    Grenada,
    Guadeloupe,
    Guam,
    Guatemala,
    Guernsey,
    Guinea,
    GuineaBissau,
    Guyana,
    Haiti,
    HeardAndMcDonaldIslands,
    Honduras,
    HongKong,
    Hungary,
    Iceland,
    India,
    Indonesia,
    Iran,
    Iraq,
    Ireland,
    IsleOfMan,
    Israel,
    Italy,
    Jamaica,
    Japan,
    Jersey,
    Jordan,
    Kazakhstan,
    Kenya,
    Kiribati,
    Kosovo,
    Kuwait,
    Kyrgyzstan,
    Laos,
    Latvia,
    Lebanon,
    Lesotho,
    Liberia,
    Libya,
    Liechtenstein,
    Lithuania,
    Luxembourg,
    Macao,
    Madagascar,
    Malawi,
    Malaysia,
    Maldives,
    Mali,
    Malta,
    MarshallIslands,
    Martinique,
    Mauritania,
    Mauritius,
    Mayotte,
    Mexico,
    Micronesia,
    Moldova,
    Monaco,
    Mongolia,
    Montenegro,
    Montserrat,
    Morocco,
    Mozambique,
    Myanmar,
    Namibia,
    Nauru,
    Nepal,
    Netherlands,
    NewCaledonia,
    NewZealand,
    Nicaragua,
    Niger,
    Nigeria,
    Niue,
    NorfolkIsland,
    NorthKorea,
    NorthMacedonia,
    NorthernMarianaIslands,
    Norway,
    Oman,
    Pakistan,
    Palau,
    Palestine,
    Panama,
    PapuaNewGuinea,
    Paraguay,
    Peru,
    Philippines,
    PitcairnIslands,
    Poland,
    Portugal,
    PuertoRico,
    Qatar,
    Romania,
    Russia,
    Rwanda,
    Reunion,
    Samoa,
    SanMarino,
    SaudiArabia,
    Senegal,
    Serbia,
    Seychelles,
    SierraLeone,
    Singapore,
    SintMaarten,
    Slovakia,
    Slovenia,
    SolomonIslands,
    Somalia,
    SouthAfrica,
    SouthGeorgiaAndTheSouthSandwichIslands,
    SouthKorea,
    SouthSudan,
    Spain,
    SriLanka,
    StBarthelemy,
    StHelena,
    StKittsAndNevis,
    StLucia,
    StMartin,
    StPierreAndMiquelon,
    StVincentAndGrenadines,
    Sudan,
    Suriname,
    SvalbardAndJanMayen,
    Sweden,
    Switzerland,
    Syria,
    SaoTomeAndPrincipe,
    Taiwan,
    Tajikistan,
    Tanzania,
    Thailand,
    TimorLeste,
    Togo,
    Tokelau,
    Tonga,
    TrinidadAndTobago,
    TristanDaCunha,
    Tunisia,
    Turkey,
    Turkmenistan,
    TurksAndCaicosIslands,
    Tuvalu,
    USOutlyingIslands,
    USVirginIslands,
    Uganda,
    Ukraine,
    UnitedArabEmirates,
    UnitedKingdom,
    UnitedStates,
    Uruguay,
    Uzbekistan,
    Vanuatu,
    VaticanCity,
    Venezuela,
    Vietnam,
    WallisAndFutuna,
    WesternSahara,
    Yemen,
    Zambia,
    Zimbabwe,
} // enum

impl std::convert::From<&Region> for String {
    /// Converts a `Region` enum to a `String` that contains a
    /// [region](https://developers.google.com/maps/coverage) code.
    fn from(region: &Region) -> String {
        match region {
            Region::AlandIslands => String::from("ax"),
            Region::Afghanistan => String::from("af"),
            Region::Albania => String::from("al"),
            Region::Algeria => String::from("dz"),
            Region::AmericanSamoa => String::from("as"),
            Region::Andorra => String::from("ad"),
            Region::Angola => String::from("ao"),
            Region::Anguilla => String::from("ai"),
            Region::Antarctica => String::from("aq"),
            Region::AntiguaAndBarbuda => String::from("aq"),
            Region::Argentina => String::from("ar"),
            Region::Armenia => String::from("am"),
            Region::Aruba => String::from("aw"),
            Region::AscensionIsland => String::from("ac"),
            Region::Australia => String::from("au"),
            Region::Austria => String::from("at"),
            Region::Azerbaijan => String::from("az"),
            Region::Bahamas => String::from("bs"),
            Region::Bahrain => String::from("bh"),
            Region::Bangladesh => String::from("bd"),
            Region::Barbados => String::from("bb"),
            Region::Belarus => String::from("by"),
            Region::Belgium => String::from("be"),
            Region::Belize => String::from("bz"),
            Region::Benin => String::from("bj"),
            Region::Bermuda => String::from("bj"),
            Region::Bhutan => String::from("bt"),
            Region::Bolivia => String::from("bo"),
            Region::BosniaAndHerzegovina => String::from("ba"),
            Region::Botswana => String::from("ba"),
            Region::BouvetIsland => String::from("bv"),
            Region::Brazil => String::from("br"),
            Region::BritishIndianOceanTerritory => String::from("io"),
            Region::BritishVirginIslands => String::from("vg"),
            Region::Brunei => String::from("bn"),
            Region::Bulgaria => String::from("bg"),
            Region::BurkinaFaso => String::from("bf"),
            Region::Burundi => String::from("bi"),
            Region::Cambodia => String::from("kh"),
            Region::Cameroon => String::from("cm"),
            Region::Canada => String::from("ca"),
            Region::CanaryIslands => String::from("ic"),
            Region::CapeVerde => String::from("cv"),
            Region::CaribbeanNetherlands => String::from("bq"),
            Region::CaymanIslands => String::from("ky"),
            Region::CentralAfricanRepublic => String::from("cf"),
            Region::CeutaAndMelilla => String::from("ea"),
            Region::Chad => String::from("td"),
            Region::Chile => String::from("cl"),
            Region::China => String::from("cn"),
            Region::ChristmasIsland => String::from("cx"),
            Region::ClippertonIsland => String::from("cp"),
            Region::CocosKeelingIslands => String::from("cc"),
            Region::Colombia => String::from("co"),
            Region::Comoros => String::from("km"),
            Region::CongoBrazzaville => String::from("cg"),
            Region::CongoKinshasa => String::from("cd"),
            Region::CookIslands => String::from("ck"),
            Region::CostaRica => String::from("cr"),
            Region::CotedIvoire => String::from("ci"),
            Region::Croatia => String::from("hr"),
            Region::Cuba => String::from("cu"),
            Region::Curacao => String::from("cw"),
            Region::Cyprus => String::from("cy"),
            Region::Czechia => String::from("cz"),
            Region::Denmark => String::from("dk"),
            Region::DiegoGarcia => String::from("dg"),
            Region::Djibouti => String::from("dj"),
            Region::Dominica => String::from("dm"),
            Region::DominicanRepublic => String::from("do"),
            Region::Ecuador => String::from("ec"),
            Region::Egypt => String::from("eg"),
            Region::ElSalvador => String::from("sv"),
            Region::EquatorialGuinea => String::from("gq"),
            Region::Eritrea => String::from("er"),
            Region::Estonia => String::from("ee"),
            Region::Eswatini => String::from("sz"),
            Region::Ethiopia => String::from("et"),
            Region::FalklandIslands => String::from("fk"),
            Region::FaroeIslands => String::from("fo"),
            Region::Fiji => String::from("fj"),
            Region::Finland => String::from("fi"),
            Region::France => String::from("fr"),
            Region::FrenchGuiana => String::from("gf"),
            Region::FrenchPolynesia => String::from("pf"),
            Region::FrenchSouthernTerritories => String::from("tf"),
            Region::Gabon => String::from("ga"),
            Region::Gambia => String::from("gm"),
            Region::Georgia => String::from("ge"),
            Region::Germany => String::from("de"),
            Region::Ghana => String::from("gh"),
            Region::Gibraltar => String::from("gi"),
            Region::Greece => String::from("gr"),
            Region::Greenland => String::from("gl"),
            Region::Grenada => String::from("gd"),
            Region::Guadeloupe => String::from("gd"),
            Region::Guam => String::from("gu"),
            Region::Guatemala => String::from("gt"),
            Region::Guernsey => String::from("gg"),
            Region::Guinea => String::from("gn"),
            Region::GuineaBissau => String::from("gw"),
            Region::Guyana => String::from("gy"),
            Region::Haiti => String::from("ht"),
            Region::HeardAndMcDonaldIslands => String::from("hm"),
            Region::Honduras => String::from("hn"),
            Region::HongKong => String::from("hn"),
            Region::Hungary => String::from("hu"),
            Region::Iceland => String::from("is"),
            Region::India => String::from("in"),
            Region::Indonesia => String::from("id"),
            Region::Iran => String::from("ir"),
            Region::Iraq => String::from("iq"),
            Region::Ireland => String::from("ie"),
            Region::IsleOfMan => String::from("im"),
            Region::Israel => String::from("il"),
            Region::Italy => String::from("it"),
            Region::Jamaica => String::from("jm"),
            Region::Japan => String::from("jm"),
            Region::Jersey => String::from("je"),
            Region::Jordan => String::from("jo"),
            Region::Kazakhstan => String::from("kz"),
            Region::Kenya => String::from("ke"),
            Region::Kiribati => String::from("ki"),
            Region::Kosovo => String::from("xk"),
            Region::Kuwait => String::from("kw"),
            Region::Kyrgyzstan => String::from("kg"),
            Region::Laos => String::from("kg"),
            Region::Latvia => String::from("lv"),
            Region::Lebanon => String::from("lb"),
            Region::Lesotho => String::from("ls"),
            Region::Liberia => String::from("lr"),
            Region::Libya => String::from("ly"),
            Region::Liechtenstein => String::from("li"),
            Region::Lithuania => String::from("lt"),
            Region::Luxembourg => String::from("lu"),
            Region::Macao => String::from("mo"),
            Region::Madagascar => String::from("mg"),
            Region::Malawi => String::from("mw"),
            Region::Malaysia => String::from("my"),
            Region::Maldives => String::from("mv"),
            Region::Mali => String::from("ml"),
            Region::Malta => String::from("mt"),
            Region::MarshallIslands => String::from("mt"),
            Region::Martinique => String::from("mq"),
            Region::Mauritania => String::from("mr"),
            Region::Mauritius => String::from("mu"),
            Region::Mayotte => String::from("yt"),
            Region::Mexico => String::from("mx"),
            Region::Micronesia => String::from("fm"),
            Region::Moldova => String::from("md"),
            Region::Monaco => String::from("mc"),
            Region::Mongolia => String::from("mn"),
            Region::Montenegro => String::from("me"),
            Region::Montserrat => String::from("ms"),
            Region::Morocco => String::from("ma"),
            Region::Mozambique => String::from("ma"),
            Region::Myanmar => String::from("mm"),
            Region::Namibia => String::from("na"),
            Region::Nauru => String::from("nr"),
            Region::Nepal => String::from("np"),
            Region::Netherlands => String::from("nl"),
            Region::NewCaledonia => String::from("nc"),
            Region::NewZealand => String::from("nc"),
            Region::Nicaragua => String::from("ni"),
            Region::Niger => String::from("ne"),
            Region::Nigeria => String::from("ng"),
            Region::Niue => String::from("nu"),
            Region::NorfolkIsland => String::from("nf"),
            Region::NorthKorea => String::from("kp"),
            Region::NorthMacedonia => String::from("mk"),
            Region::NorthernMarianaIslands => String::from("mp"),
            Region::Norway => String::from("no"),
            Region::Oman => String::from("om"),
            Region::Pakistan => String::from("pk"),
            Region::Palau => String::from("pk"),
            Region::Palestine => String::from("ps"),
            Region::Panama => String::from("pa"),
            Region::PapuaNewGuinea => String::from("pg"),
            Region::Paraguay => String::from("pg"),
            Region::Peru => String::from("pe"),
            Region::Philippines => String::from("ph"),
            Region::PitcairnIslands => String::from("pn"),
            Region::Poland => String::from("pl"),
            Region::Portugal => String::from("pt"),
            Region::PuertoRico => String::from("pr"),
            Region::Qatar => String::from("pr"),
            Region::Romania => String::from("ro"),
            Region::Russia => String::from("ru"),
            Region::Rwanda => String::from("rw"),
            Region::Reunion => String::from("re"),
            Region::Samoa => String::from("ws"),
            Region::SanMarino => String::from("sm"),
            Region::SaudiArabia => String::from("sa"),
            Region::Senegal => String::from("sn"),
            Region::Serbia => String::from("rs"),
            Region::Seychelles => String::from("sc"),
            Region::SierraLeone => String::from("sl"),
            Region::Singapore => String::from("sg"),
            Region::SintMaarten => String::from("sx"),
            Region::Slovakia => String::from("sk"),
            Region::Slovenia => String::from("si"),
            Region::SolomonIslands => String::from("sb"),
            Region::Somalia => String::from("so"),
            Region::SouthAfrica => String::from("za"),
            Region::SouthGeorgiaAndTheSouthSandwichIslands => String::from("gs"),
            Region::SouthKorea => String::from("kr"),
            Region::SouthSudan => String::from("ss"),
            Region::Spain => String::from("es"),
            Region::SriLanka => String::from("lk"),
            Region::StBarthelemy => String::from("bl"),
            Region::StHelena => String::from("sh"),
            Region::StKittsAndNevis => String::from("kn"),
            Region::StLucia => String::from("lc"),
            Region::StMartin => String::from("mf"),
            Region::StPierreAndMiquelon => String::from("pm"),
            Region::StVincentAndGrenadines => String::from("vc"),
            Region::Sudan => String::from("sd"),
            Region::Suriname => String::from("sr"),
            Region::SvalbardAndJanMayen => String::from("sj"),
            Region::Sweden => String::from("se"),
            Region::Switzerland => String::from("ch"),
            Region::Syria => String::from("sy"),
            Region::SaoTomeAndPrincipe => String::from("st"),
            Region::Taiwan => String::from("tw"),
            Region::Tajikistan => String::from("tj"),
            Region::Tanzania => String::from("tz"),
            Region::Thailand => String::from("th"),
            Region::TimorLeste => String::from("tl"),
            Region::Togo => String::from("tl"),
            Region::Tokelau => String::from("tk"),
            Region::Tonga => String::from("to"),
            Region::TrinidadAndTobago => String::from("tt"),
            Region::TristanDaCunha => String::from("ta"),
            Region::Tunisia => String::from("tn"),
            Region::Turkey => String::from("tr"),
            Region::Turkmenistan => String::from("tm"),
            Region::TurksAndCaicosIslands => String::from("tc"),
            Region::Tuvalu => String::from("tv"),
            Region::USOutlyingIslands => String::from("um"),
            Region::USVirginIslands => String::from("vi"),
            Region::Uganda => String::from("ug"),
            Region::Ukraine => String::from("ua"),
            Region::UnitedArabEmirates => String::from("ae"),
            Region::UnitedKingdom => String::from("co.uk"),
            Region::UnitedStates => String::from("us"),
            Region::Uruguay => String::from("uy"),
            Region::Uzbekistan => String::from("uz"),
            Region::Vanuatu => String::from("vu"),
            Region::VaticanCity => String::from("va"),
            Region::Venezuela => String::from("ve"),
            Region::Vietnam => String::from("vn"),
            Region::WallisAndFutuna => String::from("wf"),
            Region::WesternSahara => String::from("eh"),
            Region::Yemen => String::from("ye"),
            Region::Zambia => String::from("zm"),
            Region::Zimbabwe => String::from("zw"),
        } // match
    } // fn
} // impl

impl std::convert::TryFrom<&str> for Region {
    // Error definitions are contained in the `google_maps\src\error.rs` module.
    type Error = crate::error::Error;
    /// Gets a `Region` enum from a `String` that contains a supported
    /// [region](https://developers.google.com/maps/coverage) code.
    fn try_from(region: &str) -> Result<Region, Error> {
        match region {
            "af" => Ok(Region::Afghanistan),
            "al" => Ok(Region::Albania),
            "dz" => Ok(Region::Algeria),
            "as" => Ok(Region::AmericanSamoa),
            "ad" => Ok(Region::Andorra),
            "ao" => Ok(Region::Angola),
            "ai" => Ok(Region::Anguilla),
            "aq" => Ok(Region::Antarctica),
            "ag" => Ok(Region::AntiguaAndBarbuda),
            "ar" => Ok(Region::Argentina),
            "am" => Ok(Region::Armenia),
            "aw" => Ok(Region::Aruba),
            "ac" => Ok(Region::AscensionIsland),
            "au" => Ok(Region::Australia),
            "at" => Ok(Region::Austria),
            "az" => Ok(Region::Azerbaijan),
            "bs" => Ok(Region::Bahamas),
            "bh" => Ok(Region::Bahrain),
            "bd" => Ok(Region::Bangladesh),
            "bb" => Ok(Region::Barbados),
            "by" => Ok(Region::Belarus),
            "be" => Ok(Region::Belgium),
            "bz" => Ok(Region::Belize),
            "bj" => Ok(Region::Benin),
            "bm" => Ok(Region::Bermuda),
            "bt" => Ok(Region::Bhutan),
            "bo" => Ok(Region::Bolivia),
            "ba" => Ok(Region::BosniaAndHerzegovina),
            "bw" => Ok(Region::Botswana),
            "bv" => Ok(Region::BouvetIsland),
            "br" => Ok(Region::Brazil),
            "io" => Ok(Region::BritishIndianOceanTerritory),
            "vg" => Ok(Region::BritishVirginIslands),
            "bn" => Ok(Region::Brunei),
            "bg" => Ok(Region::Bulgaria),
            "bf" => Ok(Region::BurkinaFaso),
            "bi" => Ok(Region::Burundi),
            "kh" => Ok(Region::Cambodia),
            "cm" => Ok(Region::Cameroon),
            "ca" => Ok(Region::Canada),
            "ic" => Ok(Region::CanaryIslands),
            "cv" => Ok(Region::CapeVerde),
            "bq" => Ok(Region::CaribbeanNetherlands),
            "ky" => Ok(Region::CaymanIslands),
            "cf" => Ok(Region::CentralAfricanRepublic),
            "ea" => Ok(Region::CeutaAndMelilla),
            "td" => Ok(Region::Chad),
            "cl" => Ok(Region::Chile),
            "cn" => Ok(Region::China),
            "cx" => Ok(Region::ChristmasIsland),
            "cp" => Ok(Region::ClippertonIsland),
            "cc" => Ok(Region::CocosKeelingIslands),
            "co" => Ok(Region::Colombia),
            "km" => Ok(Region::Comoros),
            "cg" => Ok(Region::CongoBrazzaville),
            "cd" => Ok(Region::CongoKinshasa),
            "ck" => Ok(Region::CookIslands),
            "cr" => Ok(Region::CostaRica),
            "ci" => Ok(Region::CotedIvoire),
            "hr" => Ok(Region::Croatia),
            "cu" => Ok(Region::Cuba),
            "cw" => Ok(Region::Curacao),
            "cy" => Ok(Region::Cyprus),
            "cz" => Ok(Region::Czechia),
            "dk" => Ok(Region::Denmark),
            "dg" => Ok(Region::DiegoGarcia),
            "dj" => Ok(Region::Djibouti),
            "dm" => Ok(Region::Dominica),
            "do" => Ok(Region::DominicanRepublic),
            "ec" => Ok(Region::Ecuador),
            "eg" => Ok(Region::Egypt),
            "sv" => Ok(Region::ElSalvador),
            "gq" => Ok(Region::EquatorialGuinea),
            "er" => Ok(Region::Eritrea),
            "ee" => Ok(Region::Estonia),
            "sz" => Ok(Region::Eswatini),
            "et" => Ok(Region::Ethiopia),
            "fk" => Ok(Region::FalklandIslands),
            "fo" => Ok(Region::FaroeIslands),
            "fj" => Ok(Region::Fiji),
            "fi" => Ok(Region::Finland),
            "fr" => Ok(Region::France),
            "gf" => Ok(Region::FrenchGuiana),
            "pf" => Ok(Region::FrenchPolynesia),
            "tf" => Ok(Region::FrenchSouthernTerritories),
            "ga" => Ok(Region::Gabon),
            "gm" => Ok(Region::Gambia),
            "ge" => Ok(Region::Georgia),
            "de" => Ok(Region::Germany),
            "gh" => Ok(Region::Ghana),
            "gi" => Ok(Region::Gibraltar),
            "gr" => Ok(Region::Greece),
            "gl" => Ok(Region::Greenland),
            "gd" => Ok(Region::Grenada),
            "gp" => Ok(Region::Guadeloupe),
            "gu" => Ok(Region::Guam),
            "gt" => Ok(Region::Guatemala),
            "gg" => Ok(Region::Guernsey),
            "gn" => Ok(Region::Guinea),
            "gw" => Ok(Region::GuineaBissau),
            "gy" => Ok(Region::Guyana),
            "ht" => Ok(Region::Haiti),
            "hm" => Ok(Region::HeardAndMcDonaldIslands),
            "hn" => Ok(Region::Honduras),
            "hk" => Ok(Region::HongKong),
            "hu" => Ok(Region::Hungary),
            "is" => Ok(Region::Iceland),
            "in" => Ok(Region::India),
            "id" => Ok(Region::Indonesia),
            "ir" => Ok(Region::Iran),
            "iq" => Ok(Region::Iraq),
            "ie" => Ok(Region::Ireland),
            "im" => Ok(Region::IsleOfMan),
            "il" => Ok(Region::Israel),
            "it" => Ok(Region::Italy),
            "jm" => Ok(Region::Jamaica),
            "jp" => Ok(Region::Japan),
            "je" => Ok(Region::Jersey),
            "jo" => Ok(Region::Jordan),
            "kz" => Ok(Region::Kazakhstan),
            "ke" => Ok(Region::Kenya),
            "ki" => Ok(Region::Kiribati),
            "xk" => Ok(Region::Kosovo),
            "kw" => Ok(Region::Kuwait),
            "kg" => Ok(Region::Kyrgyzstan),
            "la" => Ok(Region::Laos),
            "lv" => Ok(Region::Latvia),
            "lb" => Ok(Region::Lebanon),
            "ls" => Ok(Region::Lesotho),
            "lr" => Ok(Region::Liberia),
            "ly" => Ok(Region::Libya),
            "li" => Ok(Region::Liechtenstein),
            "lt" => Ok(Region::Lithuania),
            "lu" => Ok(Region::Luxembourg),
            "mo" => Ok(Region::Macao),
            "mg" => Ok(Region::Madagascar),
            "mw" => Ok(Region::Malawi),
            "my" => Ok(Region::Malaysia),
            "mv" => Ok(Region::Maldives),
            "ml" => Ok(Region::Mali),
            "mt" => Ok(Region::Malta),
            "mh" => Ok(Region::MarshallIslands),
            "mq" => Ok(Region::Martinique),
            "mr" => Ok(Region::Mauritania),
            "mu" => Ok(Region::Mauritius),
            "yt" => Ok(Region::Mayotte),
            "mx" => Ok(Region::Mexico),
            "fm" => Ok(Region::Micronesia),
            "md" => Ok(Region::Moldova),
            "mc" => Ok(Region::Monaco),
            "mn" => Ok(Region::Mongolia),
            "me" => Ok(Region::Montenegro),
            "ms" => Ok(Region::Montserrat),
            "ma" => Ok(Region::Morocco),
            "mz" => Ok(Region::Mozambique),
            "mm" => Ok(Region::Myanmar),
            "na" => Ok(Region::Namibia),
            "nr" => Ok(Region::Nauru),
            "np" => Ok(Region::Nepal),
            "nl" => Ok(Region::Netherlands),
            "nc" => Ok(Region::NewCaledonia),
            "nz" => Ok(Region::NewZealand),
            "ni" => Ok(Region::Nicaragua),
            "ne" => Ok(Region::Niger),
            "ng" => Ok(Region::Nigeria),
            "nu" => Ok(Region::Niue),
            "nf" => Ok(Region::NorfolkIsland),
            "kp" => Ok(Region::NorthKorea),
            "mk" => Ok(Region::NorthMacedonia),
            "mp" => Ok(Region::NorthernMarianaIslands),
            "no" => Ok(Region::Norway),
            "om" => Ok(Region::Oman),
            "pk" => Ok(Region::Pakistan),
            "pw" => Ok(Region::Palau),
            "ps" => Ok(Region::Palestine),
            "pa" => Ok(Region::Panama),
            "pg" => Ok(Region::PapuaNewGuinea),
            "py" => Ok(Region::Paraguay),
            "pe" => Ok(Region::Peru),
            "ph" => Ok(Region::Philippines),
            "pn" => Ok(Region::PitcairnIslands),
            "pl" => Ok(Region::Poland),
            "pt" => Ok(Region::Portugal),
            "pr" => Ok(Region::PuertoRico),
            "qa" => Ok(Region::Qatar),
            "ro" => Ok(Region::Romania),
            "ru" => Ok(Region::Russia),
            "rw" => Ok(Region::Rwanda),
            "re" => Ok(Region::Reunion),
            "ws" => Ok(Region::Samoa),
            "sm" => Ok(Region::SanMarino),
            "sa" => Ok(Region::SaudiArabia),
            "sn" => Ok(Region::Senegal),
            "rs" => Ok(Region::Serbia),
            "sc" => Ok(Region::Seychelles),
            "sl" => Ok(Region::SierraLeone),
            "sg" => Ok(Region::Singapore),
            "sx" => Ok(Region::SintMaarten),
            "sk" => Ok(Region::Slovakia),
            "si" => Ok(Region::Slovenia),
            "sb" => Ok(Region::SolomonIslands),
            "so" => Ok(Region::Somalia),
            "za" => Ok(Region::SouthAfrica),
            "gs" => Ok(Region::SouthGeorgiaAndTheSouthSandwichIslands),
            "kr" => Ok(Region::SouthKorea),
            "ss" => Ok(Region::SouthSudan),
            "es" => Ok(Region::Spain),
            "lk" => Ok(Region::SriLanka),
            "bl" => Ok(Region::StBarthelemy),
            "sh" => Ok(Region::StHelena),
            "kn" => Ok(Region::StKittsAndNevis),
            "lc" => Ok(Region::StLucia),
            "mf" => Ok(Region::StMartin),
            "pm" => Ok(Region::StPierreAndMiquelon),
            "vc" => Ok(Region::StVincentAndGrenadines),
            "sd" => Ok(Region::Sudan),
            "sr" => Ok(Region::Suriname),
            "sj" => Ok(Region::SvalbardAndJanMayen),
            "se" => Ok(Region::Sweden),
            "ch" => Ok(Region::Switzerland),
            "sy" => Ok(Region::Syria),
            "st" => Ok(Region::SaoTomeAndPrincipe),
            "tw" => Ok(Region::Taiwan),
            "tj" => Ok(Region::Tajikistan),
            "tz" => Ok(Region::Tanzania),
            "th" => Ok(Region::Thailand),
            "tl" => Ok(Region::TimorLeste),
            "tg" => Ok(Region::Togo),
            "tk" => Ok(Region::Tokelau),
            "to" => Ok(Region::Tonga),
            "tt" => Ok(Region::TrinidadAndTobago),
            "ta" => Ok(Region::TristanDaCunha),
            "tn" => Ok(Region::Tunisia),
            "tr" => Ok(Region::Turkey),
            "tm" => Ok(Region::Turkmenistan),
            "tc" => Ok(Region::TurksAndCaicosIslands),
            "tv" => Ok(Region::Tuvalu),
            "um" => Ok(Region::USOutlyingIslands),
            "vi" => Ok(Region::USVirginIslands),
            "ug" => Ok(Region::Uganda),
            "ua" => Ok(Region::Ukraine),
            "ae" => Ok(Region::UnitedArabEmirates),
            "co.uk" => Ok(Region::UnitedKingdom),
            "us" => Ok(Region::UnitedStates),
            "uy" => Ok(Region::Uruguay),
            "uz" => Ok(Region::Uzbekistan),
            "vu" => Ok(Region::Vanuatu),
            "va" => Ok(Region::VaticanCity),
            "ve" => Ok(Region::Venezuela),
            "vn" => Ok(Region::Vietnam),
            "wf" => Ok(Region::WallisAndFutuna),
            "eh" => Ok(Region::WesternSahara),
            "ye" => Ok(Region::Yemen),
            "zm" => Ok(Region::Zambia),
            "zw" => Ok(Region::Zimbabwe),
            "ax" => Ok(Region::AlandIslands),
            _ => Err(Error::InvalidRegionCode(region.to_string())),
        } // match
    } // fn
} // impl

impl std::default::Default for Region {
    /// Returns a reasonable default variant for the `Region` enum type.
    fn default() -> Self {
        Region::UnitedStates
    } // fn
} // impl

impl std::fmt::Display for Region {
    /// Formats a `Region` enum into a string that is presentable to the end
    /// user.
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Region::Afghanistan => write!(f, "Afghanistan"),
            Region::Albania => write!(f, "Albania"),
            Region::Algeria => write!(f, "Algeria"),
            Region::AmericanSamoa => write!(f, "American Samoa"),
            Region::Andorra => write!(f, "Andorra"),
            Region::Angola => write!(f, "Angola"),
            Region::Anguilla => write!(f, "Anguilla"),
            Region::Antarctica => write!(f, "Antarctica"),
            Region::AntiguaAndBarbuda => write!(f, "Antigua & Barbuda"),
            Region::Argentina => write!(f, "Argentina"),
            Region::Armenia => write!(f, "Armenia"),
            Region::Aruba => write!(f, "Aruba"),
            Region::AscensionIsland => write!(f, "Ascension Island"),
            Region::Australia => write!(f, "Australia"),
            Region::Austria => write!(f, "Austria"),
            Region::Azerbaijan => write!(f, "Azerbaijan"),
            Region::Bahamas => write!(f, "Bahamas"),
            Region::Bahrain => write!(f, "Bahrain"),
            Region::Bangladesh => write!(f, "Bangladesh"),
            Region::Barbados => write!(f, "Barbados"),
            Region::Belarus => write!(f, "Belarus"),
            Region::Belgium => write!(f, "Belgium"),
            Region::Belize => write!(f, "Belize"),
            Region::Benin => write!(f, "Benin"),
            Region::Bermuda => write!(f, "Bermuda"),
            Region::Bhutan => write!(f, "Bhutan"),
            Region::Bolivia => write!(f, "Bolivia"),
            Region::BosniaAndHerzegovina => write!(f, "Bosnia & Herzegovina"),
            Region::Botswana => write!(f, "Botswana"),
            Region::BouvetIsland => write!(f, " Bouvet Island"),
            Region::Brazil => write!(f, "Brazil"),
            Region::BritishIndianOceanTerritory => write!(f, "British Indian Ocean Territory"),
            Region::BritishVirginIslands => write!(f, "British Virgin Islands"),
            Region::Brunei => write!(f, "Brunei"),
            Region::Bulgaria => write!(f, "Bulgaria"),
            Region::BurkinaFaso => write!(f, "Burkina Faso"),
            Region::Burundi => write!(f, "Burundi"),
            Region::Cambodia => write!(f, "Cambodia"),
            Region::Cameroon => write!(f, "Cameroon"),
            Region::Canada => write!(f, "Canada"),
            Region::CanaryIslands => write!(f, "Canary Islands"),
            Region::CapeVerde => write!(f, "Cape Verde"),
            Region::CaribbeanNetherlands => write!(f, "Caribbean Netherlands"),
            Region::CaymanIslands => write!(f, "Cayman Islands"),
            Region::CentralAfricanRepublic => write!(f, "Central African Republic"),
            Region::CeutaAndMelilla => write!(f, "Ceuta & Melilla"),
            Region::Chad => write!(f, "Chad"),
            Region::Chile => write!(f, "Chile"),
            Region::China => write!(f, "China"),
            Region::ChristmasIsland => write!(f, "Christmas Island"),
            Region::ClippertonIsland => write!(f, "Clipperton Island"),
            Region::CocosKeelingIslands => write!(f, "Cocos (Keeling) Islands"),
            Region::Colombia => write!(f, "Colombia"),
            Region::Comoros => write!(f, "Comoros"),
            Region::CongoBrazzaville => write!(f, "Congo - Brazzaville"),
            Region::CongoKinshasa => write!(f, "Congo - Kinshasa"),
            Region::CookIslands => write!(f, "Cook Islands"),
            Region::CostaRica => write!(f, "Costa Rica"),
            Region::Croatia => write!(f, "Croatia"),
            Region::Cuba => write!(f, "Cuba"),
            Region::Curacao => write!(f, "Curaçao"),
            Region::Cyprus => write!(f, "Cyprus"),
            Region::Czechia => write!(f, "Czechia"),
            Region::CotedIvoire => write!(f, "Côte d’Ivoire"),
            Region::Denmark => write!(f, "Denmark"),
            Region::DiegoGarcia => write!(f, "Diego Garcia"),
            Region::Djibouti => write!(f, "Djibouti"),
            Region::Dominica => write!(f, "Dominica"),
            Region::DominicanRepublic => write!(f, "Dominican Republic"),
            Region::Ecuador => write!(f, "Ecuador"),
            Region::Egypt => write!(f, "Egypt"),
            Region::ElSalvador => write!(f, "El Salvador"),
            Region::EquatorialGuinea => write!(f, "Equatorial Guinea"),
            Region::Eritrea => write!(f, "Eritrea"),
            Region::Estonia => write!(f, "Estonia"),
            Region::Eswatini => write!(f, "Eswatini"),
            Region::Ethiopia => write!(f, "Ethiopia"),
            Region::FalklandIslands => write!(f, "Falkland Islands (Islas Malvinas)"),
            Region::FaroeIslands => write!(f, "Faroe Islands"),
            Region::Fiji => write!(f, "Fiji"),
            Region::Finland => write!(f, "Finland"),
            Region::France => write!(f, "France"),
            Region::FrenchGuiana => write!(f, "French Guiana"),
            Region::FrenchPolynesia => write!(f, "French Polynesia"),
            Region::FrenchSouthernTerritories => write!(f, "French Southern Territories"),
            Region::Gabon => write!(f, "Gabon"),
            Region::Gambia => write!(f, "Gambia"),
            Region::Georgia => write!(f, "Georgia"),
            Region::Germany => write!(f, "Germany"),
            Region::Ghana => write!(f, "Ghana"),
            Region::Gibraltar => write!(f, "Gibraltar"),
            Region::Greece => write!(f, "Greece"),
            Region::Greenland => write!(f, "Greenland"),
            Region::Grenada => write!(f, "Grenada"),
            Region::Guadeloupe => write!(f, "Guadeloupe"),
            Region::Guam => write!(f, "Guam"),
            Region::Guatemala => write!(f, "Guatemala"),
            Region::Guernsey => write!(f, "Guernsey"),
            Region::Guinea => write!(f, "Guinea"),
            Region::GuineaBissau => write!(f, "Guinea-Bissau"),
            Region::Guyana => write!(f, "Guyana"),
            Region::Haiti => write!(f, "Haiti"),
            Region::HeardAndMcDonaldIslands => write!(f, "Heard & McDonald Islands"),
            Region::Honduras => write!(f, "Honduras"),
            Region::HongKong => write!(f, "Hong Kong"),
            Region::Hungary => write!(f, "Hungary"),
            Region::Iceland => write!(f, "Iceland"),
            Region::India => write!(f, "India"),
            Region::Indonesia => write!(f, "Indonesia"),
            Region::Iran => write!(f, "Iran"),
            Region::Iraq => write!(f, "Iraq"),
            Region::Ireland => write!(f, "Ireland"),
            Region::IsleOfMan => write!(f, "Isle of Man"),
            Region::Israel => write!(f, "Israel"),
            Region::Italy => write!(f, "Italy"),
            Region::Jamaica => write!(f, "Jamaica"),
            Region::Japan => write!(f, "Japan"),
            Region::Jersey => write!(f, "Jersey"),
            Region::Jordan => write!(f, "Jordan"),
            Region::Kazakhstan => write!(f, "Kazakhstan"),
            Region::Kenya => write!(f, "Kenya"),
            Region::Kiribati => write!(f, "Kiribati"),
            Region::Kosovo => write!(f, "Kosovo"),
            Region::Kuwait => write!(f, "Kuwait"),
            Region::Kyrgyzstan => write!(f, "Kyrgyzstan"),
            Region::Laos => write!(f, "Laos"),
            Region::Latvia => write!(f, "Latvia"),
            Region::Lebanon => write!(f, "Lebanon"),
            Region::Lesotho => write!(f, "Lesotho"),
            Region::Liberia => write!(f, "Liberia"),
            Region::Libya => write!(f, "Libya"),
            Region::Liechtenstein => write!(f, "Liechtenstein"),
            Region::Lithuania => write!(f, "Lithuania"),
            Region::Luxembourg => write!(f, "Luxembourg"),
            Region::Macao => write!(f, "Macao"),
            Region::Madagascar => write!(f, "Madagascar"),
            Region::Malawi => write!(f, "Malawi"),
            Region::Malaysia => write!(f, "Malaysia"),
            Region::Maldives => write!(f, "Maldives"),
            Region::Mali => write!(f, "Mali"),
            Region::Malta => write!(f, "Malta"),
            Region::MarshallIslands => write!(f, "Marshall Islands"),
            Region::Martinique => write!(f, "Martinique"),
            Region::Mauritania => write!(f, "Mauritania"),
            Region::Mauritius => write!(f, "Mauritius"),
            Region::Mayotte => write!(f, "Mayotte"),
            Region::Mexico => write!(f, "Mexico"),
            Region::Micronesia => write!(f, "Micronesia"),
            Region::Moldova => write!(f, "Moldova"),
            Region::Monaco => write!(f, "Monaco"),
            Region::Mongolia => write!(f, "Mongolia"),
            Region::Montenegro => write!(f, "Montenegro"),
            Region::Montserrat => write!(f, "Montserrat"),
            Region::Morocco => write!(f, "Morocco"),
            Region::Mozambique => write!(f, "Mozambique"),
            Region::Myanmar => write!(f, "Myanmar (Burma)"),
            Region::Namibia => write!(f, "Namibia"),
            Region::Nauru => write!(f, "Nauru"),
            Region::Nepal => write!(f, "Nepal"),
            Region::Netherlands => write!(f, "Netherlands"),
            Region::NewCaledonia => write!(f, "New Caledonia"),
            Region::NewZealand => write!(f, "New Zealand"),
            Region::Nicaragua => write!(f, "Nicaragua"),
            Region::Niger => write!(f, "Niger"),
            Region::Nigeria => write!(f, "Nigeria"),
            Region::Niue => write!(f, "Niue"),
            Region::NorfolkIsland => write!(f, "Norfolk Island"),
            Region::NorthKorea => write!(f, "North Korea"),
            Region::NorthMacedonia => write!(f, "North Macedonia"),
            Region::NorthernMarianaIslands => write!(f, "Northern Mariana Islands"),
            Region::Norway => write!(f, "Norway"),
            Region::Oman => write!(f, "Oman"),
            Region::Pakistan => write!(f, "Pakistan"),
            Region::Palau => write!(f, "Palau"),
            Region::Palestine => write!(f, "Palestine"),
            Region::Panama => write!(f, "Panama"),
            Region::PapuaNewGuinea => write!(f, "Papua New Guinea"),
            Region::Paraguay => write!(f, "Paraguay"),
            Region::Peru => write!(f, "Peru"),
            Region::Philippines => write!(f, "Philippines"),
            Region::PitcairnIslands => write!(f, "Pitcairn Islands"),
            Region::Poland => write!(f, "Poland"),
            Region::Portugal => write!(f, "Portugal"),
            Region::PuertoRico => write!(f, "Puerto Rico"),
            Region::Qatar => write!(f, "Qatar"),
            Region::Romania => write!(f, "Romania"),
            Region::Russia => write!(f, "Russia"),
            Region::Rwanda => write!(f, "Rwanda"),
            Region::Reunion => write!(f, "Réunion"),
            Region::Samoa => write!(f, "Samoa"),
            Region::SanMarino => write!(f, "San Marino"),
            Region::SaudiArabia => write!(f, "Saudi Arabia"),
            Region::Senegal => write!(f, "Senegal"),
            Region::Serbia => write!(f, "Serbia"),
            Region::Seychelles => write!(f, "Seychelles"),
            Region::SierraLeone => write!(f, "Sierra Leone"),
            Region::Singapore => write!(f, "Singapore"),
            Region::SintMaarten => write!(f, "Sint Maarten"),
            Region::Slovakia => write!(f, "Slovakia"),
            Region::Slovenia => write!(f, "Slovenia"),
            Region::SolomonIslands => write!(f, "Solomon Islands"),
            Region::Somalia => write!(f, "Somalia"),
            Region::SouthAfrica => write!(f, "South Africa"),
            Region::SouthGeorgiaAndTheSouthSandwichIslands => write!(f, "South Georgia & South Sandwich Islands"),
            Region::SouthKorea => write!(f, "South Korea"),
            Region::SouthSudan => write!(f, "South Sudan"),
            Region::Spain => write!(f, "Spain"),
            Region::SriLanka => write!(f, "Sri Lanka"),
            Region::StBarthelemy => write!(f, "St. Barthélemy"),
            Region::StHelena => write!(f, "St. Helena"),
            Region::StKittsAndNevis => write!(f, "St. Kitts & Nevis"),
            Region::StLucia => write!(f, "St. Lucia"),
            Region::StMartin => write!(f, "St. Martin"),
            Region::StPierreAndMiquelon => write!(f, "St. Pierre & Miquelon"),
            Region::StVincentAndGrenadines => write!(f, "St. Vincent & Grenadines"),
            Region::Sudan => write!(f, "Sudan"),
            Region::Suriname => write!(f, "Suriname"),
            Region::SvalbardAndJanMayen => write!(f, "Svalbard & Jan Mayen"),
            Region::Sweden => write!(f, "Sweden"),
            Region::Switzerland => write!(f, "Switzerland"),
            Region::Syria => write!(f, "Syria"),
            Region::SaoTomeAndPrincipe => write!(f, "São Tomé & Príncipe"),
            Region::Taiwan => write!(f, "Taiwan"),
            Region::Tajikistan => write!(f, "Tajikistan"),
            Region::Tanzania => write!(f, "Tanzania"),
            Region::Thailand => write!(f, "Thailand"),
            Region::TimorLeste => write!(f, "Timor-Leste"),
            Region::Togo => write!(f, "Togo"),
            Region::Tokelau => write!(f, "Tokelau"),
            Region::Tonga => write!(f, "Tonga"),
            Region::TrinidadAndTobago => write!(f, "Trinidad & Tobago"),
            Region::TristanDaCunha => write!(f, "Tristan da Cunha"),
            Region::Tunisia => write!(f, "Tunisia"),
            Region::Turkey => write!(f, "Turkey"),
            Region::Turkmenistan => write!(f, "Turkmenistan"),
            Region::TurksAndCaicosIslands => write!(f, "Turks & Caicos Islands"),
            Region::Tuvalu => write!(f, "Tuvalu"),
            Region::USOutlyingIslands => write!(f, "U.S. Outlying Islands"),
            Region::USVirginIslands => write!(f, "U.S. Virgin Islands"),
            Region::Uganda => write!(f, "Uganda"),
            Region::Ukraine => write!(f, "Ukraine"),
            Region::UnitedArabEmirates => write!(f, "United Arab Emirates"),
            Region::UnitedKingdom => write!(f, "United Kingdom"),
            Region::UnitedStates => write!(f, "United States"),
            Region::Uruguay => write!(f, "Uruguay"),
            Region::Uzbekistan => write!(f, "Uzbekistan"),
            Region::Vanuatu => write!(f, "Vanuatu"),
            Region::VaticanCity => write!(f, "Vatican City"),
            Region::Venezuela => write!(f, "Venezuela"),
            Region::Vietnam => write!(f, "Vietnam"),
            Region::WallisAndFutuna => write!(f, "Wallis & Futuna"),
            Region::WesternSahara => write!(f, "Western Sahara"),
            Region::Yemen => write!(f, "Yemen"),
            Region::Zambia => write!(f, "Zambia"),
            Region::Zimbabwe => write!(f, "Zimbabwe"),
            Region::AlandIslands => write!(f, "Åland Islands"),
        } // match
    } // fn
} // impl