use std::collections::HashMap;
use std::cell::RefCell;
use std::default::Default;
use std::collections::BTreeMap;
use serde_json as json;
use std::io;
use std::fs;
use std::mem;
use std::thread::sleep;

use crate::client;

// ##############
// UTILITIES ###
// ############

/// Identifies the an OAuth2 authorization scope.
/// A scope is needed when requesting an
/// [authorization token](https://developers.google.com/youtube/v3/guides/authentication).
#[derive(PartialEq, Eq, Hash)]
pub enum Scope {
    /// See, edit, configure, and delete your Google Cloud data and see the email address for your Google Account.
    CloudPlatform,
}

impl AsRef<str> for Scope {
    fn as_ref(&self) -> &str {
        match *self {
            Scope::CloudPlatform => "https://www.googleapis.com/auth/cloud-platform",
        }
    }
}

impl Default for Scope {
    fn default() -> Scope {
        Scope::CloudPlatform
    }
}



// ########
// HUB ###
// ######

/// Central instance to access all Apigee related resource activities
///
/// # Examples
///
/// Instantiate a new hub
///
/// ```test_harness,no_run
/// extern crate hyper;
/// extern crate hyper_rustls;
/// extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1EndpointAttachment;
/// use apigee1::{Result, Error};
/// # async fn dox() {
/// use std::default::Default;
/// use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// // Get an ApplicationSecret instance by some means. It contains the `client_id` and 
/// // `client_secret`, among other things.
/// let secret: oauth2::ApplicationSecret = Default::default();
/// // Instantiate the authenticator. It will choose a suitable authentication flow for you, 
/// // unless you replace  `None` with the desired Flow.
/// // Provide your own `AuthenticatorDelegate` to adjust the way it operates and get feedback about 
/// // what's going on. You probably want to bring in your own `TokenStorage` to persist tokens and
/// // retrieve them from storage.
/// let auth = oauth2::InstalledFlowAuthenticator::builder(
///         secret,
///         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
///     ).build().await.unwrap();
/// let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1EndpointAttachment::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().endpoint_attachments_create(req, "parent")
///              .endpoint_attachment_id("At")
///              .doit().await;
/// 
/// match result {
///     Err(e) => match e {
///         // The Error enum provides details about what exactly happened.
///         // You can also just use its `Debug`, `Display` or `Error` traits
///          Error::HttpError(_)
///         |Error::Io(_)
///         |Error::MissingAPIKey
///         |Error::MissingToken(_)
///         |Error::Cancelled
///         |Error::UploadSizeLimitExceeded(_, _)
///         |Error::Failure(_)
///         |Error::BadRequest(_)
///         |Error::FieldClash(_)
///         |Error::JsonDecodeError(_, _) => println!("{}", e),
///     },
///     Ok(res) => println!("Success: {:?}", res),
/// }
/// # }
/// ```
#[derive(Clone)]
pub struct Apigee<> {
    pub client: hyper::Client<hyper_rustls::HttpsConnector<hyper::client::connect::HttpConnector>, hyper::body::Body>,
    pub auth: oauth2::authenticator::Authenticator<hyper_rustls::HttpsConnector<hyper::client::connect::HttpConnector>>,
    _user_agent: String,
    _base_url: String,
    _root_url: String,
}

impl<'a, > client::Hub for Apigee<> {}

impl<'a, > Apigee<> {

    pub fn new(client: hyper::Client<hyper_rustls::HttpsConnector<hyper::client::connect::HttpConnector>, hyper::body::Body>, authenticator: oauth2::authenticator::Authenticator<hyper_rustls::HttpsConnector<hyper::client::connect::HttpConnector>>) -> Apigee<> {
        Apigee {
            client,
            auth: authenticator,
            _user_agent: "google-api-rust-client/3.0.0".to_string(),
            _base_url: "https://apigee.googleapis.com/".to_string(),
            _root_url: "https://apigee.googleapis.com/".to_string(),
        }
    }

    pub fn hybrid(&'a self) -> HybridMethods<'a> {
        HybridMethods { hub: &self }
    }
    pub fn organizations(&'a self) -> OrganizationMethods<'a> {
        OrganizationMethods { hub: &self }
    }
    pub fn projects(&'a self) -> ProjectMethods<'a> {
        ProjectMethods { hub: &self }
    }

    /// Set the user-agent header field to use in all requests to the server.
    /// It defaults to `google-api-rust-client/3.0.0`.
    ///
    /// Returns the previously set user-agent.
    pub fn user_agent(&mut self, agent_name: String) -> String {
        mem::replace(&mut self._user_agent, agent_name)
    }

    /// Set the base url to use in all requests to the server.
    /// It defaults to `https://apigee.googleapis.com/`.
    ///
    /// Returns the previously set base url.
    pub fn base_url(&mut self, new_base_url: String) -> String {
        mem::replace(&mut self._base_url, new_base_url)
    }

    /// Set the root url to use in all requests to the server.
    /// It defaults to `https://apigee.googleapis.com/`.
    ///
    /// Returns the previously set root url.
    pub fn root_url(&mut self, new_root_url: String) -> String {
        mem::replace(&mut self._root_url, new_root_url)
    }
}


// ############
// SCHEMAS ###
// ##########
/// Message that represents an arbitrary HTTP body. It should only be used for payload formats that can't be represented as JSON, such as raw binary or an HTML page. This message can be used both in streaming and non-streaming API methods in the request as well as the response. It can be used as a top-level request field, which is convenient if one wants to extract parameters from either the URL or HTTP template into the request fields and also want access to the raw HTTP body. Example: message GetResourceRequest { // A unique request id. string request_id = 1; // The raw HTTP body is bound to this field. google.api.HttpBody http_body = 2; } service ResourceService { rpc GetResource(GetResourceRequest) returns (google.api.HttpBody); rpc UpdateResource(google.api.HttpBody) returns (google.protobuf.Empty); } Example with streaming methods: service CaldavService { rpc GetCalendar(stream google.api.HttpBody) returns (stream google.api.HttpBody); rpc UpdateCalendar(stream google.api.HttpBody) returns (stream google.api.HttpBody); } Use of this type only changes how the request and response bodies are handled, all other features will continue to work unchanged.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis revisions get organizations](OrganizationApiRevisionGetCall) (response)
/// * [apis revisions update api proxy revision organizations](OrganizationApiRevisionUpdateApiProxyRevisionCall) (request)
/// * [apis create organizations](OrganizationApiCreateCall) (request)
/// * [environments keystores aliases create organizations](OrganizationEnvironmentKeystoreAliaseCreateCall) (request)
/// * [environments keystores aliases csr organizations](OrganizationEnvironmentKeystoreAliaseCsrCall) (response)
/// * [environments keystores aliases get certificate organizations](OrganizationEnvironmentKeystoreAliaseGetCertificateCall) (response)
/// * [environments keystores aliases update organizations](OrganizationEnvironmentKeystoreAliaseUpdateCall) (request)
/// * [environments queries get result organizations](OrganizationEnvironmentQueryGetResultCall) (response)
/// * [environments resourcefiles create organizations](OrganizationEnvironmentResourcefileCreateCall) (request)
/// * [environments resourcefiles get organizations](OrganizationEnvironmentResourcefileGetCall) (response)
/// * [environments resourcefiles update organizations](OrganizationEnvironmentResourcefileUpdateCall) (request)
/// * [host queries get result organizations](OrganizationHostQueryGetResultCall) (response)
/// * [sharedflows revisions get organizations](OrganizationSharedflowRevisionGetCall) (response)
/// * [sharedflows revisions update shared flow revision organizations](OrganizationSharedflowRevisionUpdateSharedFlowRevisionCall) (request)
/// * [sharedflows create organizations](OrganizationSharedflowCreateCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleApiHttpBody {
    /// The HTTP Content-Type header value specifying the content type of the body.
    #[serde(rename="contentType")]
    pub content_type: Option<String>,
    /// The HTTP request/response body as raw binary.
    pub data: Option<String>,
    /// Application specific response metadata. Must be set in the first response for streaming APIs.
    pub extensions: Option<Vec<HashMap<String, String>>>,
}

impl client::RequestValue for GoogleApiHttpBody {}
impl client::ResponseResult for GoogleApiHttpBody {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Access {
    /// no description provided
    #[serde(rename="Get")]
    pub get: Option<GoogleCloudApigeeV1AccessGet>,
    /// no description provided
    #[serde(rename="Remove")]
    pub remove: Option<GoogleCloudApigeeV1AccessRemove>,
    /// no description provided
    #[serde(rename="Set")]
    pub set: Option<GoogleCloudApigeeV1AccessSet>,
}

impl client::Part for GoogleCloudApigeeV1Access {}


/// Get action. For example, "Get" : { "name" : "target.name", "value" : "default" }
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AccessGet {
    /// no description provided
    pub name: Option<String>,
    /// no description provided
    pub value: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1AccessGet {}


/// Remove action. For example, "Remove" : { "name" : "target.name", "success" : true }
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AccessRemove {
    /// no description provided
    pub name: Option<String>,
    /// no description provided
    pub success: Option<bool>,
}

impl client::Part for GoogleCloudApigeeV1AccessRemove {}


/// Set action. For example, "Set" : { "name" : "target.name", "success" : true, "value" : "default" }
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AccessSet {
    /// no description provided
    pub name: Option<String>,
    /// no description provided
    pub success: Option<bool>,
    /// no description provided
    pub value: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1AccessSet {}


/// Request for ActivateNatAddressRequest. Activate the nat address request.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances nat addresses activate organizations](OrganizationInstanceNatAddresseActivateCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ActivateNatAddressRequest { _never_set: Option<bool> }

impl client::RequestValue for GoogleCloudApigeeV1ActivateNatAddressRequest {}


/// Add-on configurations for the Apigee organization.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AddonsConfig {
    /// Configuration for the Advanced API Ops add-on.
    #[serde(rename="advancedApiOpsConfig")]
    pub advanced_api_ops_config: Option<GoogleCloudApigeeV1AdvancedApiOpsConfig>,
    /// Configuration for the Connectors Platform add-on.
    #[serde(rename="connectorsPlatformConfig")]
    pub connectors_platform_config: Option<GoogleCloudApigeeV1ConnectorsPlatformConfig>,
    /// Configuration for the Integration add-on.
    #[serde(rename="integrationConfig")]
    pub integration_config: Option<GoogleCloudApigeeV1IntegrationConfig>,
    /// Configuration for the Monetization add-on.
    #[serde(rename="monetizationConfig")]
    pub monetization_config: Option<GoogleCloudApigeeV1MonetizationConfig>,
}

impl client::Part for GoogleCloudApigeeV1AddonsConfig {}


/// Request for AdjustDeveloperBalance.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers balance adjust organizations](OrganizationDeveloperBalanceAdjustCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AdjustDeveloperBalanceRequest {
    /// * A positive value of `adjustment` means that that the API provider wants to adjust the balance for an under-charged developer i.e. the balance of the developer will decrease. * A negative value of `adjustment` means that that the API provider wants to adjust the balance for an over-charged developer i.e. the balance of the developer will increase.
    pub adjustment: Option<GoogleTypeMoney>,
}

impl client::RequestValue for GoogleCloudApigeeV1AdjustDeveloperBalanceRequest {}


/// Configuration for the Advanced API Ops add-on.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AdvancedApiOpsConfig {
    /// Flag that specifies whether the Advanced API Ops add-on is enabled.
    pub enabled: Option<bool>,
}

impl client::Part for GoogleCloudApigeeV1AdvancedApiOpsConfig {}


/// Reference to a certificate or key/certificate pair.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments keystores aliases create organizations](OrganizationEnvironmentKeystoreAliaseCreateCall) (response)
/// * [environments keystores aliases delete organizations](OrganizationEnvironmentKeystoreAliaseDeleteCall) (response)
/// * [environments keystores aliases get organizations](OrganizationEnvironmentKeystoreAliaseGetCall) (response)
/// * [environments keystores aliases update organizations](OrganizationEnvironmentKeystoreAliaseUpdateCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Alias {
    /// Resource ID for this alias. Values must match the regular expression `[^/]{1,255}`.
    pub alias: Option<String>,
    /// Chain of certificates under this alias.
    #[serde(rename="certsInfo")]
    pub certs_info: Option<GoogleCloudApigeeV1Certificate>,
    /// Type of alias.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1Alias {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AliasRevisionConfig {
    /// Location of the alias file. For example, a Google Cloud Storage URI.
    pub location: Option<String>,
    /// Name of the alias revision included in the keystore in the following format: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}/revisions/{rev}`
    pub name: Option<String>,
    /// no description provided
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1AliasRevisionConfig {}


/// the Api category resource wrapped with response status, error_code etc.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sites apicategories create organizations](OrganizationSiteApicategoryCreateCall) (response)
/// * [sites apicategories get organizations](OrganizationSiteApicategoryGetCall) (response)
/// * [sites apicategories patch organizations](OrganizationSiteApicategoryPatchCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiCategory {
    /// Details of category.
    pub data: Option<GoogleCloudApigeeV1ApiCategoryData>,
    /// ID that can be used to find errors in the log files.
    #[serde(rename="errorCode")]
    pub error_code: Option<String>,
    /// Description of the operation.
    pub message: Option<String>,
    /// ID that can be used to find request details in the log files.
    #[serde(rename="requestId")]
    pub request_id: Option<String>,
    /// Status of the operation.
    pub status: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ApiCategory {}


/// the Api category resource.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sites apicategories create organizations](OrganizationSiteApicategoryCreateCall) (request)
/// * [sites apicategories patch organizations](OrganizationSiteApicategoryPatchCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiCategoryData {
    /// ID of the category (a UUID).
    pub id: Option<String>,
    /// Name of the category.
    pub name: Option<String>,
    /// Name of the portal.
    #[serde(rename="siteId")]
    pub site_id: Option<String>,
    /// Time the category was last modified in milliseconds since epoch.
    #[serde(rename="updateTime")]
    pub update_time: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1ApiCategoryData {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts create organizations](OrganizationApiproductCreateCall) (request|response)
/// * [apiproducts delete organizations](OrganizationApiproductDeleteCall) (response)
/// * [apiproducts get organizations](OrganizationApiproductGetCall) (response)
/// * [apiproducts update organizations](OrganizationApiproductUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiProduct {
    /// no description provided
    #[serde(rename="apiResources")]
    pub api_resources: Option<Vec<String>>,
    /// Flag that specifies how API keys are approved to access the APIs defined by the API product. If set to `manual`, the consumer key is generated and returned in "pending" state. In this case, the API keys won't work until they have been explicitly approved. If set to `auto`, the consumer key is generated and returned in "approved" state and can be used immediately. **Note:** Typically, `auto` is used to provide access to free or trial API products that provide limited quota or capabilities.
    #[serde(rename="approvalType")]
    pub approval_type: Option<String>,
    /// Array of attributes that may be used to extend the default API product profile with customer-specific metadata. You can specify a maximum of 18 attributes. Use this property to specify the access level of the API product as either `public`, `private`, or `internal`. Only products marked `public` are available to developers in the Apigee developer portal. For example, you can set a product to `internal` while it is in development and then change access to `public` when it is ready to release on the portal. API products marked as `private` do not appear on the portal, but can be accessed by external developers.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Response only. Creation time of this environment as milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Description of the API product. Include key information about the API product that is not captured by other fields. Comma-separated list of API resources to be bundled in the API product. By default, the resource paths are mapped from the `proxy.pathsuffix` variable. The proxy path suffix is defined as the URI fragment following the ProxyEndpoint base path. For example, if the `apiResources` element is defined to be `/forecastrss` and the base path defined for the API proxy is `/weather`, then only requests to `/weather/forecastrss` are permitted by the API product. You can select a specific path, or you can select all subpaths with the following wildcard: - `/**`: Indicates that all sub-URIs are included. - `/*` : Indicates that only URIs one level down are included. By default, / supports the same resources as /** as well as the base path defined by the API proxy. For example, if the base path of the API proxy is `/v1/weatherapikey`, then the API product supports requests to `/v1/weatherapikey` and to any sub-URIs, such as `/v1/weatherapikey/forecastrss`, `/v1/weatherapikey/region/CA`, and so on. For more information, see Managing API products.
    pub description: Option<String>,
    /// Name displayed in the UI or developer portal to developers registering for API access.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Comma-separated list of environment names to which the API product is bound. Requests to environments that are not listed are rejected. By specifying one or more environments, you can bind the resources listed in the API product to a specific environment, preventing developers from accessing those resources through API proxies deployed in another environment. This setting is used, for example, to prevent resources associated with API proxies in `prod` from being accessed by API proxies deployed in `test`.
    pub environments: Option<Vec<String>>,
    /// Configuration used to group Apigee proxies or remote services with graphQL operation name, graphQL operation type and quotas. This grouping allows us to precisely set quota for a particular combination of graphQL name and operation type for a particular proxy request. If graphQL name is not set, this would imply quota will be applied on all graphQL requests matching the operation type.
    #[serde(rename="graphqlOperationGroup")]
    pub graphql_operation_group: Option<GoogleCloudApigeeV1GraphQLOperationGroup>,
    /// Response only. Modified time of this environment as milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Internal name of the API product. Characters you can use in the name are restricted to: `A-Z0-9._\-$ %`. **Note:** The internal name cannot be edited when updating the API product.
    pub name: Option<String>,
    /// Configuration used to group Apigee proxies or remote services with resources, method types, and quotas. The resource refers to the resource URI (excluding the base path). With this grouping, the API product creator is able to fine-tune and give precise control over which REST methods have access to specific resources and how many calls can be made (using the `quota` setting). **Note:** The `api_resources` setting cannot be specified for both the API product and operation group; otherwise the call will fail.
    #[serde(rename="operationGroup")]
    pub operation_group: Option<GoogleCloudApigeeV1OperationGroup>,
    /// Comma-separated list of API proxy names to which this API product is bound. By specifying API proxies, you can associate resources in the API product with specific API proxies, preventing developers from accessing those resources through other API proxies. Apigee rejects requests to API proxies that are not listed. **Note:** The API proxy names must already exist in the specified environment as they will be validated upon creation.
    pub proxies: Option<Vec<String>>,
    /// Number of request messages permitted per app by this API product for the specified `quotaInterval` and `quotaTimeUnit`. For example, a `quota` of 50, for a `quotaInterval` of 12 and a `quotaTimeUnit` of hours means 50 requests are allowed every 12 hours.
    pub quota: Option<String>,
    /// Time interval over which the number of request messages is calculated.
    #[serde(rename="quotaInterval")]
    pub quota_interval: Option<String>,
    /// Time unit defined for the `quotaInterval`. Valid values include `minute`, `hour`, `day`, or `month`.
    #[serde(rename="quotaTimeUnit")]
    pub quota_time_unit: Option<String>,
    /// Comma-separated list of OAuth scopes that are validated at runtime. Apigee validates that the scopes in any access token presented match the scopes defined in the OAuth policy associated with the API product.
    pub scopes: Option<Vec<String>>,
}

impl client::RequestValue for GoogleCloudApigeeV1ApiProduct {}
impl client::ResponseResult for GoogleCloudApigeeV1ApiProduct {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiProductRef {
    /// Name of the API product.
    pub apiproduct: Option<String>,
    /// Status of the API product. Valid values are `approved` or `revoked`.
    pub status: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ApiProductRef {}


/// Metadata describing the API proxy
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis delete organizations](OrganizationApiDeleteCall) (response)
/// * [apis get organizations](OrganizationApiGetCall) (response)
/// * [apis patch organizations](OrganizationApiPatchCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiProxy {
    /// Output only. The type of the API proxy.
    #[serde(rename="apiProxyType")]
    pub api_proxy_type: Option<String>,
    /// User labels applied to this API Proxy.
    pub labels: Option<HashMap<String, String>>,
    /// Output only. The id of the most recently created revision for this api proxy.
    #[serde(rename="latestRevisionId")]
    pub latest_revision_id: Option<String>,
    /// Output only. Metadata describing the API proxy.
    #[serde(rename="metaData")]
    pub meta_data: Option<GoogleCloudApigeeV1EntityMetadata>,
    /// Output only. Name of the API proxy.
    pub name: Option<String>,
    /// Output only. Whether this proxy is read-only. A read-only proxy cannot have new revisions created through calls to CreateApiProxyRevision. A proxy is read-only if it was generated by an archive.
    #[serde(rename="readOnly")]
    pub read_only: Option<bool>,
    /// Output only. List of revisons defined for the API proxy.
    pub revision: Option<Vec<String>>,
}

impl client::RequestValue for GoogleCloudApigeeV1ApiProxy {}
impl client::ResponseResult for GoogleCloudApigeeV1ApiProxy {}


/// API proxy revision.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis revisions delete organizations](OrganizationApiRevisionDeleteCall) (response)
/// * [apis revisions update api proxy revision organizations](OrganizationApiRevisionUpdateApiProxyRevisionCall) (response)
/// * [apis create organizations](OrganizationApiCreateCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiProxyRevision {
    /// Output only. The archive that generated this proxy revision. This field is only present on proxy revisions that were generated by an archive. Proxies generated by archives cannot be updated, deleted, or deployed to other environments. Format: `organizations/*/environments/*/archiveDeployments/*`
    pub archive: Option<String>,
    /// Base URL of the API proxy.
    pub basepaths: Option<Vec<String>>,
    /// Version of the API proxy configuration schema to which the API proxy conforms. Currently, the only supported value is 4.0 (`majorVersion.minorVersion`). This setting may be used in the future to track the evolution of the API proxy format.
    #[serde(rename="configurationVersion")]
    pub configuration_version: Option<GoogleCloudApigeeV1ConfigVersion>,
    /// Revision number, app name, and organization for the API proxy.
    #[serde(rename="contextInfo")]
    pub context_info: Option<String>,
    /// Time that the API proxy revision was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Description of the API proxy revision.
    pub description: Option<String>,
    /// Human-readable name of the API proxy.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Metadata describing the API proxy revision as a key-value map.
    #[serde(rename="entityMetaDataAsProperties")]
    pub entity_meta_data_as_properties: Option<HashMap<String, String>>,
    /// List of IntegrationEndpoints in the '/integration-endpoints' directory of the API proxy. This is a 'manifest' setting designed to provide visibility into the contents of the API proxy.
    #[serde(rename="integrationEndpoints")]
    pub integration_endpoints: Option<Vec<String>>,
    /// Time that the API proxy revision was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Name of the API proxy.
    pub name: Option<String>,
    /// List of policy names included in the API proxy revision..
    pub policies: Option<Vec<String>>,
    /// List of proxy names included in the API proxy revision.
    pub proxies: Option<Vec<String>>,
    /// List of ProxyEndpoints in the `/proxies` directory of the API proxy. Typically, this element is included only when the API proxy was created using the Edge UI. This is a 'manifest' setting designed to provide visibility into the contents of the API proxy.
    #[serde(rename="proxyEndpoints")]
    pub proxy_endpoints: Option<Vec<String>>,
    /// List of resource files included in the API proxy revision.
    #[serde(rename="resourceFiles")]
    pub resource_files: Option<GoogleCloudApigeeV1ResourceFiles>,
    /// List of the resources included in the API proxy revision formatted as "{type}://{name}".
    pub resources: Option<Vec<String>>,
    /// API proxy revision.
    pub revision: Option<String>,
    /// List of the shared flows included in the API proxy revision.
    #[serde(rename="sharedFlows")]
    pub shared_flows: Option<Vec<String>>,
    /// OpenAPI Specification that is associated with the API proxy. The value is set to a URL or to a path in the specification store.
    pub spec: Option<String>,
    /// List of TargetEndpoints in the `/targets` directory of the API proxy. Typically, this element is included only when the API proxy was created using the Edge UI. This is a 'manifest' setting designed to provide visibility into the contents of the API proxy.
    #[serde(rename="targetEndpoints")]
    pub target_endpoints: Option<Vec<String>>,
    /// List of TargetServers referenced in any TargetEndpoint in the API proxy. Typically, you will see this element only when the API proxy was created using the Edge UI. This is a 'manifest' setting designed to provide visibility into the contents of the API proxy.
    #[serde(rename="targetServers")]
    pub target_servers: Option<Vec<String>>,
    /// List of the targets included in the API proxy revision.
    pub targets: Option<Vec<String>>,
    /// List of the teams included in the API proxy revision.
    pub teams: Option<Vec<String>>,
    /// Type. Set to `Application`. Maintained for compatibility with the Apigee Edge API.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ApiProxyRevision {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sites apicategories delete organizations](OrganizationSiteApicategoryDeleteCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ApiResponseWrapper {
    /// ID that can be used to find errors in the log files.
    #[serde(rename="errorCode")]
    pub error_code: Option<String>,
    /// Description of the operation.
    pub message: Option<String>,
    /// ID that can be used to find request details in the log files.
    #[serde(rename="requestId")]
    pub request_id: Option<String>,
    /// Status of the operation.
    pub status: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ApiResponseWrapper {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apps get organizations](OrganizationAppGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1App {
    /// List of API products associated with the app.
    #[serde(rename="apiProducts")]
    pub api_products: Option<Vec<GoogleCloudApigeeV1ApiProductRef>>,
    /// ID of the app.
    #[serde(rename="appId")]
    pub app_id: Option<String>,
    /// List of attributes.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Callback URL used by OAuth 2.0 authorization servers to communicate authorization codes back to apps.
    #[serde(rename="callbackUrl")]
    pub callback_url: Option<String>,
    /// Name of the company that owns the app.
    #[serde(rename="companyName")]
    pub company_name: Option<String>,
    /// Output only. Unix time when the app was created.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Output only. Set of credentials for the app. Credentials are API key/secret pairs associated with API products.
    pub credentials: Option<Vec<GoogleCloudApigeeV1Credential>>,
    /// ID of the developer.
    #[serde(rename="developerId")]
    pub developer_id: Option<String>,
    /// Duration, in milliseconds, of the consumer key that will be generated for the app. The default value, -1, indicates an infinite validity period. Once set, the expiration can't be updated. json key: keyExpiresIn
    #[serde(rename="keyExpiresIn")]
    pub key_expires_in: Option<String>,
    /// Output only. Last modified time as milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Name of the app.
    pub name: Option<String>,
    /// Scopes to apply to the app. The specified scope names must already exist on the API product that you associate with the app.
    pub scopes: Option<Vec<String>>,
    /// Status of the credential.
    pub status: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1App {}


/// Archive Deployment information.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments create organizations](OrganizationEnvironmentArchiveDeploymentCreateCall) (request)
/// * [environments archive deployments get organizations](OrganizationEnvironmentArchiveDeploymentGetCall) (response)
/// * [environments archive deployments patch organizations](OrganizationEnvironmentArchiveDeploymentPatchCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ArchiveDeployment {
    /// Output only. The time at which the Archive Deployment was created in milliseconds since the epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Input only. The Google Cloud Storage signed URL returned from GenerateUploadUrl and used to upload the Archive zip file.
    #[serde(rename="gcsUri")]
    pub gcs_uri: Option<String>,
    /// User-supplied key-value pairs used to organize ArchiveDeployments. Label keys must be between 1 and 63 characters long, have a UTF-8 encoding of maximum 128 bytes, and must conform to the following PCRE regular expression: \p{Ll}\p{Lo}{0,62} Label values must be between 1 and 63 characters long, have a UTF-8 encoding of maximum 128 bytes, and must conform to the following PCRE regular expression: [\p{Ll}\p{Lo}\p{N}_-]{0,63} No more than 64 labels can be associated with a given store.
    pub labels: Option<HashMap<String, String>>,
    /// Name of the Archive Deployment in the following format: `organizations/{org}/environments/{env}/archiveDeployments/{id}`.
    pub name: Option<String>,
    /// Output only. A reference to the LRO that created this Archive Deployment in the following format: `organizations/{org}/operations/{id}`
    pub operation: Option<String>,
    /// Output only. The time at which the Archive Deployment was updated in milliseconds since the epoch.
    #[serde(rename="updatedAt")]
    pub updated_at: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1ArchiveDeployment {}
impl client::ResponseResult for GoogleCloudApigeeV1ArchiveDeployment {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments queries create organizations](OrganizationEnvironmentQueryCreateCall) (response)
/// * [environments queries get organizations](OrganizationEnvironmentQueryGetCall) (response)
/// * [host queries create organizations](OrganizationHostQueryCreateCall) (response)
/// * [host queries get organizations](OrganizationHostQueryGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AsyncQuery {
    /// Creation time of the query.
    pub created: Option<String>,
    /// Hostname is available only when query is executed at host level.
    #[serde(rename="envgroupHostname")]
    pub envgroup_hostname: Option<String>,
    /// Error is set when query fails.
    pub error: Option<String>,
    /// ExecutionTime is available only after the query is completed.
    #[serde(rename="executionTime")]
    pub execution_time: Option<String>,
    /// Asynchronous Query Name.
    pub name: Option<String>,
    /// Contains information like metrics, dimenstions etc of the AsyncQuery.
    #[serde(rename="queryParams")]
    pub query_params: Option<GoogleCloudApigeeV1QueryMetadata>,
    /// Asynchronous Report ID.
    #[serde(rename="reportDefinitionId")]
    pub report_definition_id: Option<String>,
    /// Result is available only after the query is completed.
    pub result: Option<GoogleCloudApigeeV1AsyncQueryResult>,
    /// ResultFileSize is available only after the query is completed.
    #[serde(rename="resultFileSize")]
    pub result_file_size: Option<String>,
    /// ResultRows is available only after the query is completed.
    #[serde(rename="resultRows")]
    pub result_rows: Option<String>,
    /// Self link of the query. Example: `/organizations/myorg/environments/myenv/queries/9cfc0d85-0f30-46d6-ae6f-318d0cb961bd` or following format if query is running at host level: `/organizations/myorg/hostQueries/9cfc0d85-0f30-46d6-ae6f-318d0cb961bd`
    #[serde(rename="self")]
    pub self_: Option<String>,
    /// Query state could be "enqueued", "running", "completed", "failed".
    pub state: Option<String>,
    /// Last updated timestamp for the query.
    pub updated: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1AsyncQuery {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AsyncQueryResult {
    /// Query result will be unaccessable after this time.
    pub expires: Option<String>,
    /// Self link of the query results. Example: `/organizations/myorg/environments/myenv/queries/9cfc0d85-0f30-46d6-ae6f-318d0cb961bd/result` or following format if query is running at host level: `/organizations/myorg/hostQueries/9cfc0d85-0f30-46d6-ae6f-318d0cb961bd/result`
    #[serde(rename="self")]
    pub self_: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1AsyncQueryResult {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [host queries get result view organizations](OrganizationHostQueryGetResultViewCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1AsyncQueryResultView {
    /// Error code when there is a failure.
    pub code: Option<i32>,
    /// Error message when there is a failure.
    pub error: Option<String>,
    /// Metadata contains information like metrics, dimenstions etc of the AsyncQuery.
    pub metadata: Option<GoogleCloudApigeeV1QueryMetadata>,
    /// Rows of query result. Each row is a JSON object. Example: {sum(message_count): 1, developer_app: "(not set)",…}
    pub rows: Option<Vec<String>>,
    /// State of retrieving ResultView.
    pub state: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1AsyncQueryResultView {}


/// Key-value pair to store extra metadata.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts attributes delete organizations](OrganizationApiproductAttributeDeleteCall) (response)
/// * [apiproducts attributes get organizations](OrganizationApiproductAttributeGetCall) (response)
/// * [apiproducts attributes update api product attribute organizations](OrganizationApiproductAttributeUpdateApiProductAttributeCall) (request|response)
/// * [developers apps attributes delete organizations](OrganizationDeveloperAppAttributeDeleteCall) (response)
/// * [developers apps attributes get organizations](OrganizationDeveloperAppAttributeGetCall) (response)
/// * [developers apps attributes update developer app attribute organizations](OrganizationDeveloperAppAttributeUpdateDeveloperAppAttributeCall) (request|response)
/// * [developers attributes delete organizations](OrganizationDeveloperAttributeDeleteCall) (response)
/// * [developers attributes get organizations](OrganizationDeveloperAttributeGetCall) (response)
/// * [developers attributes update developer attribute organizations](OrganizationDeveloperAttributeUpdateDeveloperAttributeCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Attribute {
    /// API key of the attribute.
    pub name: Option<String>,
    /// Value of the attribute.
    pub value: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Attribute {}
impl client::ResponseResult for GoogleCloudApigeeV1Attribute {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts attributes list organizations](OrganizationApiproductAttributeListCall) (response)
/// * [apiproducts attributes organizations](OrganizationApiproductAttributeCall) (request|response)
/// * [developers apps attributes list organizations](OrganizationDeveloperAppAttributeListCall) (response)
/// * [developers apps attributes organizations](OrganizationDeveloperAppAttributeCall) (request|response)
/// * [developers attributes list organizations](OrganizationDeveloperAttributeListCall) (response)
/// * [developers attributes organizations](OrganizationDeveloperAttributeCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Attributes {
    /// List of attributes.
    pub attribute: Option<Vec<GoogleCloudApigeeV1Attribute>>,
}

impl client::RequestValue for GoogleCloudApigeeV1Attributes {}
impl client::ResponseResult for GoogleCloudApigeeV1Attributes {}


/// CanaryEvaluation represents the canary analysis between two versions of the runtime that is serving requests.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances canaryevaluations create organizations](OrganizationInstanceCanaryevaluationCreateCall) (request)
/// * [instances canaryevaluations get organizations](OrganizationInstanceCanaryevaluationGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CanaryEvaluation {
    /// Required. The stable version that is serving requests.
    pub control: Option<String>,
    /// Output only. Create time of the canary evaluation.
    #[serde(rename="createTime")]
    pub create_time: Option<String>,
    /// Required. End time for the evaluation's analysis.
    #[serde(rename="endTime")]
    pub end_time: Option<String>,
    /// Required. Labels used to filter the metrics used for a canary evaluation.
    #[serde(rename="metricLabels")]
    pub metric_labels: Option<GoogleCloudApigeeV1CanaryEvaluationMetricLabels>,
    /// Output only. Name of the canary evalution.
    pub name: Option<String>,
    /// Required. Start time for the canary evaluation's analysis.
    #[serde(rename="startTime")]
    pub start_time: Option<String>,
    /// Output only. The current state of the canary evaluation.
    pub state: Option<String>,
    /// Required. The newer version that is serving requests.
    pub treatment: Option<String>,
    /// Output only. The resulting verdict of the canary evaluations: NONE, PASS, or FAIL.
    pub verdict: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1CanaryEvaluation {}
impl client::ResponseResult for GoogleCloudApigeeV1CanaryEvaluation {}


/// Labels that can be used to filter Apigee metrics.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CanaryEvaluationMetricLabels {
    /// The environment ID associated with the metrics.
    pub env: Option<String>,
    /// Required. The instance ID associated with the metrics. In Apigee Hybrid, the value is configured during installation.
    pub instance_id: Option<String>,
    /// Required. The location associated with the metrics.
    pub location: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1CanaryEvaluationMetricLabels {}


/// X.509 certificate as defined in RFC 5280.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CertInfo {
    /// X.509 basic constraints extension.
    #[serde(rename="basicConstraints")]
    pub basic_constraints: Option<String>,
    /// X.509 `notAfter` validity period in milliseconds since epoch.
    #[serde(rename="expiryDate")]
    pub expiry_date: Option<String>,
    /// Flag that specifies whether the certificate is valid. Flag is set to `Yes` if the certificate is valid, `No` if expired, or `Not yet` if not yet valid.
    #[serde(rename="isValid")]
    pub is_valid: Option<String>,
    /// X.509 issuer.
    pub issuer: Option<String>,
    /// Public key component of the X.509 subject public key info.
    #[serde(rename="publicKey")]
    pub public_key: Option<String>,
    /// X.509 serial number.
    #[serde(rename="serialNumber")]
    pub serial_number: Option<String>,
    /// X.509 signatureAlgorithm.
    #[serde(rename="sigAlgName")]
    pub sig_alg_name: Option<String>,
    /// X.509 subject.
    pub subject: Option<String>,
    /// X.509 subject alternative names (SANs) extension.
    #[serde(rename="subjectAlternativeNames")]
    pub subject_alternative_names: Option<Vec<String>>,
    /// X.509 `notBefore` validity period in milliseconds since epoch.
    #[serde(rename="validFrom")]
    pub valid_from: Option<String>,
    /// X.509 version.
    pub version: Option<i32>,
}

impl client::Part for GoogleCloudApigeeV1CertInfo {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Certificate {
    /// Chain of certificates under this name.
    #[serde(rename="certInfo")]
    pub cert_info: Option<Vec<GoogleCloudApigeeV1CertInfo>>,
}

impl client::Part for GoogleCloudApigeeV1Certificate {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CommonNameConfig {
    /// no description provided
    #[serde(rename="matchWildCards")]
    pub match_wild_cards: Option<bool>,
    /// no description provided
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1CommonNameConfig {}


/// Version of the API proxy configuration schema. Currently, only 4.0 is supported.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ConfigVersion {
    /// Major version of the API proxy configuration schema.
    #[serde(rename="majorVersion")]
    pub major_version: Option<i32>,
    /// Minor version of the API proxy configuration schema.
    #[serde(rename="minorVersion")]
    pub minor_version: Option<i32>,
}

impl client::Part for GoogleCloudApigeeV1ConfigVersion {}


/// Configuration for the Connectors Platform add-on.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ConnectorsPlatformConfig {
    /// Flag that specifies whether the Connectors Platform add-on is enabled.
    pub enabled: Option<bool>,
    /// Output only. Time at which the Connectors Platform add-on expires in milliseconds since epoch. If unspecified, the add-on will never expire.
    #[serde(rename="expiresAt")]
    pub expires_at: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ConnectorsPlatformConfig {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Credential {
    /// List of API products this credential can be used for.
    #[serde(rename="apiProducts")]
    pub api_products: Option<Vec<GoogleCloudApigeeV1ApiProductRef>>,
    /// List of attributes associated with this credential.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Consumer key.
    #[serde(rename="consumerKey")]
    pub consumer_key: Option<String>,
    /// Secret key.
    #[serde(rename="consumerSecret")]
    pub consumer_secret: Option<String>,
    /// Time the credential will expire in milliseconds since epoch.
    #[serde(rename="expiresAt")]
    pub expires_at: Option<String>,
    /// Time the credential was issued in milliseconds since epoch.
    #[serde(rename="issuedAt")]
    pub issued_at: Option<String>,
    /// List of scopes to apply to the app. Specified scopes must already exist on the API product that you associate with the app.
    pub scopes: Option<Vec<String>>,
    /// Status of the credential. Valid values include `approved` or `revoked`.
    pub status: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Credential {}


/// Request for CreditDeveloperBalance.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers balance credit organizations](OrganizationDeveloperBalanceCreditCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CreditDeveloperBalanceRequest {
    /// The amount of money to be credited. The wallet corresponding to the currency specified within `transaction_amount` will be updated. For example, if you specified `currency_code` within `transaction_amount` as "USD", then the amount would be added to the wallet which has the "USD" currency or if no such wallet exists, a new wallet will be created with the "USD" currency.
    #[serde(rename="transactionAmount")]
    pub transaction_amount: Option<GoogleTypeMoney>,
    /// Each transaction_id uniquely identifies a credit balance request. If multiple requests are received with the same transaction_id, only one of them will be considered.
    #[serde(rename="transactionId")]
    pub transaction_id: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1CreditDeveloperBalanceRequest {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [reports create organizations](OrganizationReportCreateCall) (request|response)
/// * [reports get organizations](OrganizationReportGetCall) (response)
/// * [reports update organizations](OrganizationReportUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CustomReport {
    /// This field contains the chart type for the report
    #[serde(rename="chartType")]
    pub chart_type: Option<String>,
    /// Legacy field: not used. This field contains a list of comments associated with custom report
    pub comments: Option<Vec<String>>,
    /// Output only. Unix time when the app was created json key: createdAt
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// This contains the list of dimensions for the report
    pub dimensions: Option<Vec<String>>,
    /// This is the display name for the report
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Output only. Environment name
    pub environment: Option<String>,
    /// This field contains the filter expression
    pub filter: Option<String>,
    /// Legacy field: not used. Contains the from time for the report
    #[serde(rename="fromTime")]
    pub from_time: Option<String>,
    /// Output only. Modified time of this entity as milliseconds since epoch. json key: lastModifiedAt
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Output only. Last viewed time of this entity as milliseconds since epoch
    #[serde(rename="lastViewedAt")]
    pub last_viewed_at: Option<String>,
    /// Legacy field: not used This field contains the limit for the result retrieved
    pub limit: Option<String>,
    /// Required. This contains the list of metrics
    pub metrics: Option<Vec<GoogleCloudApigeeV1CustomReportMetric>>,
    /// Required. Unique identifier for the report T his is a legacy field used to encode custom report unique id
    pub name: Option<String>,
    /// Legacy field: not used. This field contains the offset for the data
    pub offset: Option<String>,
    /// Output only. Organization name
    pub organization: Option<String>,
    /// This field contains report properties such as ui metadata etc.
    pub properties: Option<Vec<GoogleCloudApigeeV1ReportProperty>>,
    /// Legacy field: not used much. Contains the list of sort by columns
    #[serde(rename="sortByCols")]
    pub sort_by_cols: Option<Vec<String>>,
    /// Legacy field: not used much. Contains the sort order for the sort columns
    #[serde(rename="sortOrder")]
    pub sort_order: Option<String>,
    /// Legacy field: not used. This field contains a list of tags associated with custom report
    pub tags: Option<Vec<String>>,
    /// This field contains the time unit of aggregation for the report
    #[serde(rename="timeUnit")]
    pub time_unit: Option<String>,
    /// Legacy field: not used. Contains the end time for the report
    #[serde(rename="toTime")]
    pub to_time: Option<String>,
    /// Legacy field: not used. This field contains the top k parameter value for restricting the result
    pub topk: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1CustomReport {}
impl client::ResponseResult for GoogleCloudApigeeV1CustomReport {}


/// This encapsulates a metric property of the form sum(message_count) where name is message_count and function is sum
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1CustomReportMetric {
    /// aggregate function
    pub function: Option<String>,
    /// name of the metric
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1CustomReportMetric {}


/// Data collector configuration.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [datacollectors create organizations](OrganizationDatacollectorCreateCall) (request|response)
/// * [datacollectors get organizations](OrganizationDatacollectorGetCall) (response)
/// * [datacollectors patch organizations](OrganizationDatacollectorPatchCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DataCollector {
    /// Output only. The time at which the data collector was created in milliseconds since the epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// A description of the data collector.
    pub description: Option<String>,
    /// Output only. The time at which the Data Collector was last updated in milliseconds since the epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// ID of the data collector. Must begin with `dc_`.
    pub name: Option<String>,
    /// Immutable. The type of data this data collector will collect.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1DataCollector {}
impl client::ResponseResult for GoogleCloudApigeeV1DataCollector {}


/// Data collector and its configuration.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DataCollectorConfig {
    /// Name of the data collector in the following format: `organizations/{org}/datacollectors/{datacollector}`
    pub name: Option<String>,
    /// Data type accepted by the data collector.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DataCollectorConfig {}


/// The data store defines the connection to export data repository (Cloud Storage, BigQuery), including the credentials used to access the data repository.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [analytics datastores create organizations](OrganizationAnalyticDatastoreCreateCall) (request|response)
/// * [analytics datastores get organizations](OrganizationAnalyticDatastoreGetCall) (response)
/// * [analytics datastores test organizations](OrganizationAnalyticDatastoreTestCall) (request)
/// * [analytics datastores update organizations](OrganizationAnalyticDatastoreUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Datastore {
    /// Output only. Datastore create time, in milliseconds since the epoch of 1970-01-01T00:00:00Z
    #[serde(rename="createTime")]
    pub create_time: Option<String>,
    /// Datastore Configurations.
    #[serde(rename="datastoreConfig")]
    pub datastore_config: Option<GoogleCloudApigeeV1DatastoreConfig>,
    /// Required. Display name in UI
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Output only. Datastore last update time, in milliseconds since the epoch of 1970-01-01T00:00:00Z
    #[serde(rename="lastUpdateTime")]
    pub last_update_time: Option<String>,
    /// Output only. Organization that the datastore belongs to
    pub org: Option<String>,
    /// Output only. Resource link of Datastore. Example: `/organizations/{org}/analytics/datastores/{uuid}`
    #[serde(rename="self")]
    pub self_: Option<String>,
    /// Destination storage type. Supported types `gcs` or `bigquery`.
    #[serde(rename="targetType")]
    pub target_type: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Datastore {}
impl client::ResponseResult for GoogleCloudApigeeV1Datastore {}


/// Configuration detail for datastore
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DatastoreConfig {
    /// Name of the Cloud Storage bucket. Required for `gcs` target_type.
    #[serde(rename="bucketName")]
    pub bucket_name: Option<String>,
    /// BigQuery dataset name Required for `bigquery` target_type.
    #[serde(rename="datasetName")]
    pub dataset_name: Option<String>,
    /// Path of Cloud Storage bucket Required for `gcs` target_type.
    pub path: Option<String>,
    /// Required. GCP project in which the datastore exists
    #[serde(rename="projectId")]
    pub project_id: Option<String>,
    /// Prefix of BigQuery table Required for `bigquery` target_type.
    #[serde(rename="tablePrefix")]
    pub table_prefix: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DatastoreConfig {}


/// Date range of the data to export.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DateRange {
    /// Required. End date (exclusive) of the data to export in the format `yyyy-mm-dd`. The date range ends at 00:00:00 UTC on the end date- which will not be in the output.
    pub end: Option<String>,
    /// Required. Start date of the data to export in the format `yyyy-mm-dd`. The date range begins at 00:00:00 UTC on the start date.
    pub start: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DateRange {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments get debugmask organizations](OrganizationEnvironmentGetDebugmaskCall) (response)
/// * [environments update debugmask organizations](OrganizationEnvironmentUpdateDebugmaskCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DebugMask {
    /// List of JSON paths that specify the JSON elements to be filtered from JSON payloads in error flows.
    #[serde(rename="faultJSONPaths")]
    pub fault_json_paths: Option<Vec<String>>,
    /// List of XPaths that specify the XML elements to be filtered from XML payloads in error flows.
    #[serde(rename="faultXPaths")]
    pub fault_x_paths: Option<Vec<String>>,
    /// Name of the debug mask.
    pub name: Option<String>,
    /// Map of namespaces to URIs.
    pub namespaces: Option<HashMap<String, String>>,
    /// List of JSON paths that specify the JSON elements to be filtered from JSON request message payloads.
    #[serde(rename="requestJSONPaths")]
    pub request_json_paths: Option<Vec<String>>,
    /// List of XPaths that specify the XML elements to be filtered from XML request message payloads.
    #[serde(rename="requestXPaths")]
    pub request_x_paths: Option<Vec<String>>,
    /// List of JSON paths that specify the JSON elements to be filtered from JSON response message payloads.
    #[serde(rename="responseJSONPaths")]
    pub response_json_paths: Option<Vec<String>>,
    /// List of XPaths that specify the XML elements to be filtered from XML response message payloads.
    #[serde(rename="responseXPaths")]
    pub response_x_paths: Option<Vec<String>>,
    /// List of variables that should be masked from the debug output.
    pub variables: Option<Vec<String>>,
}

impl client::RequestValue for GoogleCloudApigeeV1DebugMask {}
impl client::ResponseResult for GoogleCloudApigeeV1DebugMask {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments apis revisions debugsessions create organizations](OrganizationEnvironmentApiRevisionDebugsessionCreateCall) (request|response)
/// * [environments apis revisions debugsessions get organizations](OrganizationEnvironmentApiRevisionDebugsessionGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DebugSession {
    /// Optional. The number of request to be traced. Min = 1, Max = 15, Default = 10.
    pub count: Option<i32>,
    /// Output only. The first transaction creation timestamp, recorded by UAP.
    #[serde(rename="createTime")]
    pub create_time: Option<String>,
    /// Optional. A conditional statement which is evaluated against the request message to determine if it should be traced. Syntax matches that of on API Proxy bundle flow Condition.
    pub filter: Option<String>,
    /// A unique ID for this DebugSession.
    pub name: Option<String>,
    /// Optional. The time in seconds after which this DebugSession should end. This value will override the value in query param, if both are provided.
    pub timeout: Option<String>,
    /// Optional. The maximum number of bytes captured from the response payload. Min = 0, Max = 5120, Default = 5120.
    pub tracesize: Option<i32>,
    /// Optional. The length of time, in seconds, that this debug session is valid, starting from when it's received in the control plane. Min = 1, Max = 15, Default = 10.
    pub validity: Option<i32>,
}

impl client::RequestValue for GoogleCloudApigeeV1DebugSession {}
impl client::ResponseResult for GoogleCloudApigeeV1DebugSession {}


/// A transaction contains all of the debug information of the entire message flow of an API call processed by the runtime plane. The information is collected and recorded at critical points of the message flow in the runtime apiproxy.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments apis revisions debugsessions data get organizations](OrganizationEnvironmentApiRevisionDebugsessionDataGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DebugSessionTransaction {
    /// Flag indicating whether a transaction is completed or not
    pub completed: Option<bool>,
    /// List of debug data collected by runtime plane at various defined points in the flow.
    pub point: Option<Vec<GoogleCloudApigeeV1Point>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1DebugSessionTransaction {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [reports delete organizations](OrganizationReportDeleteCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeleteCustomReportResponse {
    /// The response contains only a message field.
    pub message: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1DeleteCustomReportResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments apis revisions deploy organizations](OrganizationEnvironmentApiRevisionDeployCall) (response)
/// * [environments apis revisions get deployments organizations](OrganizationEnvironmentApiRevisionGetDeploymentCall) (response)
/// * [environments sharedflows revisions deploy organizations](OrganizationEnvironmentSharedflowRevisionDeployCall) (response)
/// * [environments sharedflows revisions get deployments organizations](OrganizationEnvironmentSharedflowRevisionGetDeploymentCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Deployment {
    /// API proxy.
    #[serde(rename="apiProxy")]
    pub api_proxy: Option<String>,
    /// Time the API proxy was marked `deployed` in the control plane in millisconds since epoch.
    #[serde(rename="deployStartTime")]
    pub deploy_start_time: Option<String>,
    /// Environment.
    pub environment: Option<String>,
    /// Errors reported for this deployment. Populated only when state == ERROR. This field is not populated in List APIs.
    pub errors: Option<Vec<GoogleRpcStatus>>,
    /// Status reported by each runtime instance. This field is not populated in List APIs.
    pub instances: Option<Vec<GoogleCloudApigeeV1InstanceDeploymentStatus>>,
    /// Status reported by runtime pods. This field is not populated for List APIs. **Note**: **This field is deprecated**. Runtime versions 1.3 and above report instance level status rather than pod status.
    pub pods: Option<Vec<GoogleCloudApigeeV1PodStatus>>,
    /// API proxy revision.
    pub revision: Option<String>,
    /// Conflicts in the desired state routing configuration. The presence of conflicts does not cause the state to be `ERROR`, but it will mean that some of the deployment's base paths are not routed to its environment. If the conflicts change, the state will transition to `PROGRESSING` until the latest configuration is rolled out to all instances. This field is not populated in List APIs.
    #[serde(rename="routeConflicts")]
    pub route_conflicts: Option<Vec<GoogleCloudApigeeV1DeploymentChangeReportRoutingConflict>>,
    /// The full resource name of Cloud IAM Service Account that this deployment is using, eg, `projects/-/serviceAccounts/{email}`.
    #[serde(rename="serviceAccount")]
    pub service_account: Option<String>,
    /// Current state of the deployment. This field is not populated in List APIs.
    pub state: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1Deployment {}


/// Response for GenerateDeployChangeReport and GenerateUndeployChangeReport. This report contains any validation failures that would cause the deployment to be rejected, as well changes and conflicts in routing that may occur due to the new deployment. The existence of a routing warning does not necessarily imply that the deployment request is bad, if the desired state of the deployment request is to effect a routing change. The primary purposes of the routing messages are: 1) To inform users of routing changes that may have an effect on traffic currently being routed to other existing deployments. 2) To warn users if some base path in the proxy will not receive traffic due to an existing deployment having already claimed that base path. The presence of routing conflicts/changes will not cause non-dry-run DeployApiProxy/UndeployApiProxy requests to be rejected.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments apis revisions deployments generate deploy change report organizations](OrganizationEnvironmentApiRevisionDeploymentGenerateDeployChangeReportCall) (response)
/// * [environments apis revisions deployments generate undeploy change report organizations](OrganizationEnvironmentApiRevisionDeploymentGenerateUndeployChangeReportCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeploymentChangeReport {
    /// All routing changes that may result from a deployment request.
    #[serde(rename="routingChanges")]
    pub routing_changes: Option<Vec<GoogleCloudApigeeV1DeploymentChangeReportRoutingChange>>,
    /// All base path conflicts detected for a deployment request.
    #[serde(rename="routingConflicts")]
    pub routing_conflicts: Option<Vec<GoogleCloudApigeeV1DeploymentChangeReportRoutingConflict>>,
    /// Validation errors that would cause the deployment change request to be rejected.
    #[serde(rename="validationErrors")]
    pub validation_errors: Option<GoogleRpcPreconditionFailure>,
}

impl client::ResponseResult for GoogleCloudApigeeV1DeploymentChangeReport {}


/// Describes a potential routing change that may occur as a result of some deployment operation.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeploymentChangeReportRoutingChange {
    /// Human-readable description of this routing change.
    pub description: Option<String>,
    /// Name of the environment group affected by this routing change.
    #[serde(rename="environmentGroup")]
    pub environment_group: Option<String>,
    /// Base path/deployment that may stop receiving some traffic.
    #[serde(rename="fromDeployment")]
    pub from_deployment: Option<GoogleCloudApigeeV1DeploymentChangeReportRoutingDeployment>,
    /// Set to `true` if using sequenced rollout would make this routing change safer. **Note**: This does not necessarily imply that automated sequenced rollout mode is supported for the operation.
    #[serde(rename="shouldSequenceRollout")]
    pub should_sequence_rollout: Option<bool>,
    /// Base path/deployment that may start receiving that traffic. May be null if no deployment is able to receive the traffic.
    #[serde(rename="toDeployment")]
    pub to_deployment: Option<GoogleCloudApigeeV1DeploymentChangeReportRoutingDeployment>,
}

impl client::Part for GoogleCloudApigeeV1DeploymentChangeReportRoutingChange {}


/// Describes a routing conflict that may cause a deployment not to receive traffic at some base path.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeploymentChangeReportRoutingConflict {
    /// Existing base path/deployment causing the conflict.
    #[serde(rename="conflictingDeployment")]
    pub conflicting_deployment: Option<GoogleCloudApigeeV1DeploymentChangeReportRoutingDeployment>,
    /// Human-readable description of this conflict.
    pub description: Option<String>,
    /// Name of the environment group in which this conflict exists.
    #[serde(rename="environmentGroup")]
    pub environment_group: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DeploymentChangeReportRoutingConflict {}


/// Tuple representing a base path and the deployment containing it.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeploymentChangeReportRoutingDeployment {
    /// Name of the deployed API proxy revision containing the base path.
    #[serde(rename="apiProxy")]
    pub api_proxy: Option<String>,
    /// Base path receiving traffic.
    pub basepath: Option<String>,
    /// Name of the environment in which the proxy is deployed.
    pub environment: Option<String>,
    /// Name of the deployed API proxy revision containing the base path.
    pub revision: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DeploymentChangeReportRoutingDeployment {}


/// NEXT ID: 9
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeploymentConfig {
    /// Additional key-value metadata for the deployment.
    pub attributes: Option<HashMap<String, String>>,
    /// Base path where the application will be hosted. Defaults to "/".
    #[serde(rename="basePath")]
    pub base_path: Option<String>,
    /// Location of the API proxy bundle as a URI.
    pub location: Option<String>,
    /// Name of the API or shared flow revision to be deployed in the following format: `organizations/{org}/apis/{api}/revisions/{rev}` or `organizations/{org}/sharedflows/{sharedflow}/revisions/{rev}`
    pub name: Option<String>,
    /// Unique ID of the API proxy revision.
    #[serde(rename="proxyUid")]
    pub proxy_uid: Option<String>,
    /// The service account identity associated with this deployment. If non-empty, will be in the following format: `projects/-/serviceAccounts/{account_email}`
    #[serde(rename="serviceAccount")]
    pub service_account: Option<String>,
    /// Unique ID. The ID will only change if the deployment is deleted and recreated.
    pub uid: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DeploymentConfig {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers create organizations](OrganizationDeveloperCreateCall) (request|response)
/// * [developers delete organizations](OrganizationDeveloperDeleteCall) (response)
/// * [developers get organizations](OrganizationDeveloperGetCall) (response)
/// * [developers update organizations](OrganizationDeveloperUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Developer {
    /// Access type.
    #[serde(rename="accessType")]
    pub access_type: Option<String>,
    /// Developer app family.
    #[serde(rename="appFamily")]
    pub app_family: Option<String>,
    /// List of apps associated with the developer.
    pub apps: Option<Vec<String>>,
    /// Optional. Developer attributes (name/value pairs). The custom attribute limit is 18.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// List of companies associated with the developer.
    pub companies: Option<Vec<String>>,
    /// Output only. Time at which the developer was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// ID of the developer. **Note**: IDs are generated internally by Apigee and are not guaranteed to stay the same over time.
    #[serde(rename="developerId")]
    pub developer_id: Option<String>,
    /// Required. Email address of the developer. This value is used to uniquely identify the developer in Apigee hybrid. Note that the email address has to be in lowercase only.
    pub email: Option<String>,
    /// Required. First name of the developer.
    #[serde(rename="firstName")]
    pub first_name: Option<String>,
    /// Output only. Time at which the developer was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Required. Last name of the developer.
    #[serde(rename="lastName")]
    pub last_name: Option<String>,
    /// Output only. Name of the Apigee organization in which the developer resides.
    #[serde(rename="organizationName")]
    pub organization_name: Option<String>,
    /// Output only. Status of the developer. Valid values are `active` and `inactive`.
    pub status: Option<String>,
    /// Required. User name of the developer. Not used by Apigee hybrid.
    #[serde(rename="userName")]
    pub user_name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Developer {}
impl client::ResponseResult for GoogleCloudApigeeV1Developer {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers apps create organizations](OrganizationDeveloperAppCreateCall) (request|response)
/// * [developers apps delete organizations](OrganizationDeveloperAppDeleteCall) (response)
/// * [developers apps generate key pair or update developer app status organizations](OrganizationDeveloperAppGenerateKeyPairOrUpdateDeveloperAppStatuCall) (request|response)
/// * [developers apps get organizations](OrganizationDeveloperAppGetCall) (response)
/// * [developers apps update organizations](OrganizationDeveloperAppUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperApp {
    /// List of API products associated with the developer app.
    #[serde(rename="apiProducts")]
    pub api_products: Option<Vec<String>>,
    /// Developer app family.
    #[serde(rename="appFamily")]
    pub app_family: Option<String>,
    /// ID of the developer app.
    #[serde(rename="appId")]
    pub app_id: Option<String>,
    /// List of attributes for the developer app.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Callback URL used by OAuth 2.0 authorization servers to communicate authorization codes back to developer apps.
    #[serde(rename="callbackUrl")]
    pub callback_url: Option<String>,
    /// Output only. Time the developer app was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Output only. Set of credentials for the developer app consisting of the consumer key/secret pairs associated with the API products.
    pub credentials: Option<Vec<GoogleCloudApigeeV1Credential>>,
    /// ID of the developer.
    #[serde(rename="developerId")]
    pub developer_id: Option<String>,
    /// Expiration time, in milliseconds, for the consumer key that is generated for the developer app. If not set or left to the default value of `-1`, the API key never expires. The expiration time can't be updated after it is set.
    #[serde(rename="keyExpiresIn")]
    pub key_expires_in: Option<String>,
    /// Output only. Time the developer app was modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Name of the developer app.
    pub name: Option<String>,
    /// Scopes to apply to the developer app. The specified scopes must already exist for the API product that you associate with the developer app.
    pub scopes: Option<Vec<String>>,
    /// Status of the credential. Valid values include `approved` or `revoked`.
    pub status: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1DeveloperApp {}
impl client::ResponseResult for GoogleCloudApigeeV1DeveloperApp {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers apps keys apiproducts delete organizations](OrganizationDeveloperAppKeyApiproductDeleteCall) (response)
/// * [developers apps keys create create organizations](OrganizationDeveloperAppKeyCreateCreateCall) (request|response)
/// * [developers apps keys create organizations](OrganizationDeveloperAppKeyCreateCall) (request|response)
/// * [developers apps keys delete organizations](OrganizationDeveloperAppKeyDeleteCall) (response)
/// * [developers apps keys get organizations](OrganizationDeveloperAppKeyGetCall) (response)
/// * [developers apps keys replace developer app key organizations](OrganizationDeveloperAppKeyReplaceDeveloperAppKeyCall) (request|response)
/// * [developers apps keys update developer app key organizations](OrganizationDeveloperAppKeyUpdateDeveloperAppKeyCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperAppKey {
    /// List of API products for which the credential can be used. **Note**: Do not specify the list of API products when creating a consumer key and secret for a developer app. Instead, use the UpdateDeveloperAppKey API to make the association after the consumer key and secret are created.
    #[serde(rename="apiProducts")]
    pub api_products: Option<Vec<String>>,
    /// List of attributes associated with the credential.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Consumer key.
    #[serde(rename="consumerKey")]
    pub consumer_key: Option<String>,
    /// Secret key.
    #[serde(rename="consumerSecret")]
    pub consumer_secret: Option<String>,
    /// Time the developer app expires in milliseconds since epoch.
    #[serde(rename="expiresAt")]
    pub expires_at: Option<String>,
    /// Input only. Expiration time, in seconds, for the consumer key. If not set or left to the default value of `-1`, the API key never expires. The expiration time can't be updated after it is set.
    #[serde(rename="expiresInSeconds")]
    pub expires_in_seconds: Option<String>,
    /// Time the developer app was created in milliseconds since epoch.
    #[serde(rename="issuedAt")]
    pub issued_at: Option<String>,
    /// Scopes to apply to the app. The specified scope names must already be defined for the API product that you associate with the app.
    pub scopes: Option<Vec<String>>,
    /// Status of the credential. Valid values include `approved` or `revoked`.
    pub status: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1DeveloperAppKey {}
impl client::ResponseResult for GoogleCloudApigeeV1DeveloperAppKey {}


/// Account balance for the developer.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers balance adjust organizations](OrganizationDeveloperBalanceAdjustCall) (response)
/// * [developers balance credit organizations](OrganizationDeveloperBalanceCreditCall) (response)
/// * [developers get balance organizations](OrganizationDeveloperGetBalanceCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperBalance {
    /// Output only. List of all wallets. Each individual wallet stores the account balance for a particular currency.
    pub wallets: Option<Vec<GoogleCloudApigeeV1DeveloperBalanceWallet>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1DeveloperBalance {}


/// Wallet used to manage an account balance for a particular currency.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperBalanceWallet {
    /// Current remaining balance of the developer for a particular currency.
    pub balance: Option<GoogleTypeMoney>,
    /// Output only. Time at which the developer last added credit to the account in milliseconds since epoch.
    #[serde(rename="lastCreditTime")]
    pub last_credit_time: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DeveloperBalanceWallet {}


/// Monetization configuration for the developer.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers get monetization config organizations](OrganizationDeveloperGetMonetizationConfigCall) (response)
/// * [developers update monetization config organizations](OrganizationDeveloperUpdateMonetizationConfigCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperMonetizationConfig {
    /// Billing type.
    #[serde(rename="billingType")]
    pub billing_type: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1DeveloperMonetizationConfig {}
impl client::ResponseResult for GoogleCloudApigeeV1DeveloperMonetizationConfig {}


/// Structure of a DeveloperSubscription.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers subscriptions create organizations](OrganizationDeveloperSubscriptionCreateCall) (request|response)
/// * [developers subscriptions expire organizations](OrganizationDeveloperSubscriptionExpireCall) (response)
/// * [developers subscriptions get organizations](OrganizationDeveloperSubscriptionGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DeveloperSubscription {
    /// Name of the API product for which the developer is purchasing a subscription.
    pub apiproduct: Option<String>,
    /// Output only. Time when the API product subscription was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Time when the API product subscription ends in milliseconds since epoch.
    #[serde(rename="endTime")]
    pub end_time: Option<String>,
    /// Output only. Time when the API product subscription was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Output only. Name of the API product subscription.
    pub name: Option<String>,
    /// Time when the API product subscription starts in milliseconds since epoch.
    #[serde(rename="startTime")]
    pub start_time: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1DeveloperSubscription {}
impl client::ResponseResult for GoogleCloudApigeeV1DeveloperSubscription {}


/// Encapsulates a metric grouped by dimension.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1DimensionMetric {
    /// List of metrics.
    pub metrics: Option<Vec<GoogleCloudApigeeV1Metric>>,
    /// Name of the dimension.
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1DimensionMetric {}


/// Apigee endpoint attachment. For more information, see Southbound networking patterns.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [endpoint attachments create organizations](OrganizationEndpointAttachmentCreateCall) (request)
/// * [endpoint attachments get organizations](OrganizationEndpointAttachmentGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EndpointAttachment {
    /// Output only. Host that can be used in either the HTTP target endpoint directly or as the host in target server.
    pub host: Option<String>,
    /// Required. Location of the endpoint attachment.
    pub location: Option<String>,
    /// Name of the endpoint attachment. Use the following structure in your request: `organizations/{org}/endpointAttachments/{endpoint_attachment}`
    pub name: Option<String>,
    /// Format: projects/*/regions/*/serviceAttachments/*
    #[serde(rename="serviceAttachment")]
    pub service_attachment: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1EndpointAttachment {}
impl client::ResponseResult for GoogleCloudApigeeV1EndpointAttachment {}


/// Metadata common to many entities in this API.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EntityMetadata {
    /// Time at which the API proxy was created, in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Time at which the API proxy was most recently modified, in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// The type of entity described
    #[serde(rename="subType")]
    pub sub_type: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1EntityMetadata {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments create organizations](OrganizationEnvironmentCreateCall) (request)
/// * [environments get organizations](OrganizationEnvironmentGetCall) (response)
/// * [environments update organizations](OrganizationEnvironmentUpdateCall) (request|response)
/// * [environments update environment organizations](OrganizationEnvironmentUpdateEnvironmentCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Environment {
    /// Optional. API Proxy type supported by the environment. The type can be set when creating the Environment and cannot be changed.
    #[serde(rename="apiProxyType")]
    pub api_proxy_type: Option<String>,
    /// Output only. Creation time of this environment as milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Optional. Deployment type supported by the environment. The deployment type can be set when creating the environment and cannot be changed. When you enable archive deployment, you will be **prevented from performing** a [subset of actions](/apigee/docs/api-platform/local-development/overview#prevented-actions) within the environment, including: * Managing the deployment of API proxy or shared flow revisions * Creating, updating, or deleting resource files * Creating, updating, or deleting target servers
    #[serde(rename="deploymentType")]
    pub deployment_type: Option<String>,
    /// Optional. Description of the environment.
    pub description: Option<String>,
    /// Optional. Display name for this environment.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Output only. Last modification time of this environment as milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Required. Name of the environment. Values must match the regular expression `^[.\\p{Alnum}-_]{1,255}$`
    pub name: Option<String>,
    /// Optional. Key-value pairs that may be used for customizing the environment.
    pub properties: Option<GoogleCloudApigeeV1Properties>,
    /// Output only. State of the environment. Values other than ACTIVE means the resource is not ready to use.
    pub state: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Environment {}
impl client::ResponseResult for GoogleCloudApigeeV1Environment {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments get deployed config organizations](OrganizationEnvironmentGetDeployedConfigCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EnvironmentConfig {
    /// The location for the config blob of API Runtime Control, aka Envoy Adapter, for op-based authentication as a URI, e.g. a Cloud Storage URI. This is only used by Envoy-based gateways.
    #[serde(rename="arcConfigLocation")]
    pub arc_config_location: Option<String>,
    /// Time that the environment configuration was created.
    #[serde(rename="createTime")]
    pub create_time: Option<String>,
    /// List of data collectors used by the deployments in the environment.
    #[serde(rename="dataCollectors")]
    pub data_collectors: Option<Vec<GoogleCloudApigeeV1DataCollectorConfig>>,
    /// Debug mask that applies to all deployments in the environment.
    #[serde(rename="debugMask")]
    pub debug_mask: Option<GoogleCloudApigeeV1DebugMask>,
    /// List of deployments in the environment.
    pub deployments: Option<Vec<GoogleCloudApigeeV1DeploymentConfig>>,
    /// Feature flags inherited from the organization and environment.
    #[serde(rename="featureFlags")]
    pub feature_flags: Option<HashMap<String, String>>,
    /// List of flow hooks in the environment.
    pub flowhooks: Option<Vec<GoogleCloudApigeeV1FlowHookConfig>>,
    /// The location for the gateway config blob as a URI, e.g. a Cloud Storage URI. This is only used by Envoy-based gateways.
    #[serde(rename="gatewayConfigLocation")]
    pub gateway_config_location: Option<String>,
    /// List of keystores in the environment.
    pub keystores: Option<Vec<GoogleCloudApigeeV1KeystoreConfig>>,
    /// Name of the environment configuration in the following format: `organizations/{org}/environments/{env}/configs/{config}`
    pub name: Option<String>,
    /// Used by the Control plane to add context information to help detect the source of the document during diagnostics and debugging.
    pub provider: Option<String>,
    /// Name of the PubSub topic for the environment.
    #[serde(rename="pubsubTopic")]
    pub pubsub_topic: Option<String>,
    /// List of resource references in the environment.
    #[serde(rename="resourceReferences")]
    pub resource_references: Option<Vec<GoogleCloudApigeeV1ReferenceConfig>>,
    /// List of resource versions in the environment.
    pub resources: Option<Vec<GoogleCloudApigeeV1ResourceConfig>>,
    /// Revision ID of the environment configuration. The higher the value, the more recently the configuration was deployed.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
    /// DEPRECATED: Use revision_id.
    #[serde(rename="sequenceNumber")]
    pub sequence_number: Option<String>,
    /// List of target servers in the environment. Disabled target servers are not displayed.
    pub targets: Option<Vec<GoogleCloudApigeeV1TargetServerConfig>>,
    /// Trace configurations. Contains config for the environment and config overrides for specific API proxies.
    #[serde(rename="traceConfig")]
    pub trace_config: Option<GoogleCloudApigeeV1RuntimeTraceConfig>,
    /// Unique ID for the environment configuration. The ID will only change if the environment is deleted and recreated.
    pub uid: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1EnvironmentConfig {}


/// EnvironmentGroup configuration. An environment group is used to group one or more Apigee environments under a single host name.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [envgroups create organizations](OrganizationEnvgroupCreateCall) (request)
/// * [envgroups get organizations](OrganizationEnvgroupGetCall) (response)
/// * [envgroups patch organizations](OrganizationEnvgroupPatchCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EnvironmentGroup {
    /// Output only. The time at which the environment group was created as milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Required. Host names for this environment group.
    pub hostnames: Option<Vec<String>>,
    /// Output only. The time at which the environment group was last updated as milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// ID of the environment group.
    pub name: Option<String>,
    /// Output only. State of the environment group. Values other than ACTIVE means the resource is not ready to use.
    pub state: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1EnvironmentGroup {}
impl client::ResponseResult for GoogleCloudApigeeV1EnvironmentGroup {}


/// EnvironmentGroupAttachment is a resource which defines an attachment of an environment to an environment group.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [envgroups attachments create organizations](OrganizationEnvgroupAttachmentCreateCall) (request)
/// * [envgroups attachments get organizations](OrganizationEnvgroupAttachmentGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EnvironmentGroupAttachment {
    /// Output only. The time at which the environment group attachment was created as milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Required. ID of the attached environment.
    pub environment: Option<String>,
    /// ID of the environment group attachment.
    pub name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1EnvironmentGroupAttachment {}
impl client::ResponseResult for GoogleCloudApigeeV1EnvironmentGroupAttachment {}


/// EnvironmentGroupConfig is a revisioned snapshot of an EnvironmentGroup and its associated routing rules.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1EnvironmentGroupConfig {
    /// Host names for the environment group.
    pub hostnames: Option<Vec<String>>,
    /// Name of the environment group in the following format: `organizations/{org}/envgroups/{envgroup}`.
    pub name: Option<String>,
    /// Revision id that defines the ordering of the EnvironmentGroupConfig resource. The higher the revision, the more recently the configuration was deployed.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
    /// Ordered list of routing rules defining how traffic to this environment group's hostnames should be routed to different environments.
    #[serde(rename="routingRules")]
    pub routing_rules: Option<Vec<GoogleCloudApigeeV1RoutingRule>>,
    /// A unique id for the environment group config that will only change if the environment group is deleted and recreated.
    pub uid: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1EnvironmentGroupConfig {}


/// Request for ExpireDeveloperSubscription.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers subscriptions expire organizations](OrganizationDeveloperSubscriptionExpireCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ExpireDeveloperSubscriptionRequest { _never_set: Option<bool> }

impl client::RequestValue for GoogleCloudApigeeV1ExpireDeveloperSubscriptionRequest {}


/// Details of an export job.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments analytics exports create organizations](OrganizationEnvironmentAnalyticExportCreateCall) (response)
/// * [environments analytics exports get organizations](OrganizationEnvironmentAnalyticExportGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Export {
    /// Output only. Time the export job was created.
    pub created: Option<String>,
    /// Name of the datastore that is the destination of the export job [datastore]
    #[serde(rename="datastoreName")]
    pub datastore_name: Option<String>,
    /// Description of the export job.
    pub description: Option<String>,
    /// Output only. Error is set when export fails
    pub error: Option<String>,
    /// Output only. Execution time for this export job. If the job is still in progress, it will be set to the amount of time that has elapsed since`created`, in seconds. Else, it will set to (`updated` - `created`), in seconds.
    #[serde(rename="executionTime")]
    pub execution_time: Option<String>,
    /// Display name of the export job.
    pub name: Option<String>,
    /// Output only. Self link of the export job. A URI that can be used to retrieve the status of an export job. Example: `/organizations/myorg/environments/myenv/analytics/exports/9cfc0d85-0f30-46d6-ae6f-318d0cb961bd`
    #[serde(rename="self")]
    pub self_: Option<String>,
    /// Output only. Status of the export job. Valid values include `enqueued`, `running`, `completed`, and `failed`.
    pub state: Option<String>,
    /// Output only. Time the export job was last updated.
    pub updated: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1Export {}


/// Request body for [CreateExportRequest]
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments analytics exports create organizations](OrganizationEnvironmentAnalyticExportCreateCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ExportRequest {
    /// Optional. Delimiter used in the CSV file, if `outputFormat` is set to `csv`. Defaults to the `,` (comma) character. Supported delimiter characters include comma (`,`), pipe (`|`), and tab (`\t`).
    #[serde(rename="csvDelimiter")]
    pub csv_delimiter: Option<String>,
    /// Required. Name of the preconfigured datastore.
    #[serde(rename="datastoreName")]
    pub datastore_name: Option<String>,
    /// Required. Date range of the data to export.
    #[serde(rename="dateRange")]
    pub date_range: Option<GoogleCloudApigeeV1DateRange>,
    /// Optional. Description of the export job.
    pub description: Option<String>,
    /// Required. Display name of the export job.
    pub name: Option<String>,
    /// Optional. Output format of the export. Valid values include: `csv` or `json`. Defaults to `json`. Note: Configure the delimiter for CSV output using the `csvDelimiter` property.
    #[serde(rename="outputFormat")]
    pub output_format: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1ExportRequest {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments flowhooks attach shared flow to flow hook organizations](OrganizationEnvironmentFlowhookAttachSharedFlowToFlowHookCall) (request|response)
/// * [environments flowhooks detach shared flow from flow hook organizations](OrganizationEnvironmentFlowhookDetachSharedFlowFromFlowHookCall) (response)
/// * [environments flowhooks get organizations](OrganizationEnvironmentFlowhookGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1FlowHook {
    /// Optional. Flag that specifies whether execution should continue if the flow hook throws an exception. Set to `true` to continue execution. Set to `false` to stop execution if the flow hook throws an exception. Defaults to `true`.
    #[serde(rename="continueOnError")]
    pub continue_on_error: Option<bool>,
    /// Description of the flow hook.
    pub description: Option<String>,
    /// Output only. Where in the API call flow the flow hook is invoked. Must be one of `PreProxyFlowHook`, `PostProxyFlowHook`, `PreTargetFlowHook`, or `PostTargetFlowHook`.
    #[serde(rename="flowHookPoint")]
    pub flow_hook_point: Option<String>,
    /// Shared flow attached to this flow hook, or empty if there is none attached.
    #[serde(rename="sharedFlow")]
    pub shared_flow: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1FlowHook {}
impl client::ResponseResult for GoogleCloudApigeeV1FlowHook {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1FlowHookConfig {
    /// Flag that specifies whether the flow should abort after an error in the flow hook. Defaults to `true` (continue on error).
    #[serde(rename="continueOnError")]
    pub continue_on_error: Option<bool>,
    /// Name of the flow hook in the following format: `organizations/{org}/environments/{env}/flowhooks/{point}`. Valid `point` values include: `PreProxyFlowHook`, `PostProxyFlowHook`, `PreTargetFlowHook`, and `PostTargetFlowHook`
    pub name: Option<String>,
    /// Name of the shared flow to invoke in the following format: `organizations/{org}/sharedflows/{sharedflow}`
    #[serde(rename="sharedFlowName")]
    pub shared_flow_name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1FlowHookConfig {}


/// Request for GenerateDownloadUrl method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments generate download url organizations](OrganizationEnvironmentArchiveDeploymentGenerateDownloadUrlCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GenerateDownloadUrlRequest { _never_set: Option<bool> }

impl client::RequestValue for GoogleCloudApigeeV1GenerateDownloadUrlRequest {}


/// Response for GenerateDownloadUrl method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments generate download url organizations](OrganizationEnvironmentArchiveDeploymentGenerateDownloadUrlCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GenerateDownloadUrlResponse {
    /// The Google Cloud Storage signed URL that can be used to download the Archive zip file.
    #[serde(rename="downloadUri")]
    pub download_uri: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1GenerateDownloadUrlResponse {}


/// Request for GenerateUploadUrl method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments generate upload url organizations](OrganizationEnvironmentArchiveDeploymentGenerateUploadUrlCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GenerateUploadUrlRequest { _never_set: Option<bool> }

impl client::RequestValue for GoogleCloudApigeeV1GenerateUploadUrlRequest {}


/// Response for GenerateUploadUrl method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments generate upload url organizations](OrganizationEnvironmentArchiveDeploymentGenerateUploadUrlCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GenerateUploadUrlResponse {
    /// The Google Cloud Storage signed URL that can be used to upload a new Archive zip file.
    #[serde(rename="uploadUri")]
    pub upload_uri: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1GenerateUploadUrlResponse {}


/// Request for GetSyncAuthorization.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [get sync authorization organizations](OrganizationGetSyncAuthorizationCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GetSyncAuthorizationRequest { _never_set: Option<bool> }

impl client::RequestValue for GoogleCloudApigeeV1GetSyncAuthorizationRequest {}


/// Represents the pairing of GraphQL operation types and the GraphQL operation name.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GraphQLOperation {
    /// GraphQL operation name. The name and operation type will be used to apply quotas. If no name is specified, the quota will be applied to all GraphQL operations irrespective of their operation names in the payload.
    pub operation: Option<String>,
    /// Required. GraphQL operation types. Valid values include `query` or `mutation`. **Note**: Apigee does not currently support `subscription` types.
    #[serde(rename="operationTypes")]
    pub operation_types: Option<Vec<String>>,
}

impl client::Part for GoogleCloudApigeeV1GraphQLOperation {}


/// Binds the resources in a proxy or remote service with the GraphQL operation and its associated quota enforcement.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GraphQLOperationConfig {
    /// Required. Name of the API proxy endpoint or remote service with which the GraphQL operation and quota are associated.
    #[serde(rename="apiSource")]
    pub api_source: Option<String>,
    /// Custom attributes associated with the operation.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// Required. List of GraphQL name/operation type pairs for the proxy or remote service to which quota will be applied. If only operation types are specified, the quota will be applied to all GraphQL requests irrespective of the GraphQL name. **Note**: Currently, you can specify only a single GraphQLOperation. Specifying more than one will cause the operation to fail.
    pub operations: Option<Vec<GoogleCloudApigeeV1GraphQLOperation>>,
    /// Quota parameters to be enforced for the resources, methods, and API source combination. If none are specified, quota enforcement will not be done.
    pub quota: Option<GoogleCloudApigeeV1Quota>,
}

impl client::Part for GoogleCloudApigeeV1GraphQLOperationConfig {}


/// List of graphQL operation configuration details associated with Apigee API proxies or remote services. Remote services are non-Apigee proxies, such as Istio-Envoy.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1GraphQLOperationGroup {
    /// Flag that specifies whether the configuration is for Apigee API proxy or a remote service. Valid values include `proxy` or `remoteservice`. Defaults to `proxy`. Set to `proxy` when Apigee API proxies are associated with the API product. Set to `remoteservice` when non-Apigee proxies like Istio-Envoy are associated with the API product.
    #[serde(rename="operationConfigType")]
    pub operation_config_type: Option<String>,
    /// Required. List of operation configurations for either Apigee API proxies or other remote services that are associated with this API product.
    #[serde(rename="operationConfigs")]
    pub operation_configs: Option<Vec<GoogleCloudApigeeV1GraphQLOperationConfig>>,
}

impl client::Part for GoogleCloudApigeeV1GraphQLOperationGroup {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [get deployed ingress config organizations](OrganizationGetDeployedIngressConfigCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1IngressConfig {
    /// List of environment groups in the organization.
    #[serde(rename="environmentGroups")]
    pub environment_groups: Option<Vec<GoogleCloudApigeeV1EnvironmentGroupConfig>>,
    /// Name of the resource in the following format: `organizations/{org}/deployedIngressConfig`.
    pub name: Option<String>,
    /// Time at which the IngressConfig revision was created.
    #[serde(rename="revisionCreateTime")]
    pub revision_create_time: Option<String>,
    /// Revision id that defines the ordering on IngressConfig resources. The higher the revision, the more recently the configuration was deployed.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
    /// A unique id for the ingress config that will only change if the organization is deleted and recreated.
    pub uid: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1IngressConfig {}


/// Apigee runtime instance.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances create organizations](OrganizationInstanceCreateCall) (request)
/// * [instances get organizations](OrganizationInstanceGetCall) (response)
/// * [instances patch organizations](OrganizationInstancePatchCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Instance {
    /// Output only. Time the instance was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Optional. Description of the instance.
    pub description: Option<String>,
    /// Customer Managed Encryption Key (CMEK) used for disk and volume encryption. Required for Apigee paid subscriptions only. Use the following format: `projects/([^/]+)/locations/([^/]+)/keyRings/([^/]+)/cryptoKeys/([^/]+)`
    #[serde(rename="diskEncryptionKeyName")]
    pub disk_encryption_key_name: Option<String>,
    /// Optional. Display name for the instance.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Output only. Internal hostname or IP address of the Apigee endpoint used by clients to connect to the service.
    pub host: Option<String>,
    /// Optional. IP range represents the customer-provided CIDR block of length 22 that will be used for the Apigee instance creation. This optional range, if provided, should be freely available as part of larger named range the customer has allocated to the Service Networking peering. If this is not provided, Apigee will automatically request for any available /22 CIDR block from Service Networking. The customer should use this CIDR block for configuring their firewall needs to allow traffic from Apigee. Input format: "a.b.c.d/22", Output format: a.b.c.d/22, e.f.g.h/28"
    #[serde(rename="ipRange")]
    pub ip_range: Option<String>,
    /// Output only. Time the instance was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Required. Compute Engine location where the instance resides.
    pub location: Option<String>,
    /// Required. Resource ID of the instance. Values must match the regular expression `^a-z{0,30}[a-z\d]$`.
    pub name: Option<String>,
    /// Optional. Size of the CIDR block range that will be reserved by the instance. PAID organizations support `SLASH_16` to `SLASH_20` and defaults to `SLASH_16`. Evaluation organizations support only `SLASH_23`.
    #[serde(rename="peeringCidrRange")]
    pub peering_cidr_range: Option<String>,
    /// Output only. Port number of the exposed Apigee endpoint.
    pub port: Option<String>,
    /// Output only. Version of the runtime system running in the instance. The runtime system is the set of components that serve the API Proxy traffic in your Environments.
    #[serde(rename="runtimeVersion")]
    pub runtime_version: Option<String>,
    /// Output only. State of the instance. Values other than `ACTIVE` means the resource is not ready to use.
    pub state: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Instance {}
impl client::ResponseResult for GoogleCloudApigeeV1Instance {}


/// InstanceAttachment represents the installation of an environment onto an instance.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances attachments create organizations](OrganizationInstanceAttachmentCreateCall) (request)
/// * [instances attachments get organizations](OrganizationInstanceAttachmentGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1InstanceAttachment {
    /// Output only. Time the attachment was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// ID of the attached environment.
    pub environment: Option<String>,
    /// Output only. ID of the attachment.
    pub name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1InstanceAttachment {}
impl client::ResponseResult for GoogleCloudApigeeV1InstanceAttachment {}


/// The status of a deployment as reported by a single instance.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1InstanceDeploymentStatus {
    /// Revisions currently deployed in MPs.
    #[serde(rename="deployedRevisions")]
    pub deployed_revisions: Option<Vec<GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRevision>>,
    /// Current routes deployed in the ingress routing table. A route which is missing will appear in `missing_routes`.
    #[serde(rename="deployedRoutes")]
    pub deployed_routes: Option<Vec<GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRoute>>,
    /// ID of the instance reporting the status.
    pub instance: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1InstanceDeploymentStatus {}


/// Revisions deployed in the MPs.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRevision {
    /// Percentage of MP replicas reporting this revision.
    pub percentage: Option<i32>,
    /// API proxy revision reported as deployed.
    pub revision: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRevision {}


/// Route deployed in the ingress routing table.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRoute {
    /// Base path in the routing table.
    pub basepath: Option<String>,
    /// Environment group where this route is installed.
    pub envgroup: Option<String>,
    /// Destination environment. This will be empty if the route is not yet reported.
    pub environment: Option<String>,
    /// Percentage of ingress replicas reporting this route.
    pub percentage: Option<i32>,
}

impl client::Part for GoogleCloudApigeeV1InstanceDeploymentStatusDeployedRoute {}


/// Configuration for the Integration add-on.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1IntegrationConfig {
    /// Flag that specifies whether the Integration add-on is enabled.
    pub enabled: Option<bool>,
}

impl client::Part for GoogleCloudApigeeV1IntegrationConfig {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1KeyAliasReference {
    /// Alias ID. Must exist in the keystore referred to by the reference.
    #[serde(rename="aliasId")]
    pub alias_id: Option<String>,
    /// Reference name in the following format: `organizations/{org}/environments/{env}/references/{reference}`
    pub reference: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1KeyAliasReference {}


/// Collection of key/value string pairs.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis keyvaluemaps create organizations](OrganizationApiKeyvaluemapCreateCall) (request|response)
/// * [apis keyvaluemaps delete organizations](OrganizationApiKeyvaluemapDeleteCall) (response)
/// * [environments keyvaluemaps create organizations](OrganizationEnvironmentKeyvaluemapCreateCall) (request|response)
/// * [environments keyvaluemaps delete organizations](OrganizationEnvironmentKeyvaluemapDeleteCall) (response)
/// * [keyvaluemaps create organizations](OrganizationKeyvaluemapCreateCall) (request|response)
/// * [keyvaluemaps delete organizations](OrganizationKeyvaluemapDeleteCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1KeyValueMap {
    /// Optional. Flag that specifies whether entry values will be encrypted. Enable to encrypt entry values.
    pub encrypted: Option<bool>,
    /// Required. ID of the key value map.
    pub name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1KeyValueMap {}
impl client::ResponseResult for GoogleCloudApigeeV1KeyValueMap {}


/// Datastore for Certificates and Aliases.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments keystores create organizations](OrganizationEnvironmentKeystoreCreateCall) (request|response)
/// * [environments keystores delete organizations](OrganizationEnvironmentKeystoreDeleteCall) (response)
/// * [environments keystores get organizations](OrganizationEnvironmentKeystoreGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Keystore {
    /// Output only. Aliases in this keystore.
    pub aliases: Option<Vec<String>>,
    /// Required. Resource ID for this keystore. Values must match the regular expression `[\w[:space:]-.]{1,255}`.
    pub name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Keystore {}
impl client::ResponseResult for GoogleCloudApigeeV1Keystore {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1KeystoreConfig {
    /// Aliases in the keystore.
    pub aliases: Option<Vec<GoogleCloudApigeeV1AliasRevisionConfig>>,
    /// Resource name in the following format: `organizations/{org}/environments/{env}/keystores/{keystore}`
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1KeystoreConfig {}


/// the response for ListApiCategoriesRequest.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sites apicategories list organizations](OrganizationSiteApicategoryListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListApiCategoriesResponse {
    /// Details of categories.
    pub data: Option<Vec<GoogleCloudApigeeV1ApiCategoryData>>,
    /// ID that can be used to find errors in the log files.
    #[serde(rename="errorCode")]
    pub error_code: Option<String>,
    /// Description of the operation.
    pub message: Option<String>,
    /// ID that can be used to find request details in the log files.
    #[serde(rename="requestId")]
    pub request_id: Option<String>,
    /// Status of the operation.
    pub status: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListApiCategoriesResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts list organizations](OrganizationApiproductListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListApiProductsResponse {
    /// Lists all API product names defined for an organization.
    #[serde(rename="apiProduct")]
    pub api_product: Option<Vec<GoogleCloudApigeeV1ApiProduct>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListApiProductsResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis list organizations](OrganizationApiListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListApiProxiesResponse {
    /// no description provided
    pub proxies: Option<Vec<GoogleCloudApigeeV1ApiProxy>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListApiProxiesResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apps list organizations](OrganizationAppListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListAppsResponse {
    /// no description provided
    pub app: Option<Vec<GoogleCloudApigeeV1App>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListAppsResponse {}


/// Response for ListArchiveDeployments method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments archive deployments list organizations](OrganizationEnvironmentArchiveDeploymentListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListArchiveDeploymentsResponse {
    /// Archive Deployments in the specified environment.
    #[serde(rename="archiveDeployments")]
    pub archive_deployments: Option<Vec<GoogleCloudApigeeV1ArchiveDeployment>>,
    /// Page token that you can include in a ListArchiveDeployments request to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListArchiveDeploymentsResponse {}


/// The response for ListAsyncQueries.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments queries list organizations](OrganizationEnvironmentQueryListCall) (response)
/// * [host queries list organizations](OrganizationHostQueryListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListAsyncQueriesResponse {
    /// The asynchronous queries belong to requested resource name.
    pub queries: Option<Vec<GoogleCloudApigeeV1AsyncQuery>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListAsyncQueriesResponse {}


/// This message encapsulates a list of custom report definitions
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [reports list organizations](OrganizationReportListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListCustomReportsResponse {
    /// no description provided
    pub qualifier: Option<Vec<GoogleCloudApigeeV1CustomReport>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListCustomReportsResponse {}


/// Response for ListDataCollectors.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [datacollectors list organizations](OrganizationDatacollectorListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDataCollectorsResponse {
    /// Data collectors in the specified organization.
    #[serde(rename="dataCollectors")]
    pub data_collectors: Option<Vec<GoogleCloudApigeeV1DataCollector>>,
    /// Page token that you can include in a ListDataCollectors request to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDataCollectorsResponse {}


/// The response for ListDatastores
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [analytics datastores list organizations](OrganizationAnalyticDatastoreListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDatastoresResponse {
    /// A list of datastores
    pub datastores: Option<Vec<GoogleCloudApigeeV1Datastore>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDatastoresResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments apis revisions debugsessions list organizations](OrganizationEnvironmentApiRevisionDebugsessionListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDebugSessionsResponse {
    /// Page token that you can include in a ListDebugSessionsRequest to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
    /// Session info that includes debug session ID and the first transaction creation timestamp.
    pub sessions: Option<Vec<GoogleCloudApigeeV1Session>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDebugSessionsResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apis deployments list organizations](OrganizationApiDeploymentListCall) (response)
/// * [apis revisions deployments list organizations](OrganizationApiRevisionDeploymentListCall) (response)
/// * [deployments list organizations](OrganizationDeploymentListCall) (response)
/// * [environments apis deployments list organizations](OrganizationEnvironmentApiDeploymentListCall) (response)
/// * [environments deployments list organizations](OrganizationEnvironmentDeploymentListCall) (response)
/// * [environments sharedflows deployments list organizations](OrganizationEnvironmentSharedflowDeploymentListCall) (response)
/// * [sharedflows deployments list organizations](OrganizationSharedflowDeploymentListCall) (response)
/// * [sharedflows revisions deployments list organizations](OrganizationSharedflowRevisionDeploymentListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDeploymentsResponse {
    /// List of deployments.
    pub deployments: Option<Vec<GoogleCloudApigeeV1Deployment>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDeploymentsResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers apps list organizations](OrganizationDeveloperAppListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDeveloperAppsResponse {
    /// List of developer apps and their credentials.
    pub app: Option<Vec<GoogleCloudApigeeV1DeveloperApp>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDeveloperAppsResponse {}


/// Response for ListDeveloperSubscriptions.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers subscriptions list organizations](OrganizationDeveloperSubscriptionListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListDeveloperSubscriptionsResponse {
    /// List of all subscriptions.
    #[serde(rename="developerSubscriptions")]
    pub developer_subscriptions: Option<Vec<GoogleCloudApigeeV1DeveloperSubscription>>,
    /// Value that can be sent as `startKey` to retrieve the next page of content. If this field is omitted, there are no subsequent pages.
    #[serde(rename="nextStartKey")]
    pub next_start_key: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListDeveloperSubscriptionsResponse {}


/// Response for ListEndpointAttachments method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [endpoint attachments list organizations](OrganizationEndpointAttachmentListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListEndpointAttachmentsResponse {
    /// Endpoint attachments in the specified organization.
    #[serde(rename="endpointAttachments")]
    pub endpoint_attachments: Option<Vec<GoogleCloudApigeeV1EndpointAttachment>>,
    /// Page token that you can include in an `ListEndpointAttachments` request to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListEndpointAttachmentsResponse {}


/// Response for ListEnvironmentGroupAttachments.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [envgroups attachments list organizations](OrganizationEnvgroupAttachmentListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListEnvironmentGroupAttachmentsResponse {
    /// EnvironmentGroupAttachments for the specified environment group.
    #[serde(rename="environmentGroupAttachments")]
    pub environment_group_attachments: Option<Vec<GoogleCloudApigeeV1EnvironmentGroupAttachment>>,
    /// Page token that you can include in a ListEnvironmentGroupAttachments request to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListEnvironmentGroupAttachmentsResponse {}


/// Response for ListEnvironmentGroups.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [envgroups list organizations](OrganizationEnvgroupListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListEnvironmentGroupsResponse {
    /// EnvironmentGroups in the specified organization.
    #[serde(rename="environmentGroups")]
    pub environment_groups: Option<Vec<GoogleCloudApigeeV1EnvironmentGroup>>,
    /// Page token that you can include in a ListEnvironmentGroups request to retrieve the next page. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListEnvironmentGroupsResponse {}


/// Response for ListEnvironmentResources
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments resourcefiles list organizations](OrganizationEnvironmentResourcefileListCall) (response)
/// * [environments resourcefiles list environment resources organizations](OrganizationEnvironmentResourcefileListEnvironmentResourceCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListEnvironmentResourcesResponse {
    /// List of resources files.
    #[serde(rename="resourceFile")]
    pub resource_file: Option<Vec<GoogleCloudApigeeV1ResourceFile>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListEnvironmentResourcesResponse {}


/// The response for ListExports
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments analytics exports list organizations](OrganizationEnvironmentAnalyticExportListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListExportsResponse {
    /// Details of the export jobs.
    pub exports: Option<Vec<GoogleCloudApigeeV1Export>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListExportsResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [issuers list hybrid](HybridIssuerListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListHybridIssuersResponse {
    /// Lists of hybrid services and its trusted issuer email ids.
    pub issuers: Option<Vec<GoogleCloudApigeeV1ServiceIssuersMapping>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListHybridIssuersResponse {}


/// Response for ListInstanceAttachments.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances attachments list organizations](OrganizationInstanceAttachmentListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListInstanceAttachmentsResponse {
    /// Attachments for the instance.
    pub attachments: Option<Vec<GoogleCloudApigeeV1InstanceAttachment>>,
    /// Page token that you can include in a ListInstanceAttachments request to retrieve the next page of content. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListInstanceAttachmentsResponse {}


/// Response for ListInstances.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances list organizations](OrganizationInstanceListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListInstancesResponse {
    /// Instances in the specified organization.
    pub instances: Option<Vec<GoogleCloudApigeeV1Instance>>,
    /// Page token that you can include in a ListInstance request to retrieve the next page of content. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListInstancesResponse {}


/// Response for ListNatAddresses.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances nat addresses list organizations](OrganizationInstanceNatAddresseListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListNatAddressesResponse {
    /// List of NAT Addresses for the instance.
    #[serde(rename="natAddresses")]
    pub nat_addresses: Option<Vec<GoogleCloudApigeeV1NatAddress>>,
    /// Page token that you can include in a ListNatAddresses request to retrieve the next page of content. If omitted, no subsequent pages exist.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListNatAddressesResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [developers list organizations](OrganizationDeveloperListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListOfDevelopersResponse {
    /// List of developers.
    pub developer: Option<Vec<GoogleCloudApigeeV1Developer>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListOfDevelopersResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [list organizations](OrganizationListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListOrganizationsResponse {
    /// List of Apigee organizations and associated GCP projects.
    pub organizations: Option<Vec<GoogleCloudApigeeV1OrganizationProjectMapping>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListOrganizationsResponse {}


/// Response for ListRatePlans.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts rateplans list organizations](OrganizationApiproductRateplanListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListRatePlansResponse {
    /// Value that can be sent as `startKey` to retrieve the next page of content. If this field is omitted, there are no subsequent pages.
    #[serde(rename="nextStartKey")]
    pub next_start_key: Option<String>,
    /// List of rate plans in an organization.
    #[serde(rename="ratePlans")]
    pub rate_plans: Option<Vec<GoogleCloudApigeeV1RatePlan>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListRatePlansResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sharedflows list organizations](OrganizationSharedflowListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListSharedFlowsResponse {
    /// no description provided
    #[serde(rename="sharedFlows")]
    pub shared_flows: Option<Vec<GoogleCloudApigeeV1SharedFlow>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListSharedFlowsResponse {}


/// Response for ListTraceConfigOverrides.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments trace config overrides list organizations](OrganizationEnvironmentTraceConfigOverrideListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ListTraceConfigOverridesResponse {
    /// Token value that can be passed as `page_token` to retrieve the next page of content.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
    /// List all trace configuration overrides in an environment.
    #[serde(rename="traceConfigOverrides")]
    pub trace_config_overrides: Option<Vec<GoogleCloudApigeeV1TraceConfigOverride>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ListTraceConfigOverridesResponse {}


/// Encapsulates additional information about query execution.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Metadata {
    /// List of error messages as strings.
    pub errors: Option<Vec<String>>,
    /// List of additional information such as data source, if result was truncated. For example: ``` "notices": [ "Source:Postgres", "PG Host:uappg0rw.e2e.apigeeks.net", "query served by:4b64601e-40de-4eb1-bfb9-eeee7ac929ed", "Table used: edge.api.uapgroup2.agg_api" ]```
    pub notices: Option<Vec<String>>,
}

impl client::Part for GoogleCloudApigeeV1Metadata {}


/// Encapsulates the metric data point. For example: ```{ "name": "sum(message_count)", "values" : [ { "timestamp": 1549004400000, "value": "39.0" }, { "timestamp" : 1548997200000, "value" : "0.0" } ] }``` or ```{ "name": "sum(message_count)", "values" : ["39.0"] }```
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Metric {
    /// Metric name.
    pub name: Option<String>,
    /// List of metric values. Possible value formats include: `"values":["39.0"]` or `"values":[ { "value": "39.0", "timestamp": 1232434354} ]`
    pub values: Option<Vec<String>>,
}

impl client::Part for GoogleCloudApigeeV1Metric {}


/// Configuration for the Monetization add-on.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1MonetizationConfig {
    /// Flag that specifies whether the Monetization add-on is enabled.
    pub enabled: Option<bool>,
}

impl client::Part for GoogleCloudApigeeV1MonetizationConfig {}


/// Apigee NAT(network address translation) address. A NAT address is a static external IP address used for Internet egress traffic.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances nat addresses create organizations](OrganizationInstanceNatAddresseCreateCall) (request)
/// * [instances nat addresses get organizations](OrganizationInstanceNatAddresseGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1NatAddress {
    /// Output only. The static IPV4 address.
    #[serde(rename="ipAddress")]
    pub ip_address: Option<String>,
    /// Required. Resource ID of the NAT address.
    pub name: Option<String>,
    /// Output only. State of the nat address.
    pub state: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1NatAddress {}
impl client::ResponseResult for GoogleCloudApigeeV1NatAddress {}


/// Represents the pairing of REST resource path and the actions (verbs) allowed on the resource path.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Operation {
    /// methods refers to the REST verbs as in https://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html. When none specified, all verb types are allowed.
    pub methods: Option<Vec<String>>,
    /// Required. REST resource path associated with the API proxy or remote service.
    pub resource: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Operation {}


/// Binds the resources in an API proxy or remote service with the allowed REST methods and associated quota enforcement.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OperationConfig {
    /// Required. Name of the API proxy or remote service with which the resources, methods, and quota are associated.
    #[serde(rename="apiSource")]
    pub api_source: Option<String>,
    /// Custom attributes associated with the operation.
    pub attributes: Option<Vec<GoogleCloudApigeeV1Attribute>>,
    /// List of resource/method pairs for the API proxy or remote service to which quota will applied. **Note**: Currently, you can specify only a single resource/method pair. The call will fail if more than one resource/method pair is provided.
    pub operations: Option<Vec<GoogleCloudApigeeV1Operation>>,
    /// Quota parameters to be enforced for the resources, methods, and API source combination. If none are specified, quota enforcement will not be done.
    pub quota: Option<GoogleCloudApigeeV1Quota>,
}

impl client::Part for GoogleCloudApigeeV1OperationConfig {}


/// List of operation configuration details associated with Apigee API proxies or remote services. Remote services are non-Apigee proxies, such as Istio-Envoy.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OperationGroup {
    /// Flag that specifes whether the configuration is for Apigee API proxy or a remote service. Valid values include `proxy` or `remoteservice`. Defaults to `proxy`. Set to `proxy` when Apigee API proxies are associated with the API product. Set to `remoteservice` when non-Apigee proxies like Istio-Envoy are associated with the API product.
    #[serde(rename="operationConfigType")]
    pub operation_config_type: Option<String>,
    /// Required. List of operation configurations for either Apigee API proxies or other remote services that are associated with this API product.
    #[serde(rename="operationConfigs")]
    pub operation_configs: Option<Vec<GoogleCloudApigeeV1OperationConfig>>,
}

impl client::Part for GoogleCloudApigeeV1OperationGroup {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments optimized stats get organizations](OrganizationEnvironmentOptimizedStatGetCall) (response)
/// * [optimized host stats get organizations](OrganizationOptimizedHostStatGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OptimizedStats {
    /// Wraps the `stats` response for JavaScript Optimized Scenario with a response key. For example: ```{ "Response": { "TimeUnit": [], "metaData": { "errors": [], "notices": [ "Source:Postgres", "Table used: edge.api.aaxgroup001.agg_api", "PG Host:ruappg08-ro.production.apigeeks.net", "query served by:80c4ebca-6a10-4a2e-8faf-c60c1ee306ca" ] }, "resultTruncated": false, "stats": { "data": [ { "identifier": { "names": [ "apiproxy" ], "values": [ "sirjee" ] }, "metric": [ { "env": "prod", "name": "sum(message_count)", "values": [ 36.0 ] }, { "env": "prod", "name": "sum(is_error)", "values": [ 36.0 ] } ] } ] } } }```
    #[serde(rename="Response")]
    pub response: Option<GoogleCloudApigeeV1OptimizedStatsResponse>,
}

impl client::ResponseResult for GoogleCloudApigeeV1OptimizedStats {}


/// Encapsulates a data node as represented below: ``` { "identifier": { "names": [ "apiproxy" ], "values": [ "sirjee" ] }, "metric": [ { "env": "prod", "name": "sum(message_count)", "values": [ 36.0 ] } ] }``` or ``` { "env": "prod", "name": "sum(message_count)", "values": [ 36.0 ] }``` Depending on whether a dimension is present in the query or not the data node type can be a simple metric value or dimension identifier with list of metrics.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OptimizedStatsNode {
    /// no description provided
    pub data: Option<Vec<String>>,
}

impl client::Part for GoogleCloudApigeeV1OptimizedStatsNode {}


/// Encapsulates a response format for JavaScript Optimized Scenario.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OptimizedStatsResponse {
    /// List of time unit values. Time unit refers to an epoch timestamp value.
    #[serde(rename="TimeUnit")]
    pub time_unit: Option<Vec<String>>,
    /// Metadata information about the query executed.
    #[serde(rename="metaData")]
    pub meta_data: Option<GoogleCloudApigeeV1Metadata>,
    /// Boolean flag that indicates whether the results were truncated based on the limit parameter.
    #[serde(rename="resultTruncated")]
    pub result_truncated: Option<bool>,
    /// `stats` results.
    pub stats: Option<GoogleCloudApigeeV1OptimizedStatsNode>,
}

impl client::Part for GoogleCloudApigeeV1OptimizedStatsResponse {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [create organizations](OrganizationCreateCall) (request)
/// * [get organizations](OrganizationGetCall) (response)
/// * [update organizations](OrganizationUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Organization {
    /// Addon configurations of the Apigee organization.
    #[serde(rename="addonsConfig")]
    pub addons_config: Option<GoogleCloudApigeeV1AddonsConfig>,
    /// Required. Primary GCP region for analytics data storage. For valid values, see [Create an Apigee organization](https://cloud.google.com/apigee/docs/api-platform/get-started/create-org).
    #[serde(rename="analyticsRegion")]
    pub analytics_region: Option<String>,
    /// Not used by Apigee.
    pub attributes: Option<Vec<String>>,
    /// Compute Engine network used for Service Networking to be peered with Apigee runtime instances. See [Getting started with the Service Networking API](https://cloud.google.com/service-infrastructure/docs/service-networking/getting-started). Valid only when [RuntimeType](#RuntimeType) is set to `CLOUD`. The value must be set before the creation of a runtime instance and can be updated only when there are no runtime instances. For example: `default`. Apigee also supports shared VPC (that is, the host network project is not the same as the one that is peering with Apigee). See [Shared VPC overview](https://cloud.google.com/vpc/docs/shared-vpc). To use a shared VPC network, use the following format: `projects/{host-project-id}/{region}/networks/{network-name}`. For example: `projects/my-sharedvpc-host/global/networks/mynetwork` **Note:** Not supported for Apigee hybrid.
    #[serde(rename="authorizedNetwork")]
    pub authorized_network: Option<String>,
    /// Billing type of the Apigee organization. See [Apigee pricing](https://cloud.google.com/apigee/pricing).
    #[serde(rename="billingType")]
    pub billing_type: Option<String>,
    /// Output only. Base64-encoded public certificate for the root CA of the Apigee organization. Valid only when [RuntimeType](#RuntimeType) is `CLOUD`.
    #[serde(rename="caCertificate")]
    pub ca_certificate: Option<String>,
    /// Output only. Time that the Apigee organization was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Not used by Apigee.
    #[serde(rename="customerName")]
    pub customer_name: Option<String>,
    /// Description of the Apigee organization.
    pub description: Option<String>,
    /// Display name for the Apigee organization. Unused, but reserved for future use.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Output only. List of environments in the Apigee organization.
    pub environments: Option<Vec<String>>,
    /// Output only. Time that the Apigee organization is scheduled for deletion.
    #[serde(rename="expiresAt")]
    pub expires_at: Option<String>,
    /// Output only. Time that the Apigee organization was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Output only. Name of the Apigee organization.
    pub name: Option<String>,
    /// Configuration for the Portals settings.
    #[serde(rename="portalDisabled")]
    pub portal_disabled: Option<bool>,
    /// Output only. Project ID associated with the Apigee organization.
    #[serde(rename="projectId")]
    pub project_id: Option<String>,
    /// Properties defined in the Apigee organization profile.
    pub properties: Option<GoogleCloudApigeeV1Properties>,
    /// Cloud KMS key name used for encrypting the data that is stored and replicated across runtime instances. Update is not allowed after the organization is created. Required when [RuntimeType](#RuntimeType) is `CLOUD`. If not specified when [RuntimeType](#RuntimeType) is `TRIAL`, a Google-Managed encryption key will be used. For example: "projects/foo/locations/us/keyRings/bar/cryptoKeys/baz". **Note:** Not supported for Apigee hybrid.
    #[serde(rename="runtimeDatabaseEncryptionKeyName")]
    pub runtime_database_encryption_key_name: Option<String>,
    /// Required. Runtime type of the Apigee organization based on the Apigee subscription purchased.
    #[serde(rename="runtimeType")]
    pub runtime_type: Option<String>,
    /// Output only. State of the organization. Values other than ACTIVE means the resource is not ready to use.
    pub state: Option<String>,
    /// Output only. DEPRECATED: This will eventually be replaced by BillingType. Subscription type of the Apigee organization. Valid values include trial (free, limited, and for evaluation purposes only) or paid (full subscription has been purchased). See [Apigee pricing](https://cloud.google.com/apigee/pricing/).
    #[serde(rename="subscriptionType")]
    pub subscription_type: Option<String>,
    /// Not used by Apigee.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Organization {}
impl client::ResponseResult for GoogleCloudApigeeV1Organization {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1OrganizationProjectMapping {
    /// Name of the Apigee organization.
    pub organization: Option<String>,
    /// GCP project associated with the Apigee organization
    #[serde(rename="projectId")]
    pub project_id: Option<String>,
    /// DEPRECATED: Use `project_id`. An Apigee Organization is mapped to a single project.
    #[serde(rename="projectIds")]
    pub project_ids: Option<Vec<String>>,
}

impl client::Part for GoogleCloudApigeeV1OrganizationProjectMapping {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1PodStatus {
    /// Version of the application running in the pod.
    #[serde(rename="appVersion")]
    pub app_version: Option<String>,
    /// Status of the deployment. Valid values include: - `deployed`: Successful. - `error` : Failed. - `pending` : Pod has not yet reported on the deployment.
    #[serde(rename="deploymentStatus")]
    pub deployment_status: Option<String>,
    /// Time the deployment status was reported in milliseconds since epoch.
    #[serde(rename="deploymentStatusTime")]
    pub deployment_status_time: Option<String>,
    /// Time the proxy was deployed in milliseconds since epoch.
    #[serde(rename="deploymentTime")]
    pub deployment_time: Option<String>,
    /// Name of the pod which is reporting the status.
    #[serde(rename="podName")]
    pub pod_name: Option<String>,
    /// Overall status of the pod (not this specific deployment). Valid values include: - `active`: Up to date. - `stale` : Recently out of date. Pods that have not reported status in a long time are excluded from the output.
    #[serde(rename="podStatus")]
    pub pod_status: Option<String>,
    /// Time the pod status was reported in milliseconds since epoch.
    #[serde(rename="podStatusTime")]
    pub pod_status_time: Option<String>,
    /// Code associated with the deployment status.
    #[serde(rename="statusCode")]
    pub status_code: Option<String>,
    /// Human-readable message associated with the status code.
    #[serde(rename="statusCodeDetails")]
    pub status_code_details: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1PodStatus {}


/// Point is a group of information collected by runtime plane at critical points of the message flow of the processed API request. This is a list of supported point IDs, categorized to three major buckets. For each category, debug points that we are currently supporting are listed below: - Flow status debug points: StateChange FlowInfo Condition Execution DebugMask Error - Flow control debug points: FlowCallout Paused Resumed FlowReturn BreakFlow Error - Runtime debug points: ScriptExecutor FlowCalloutStepDefinition CustomTarget StepDefinition Oauth2ServicePoint RaiseFault NodeJS The detail information of the given debug point is stored in a list of results.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Point {
    /// Name of a step in the transaction.
    pub id: Option<String>,
    /// List of results extracted from a given debug point.
    pub results: Option<Vec<GoogleCloudApigeeV1Result>>,
}

impl client::Part for GoogleCloudApigeeV1Point {}


/// Message for compatibility with legacy Edge specification for Java Properties object in JSON.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Properties {
    /// List of all properties in the object
    pub property: Option<Vec<GoogleCloudApigeeV1Property>>,
}

impl client::Part for GoogleCloudApigeeV1Properties {}


/// A single property entry in the Properties message.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Property {
    /// The property key
    pub name: Option<String>,
    /// The property value
    pub value: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Property {}


/// Request for ProvisionOrganization.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [provision organization projects](ProjectProvisionOrganizationCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ProvisionOrganizationRequest {
    /// Primary Cloud Platform region for analytics data storage. For valid values, see [Create an organization](https://cloud.google.com/apigee/docs/hybrid/latest/precog-provision). Defaults to `us-west1`.
    #[serde(rename="analyticsRegion")]
    pub analytics_region: Option<String>,
    /// Name of the customer project's VPC network. If provided, the network needs to be peered through Service Networking. If none is provided, the organization will have access only to the public internet.
    #[serde(rename="authorizedNetwork")]
    pub authorized_network: Option<String>,
    /// Cloud Platform location for the runtime instance. Defaults to zone `us-west1-a`. If a region is provided, `EVAL` organizations will use the region for automatically selecting a zone for the runtime instance.
    #[serde(rename="runtimeLocation")]
    pub runtime_location: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1ProvisionOrganizationRequest {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments queries create organizations](OrganizationEnvironmentQueryCreateCall) (request)
/// * [host queries create organizations](OrganizationHostQueryCreateCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Query {
    /// Delimiter used in the CSV file, if `outputFormat` is set to `csv`. Defaults to the `,` (comma) character. Supported delimiter characters include comma (`,`), pipe (`|`), and tab (`\t`).
    #[serde(rename="csvDelimiter")]
    pub csv_delimiter: Option<String>,
    /// A list of dimensions. https://docs.apigee.com/api-platform/analytics/analytics-reference#dimensions
    pub dimensions: Option<Vec<String>>,
    /// Hostname needs to be specified if query intends to run at host level. This field is only allowed when query is submitted by CreateHostAsyncQuery where analytics data will be grouped by organization and hostname.
    #[serde(rename="envgroupHostname")]
    pub envgroup_hostname: Option<String>,
    /// Boolean expression that can be used to filter data. Filter expressions can be combined using AND/OR terms and should be fully parenthesized to avoid ambiguity. See Analytics metrics, dimensions, and filters reference https://docs.apigee.com/api-platform/analytics/analytics-reference for more information on the fields available to filter on. For more information on the tokens that you use to build filter expressions, see Filter expression syntax. https://docs.apigee.com/api-platform/analytics/asynch-reports-api#filter-expression-syntax
    pub filter: Option<String>,
    /// Time unit used to group the result set. Valid values include: second, minute, hour, day, week, or month. If a query includes groupByTimeUnit, then the result is an aggregation based on the specified time unit and the resultant timestamp does not include milliseconds precision. If a query omits groupByTimeUnit, then the resultant timestamp includes milliseconds precision.
    #[serde(rename="groupByTimeUnit")]
    pub group_by_time_unit: Option<String>,
    /// Maximum number of rows that can be returned in the result.
    pub limit: Option<i32>,
    /// A list of Metrics.
    pub metrics: Option<Vec<GoogleCloudApigeeV1QueryMetric>>,
    /// Asynchronous Query Name.
    pub name: Option<String>,
    /// Valid values include: `csv` or `json`. Defaults to `json`. Note: Configure the delimiter for CSV output using the csvDelimiter property.
    #[serde(rename="outputFormat")]
    pub output_format: Option<String>,
    /// Asynchronous Report ID.
    #[serde(rename="reportDefinitionId")]
    pub report_definition_id: Option<String>,
    /// Required. Time range for the query. Can use the following predefined strings to specify the time range: `last60minutes` `last24hours` `last7days` Or, specify the timeRange as a structure describing start and end timestamps in the ISO format: yyyy-mm-ddThh:mm:ssZ. Example: "timeRange": { "start": "2018-07-29T00:13:00Z", "end": "2018-08-01T00:18:00Z" }
    #[serde(rename="timeRange")]
    pub time_range: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Query {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1QueryMetadata {
    /// Dimensions of the AsyncQuery.
    pub dimensions: Option<Vec<String>>,
    /// End timestamp of the query range.
    #[serde(rename="endTimestamp")]
    pub end_timestamp: Option<String>,
    /// Metrics of the AsyncQuery. Example: ["name:message_count,func:sum,alias:sum_message_count"]
    pub metrics: Option<Vec<String>>,
    /// Output format.
    #[serde(rename="outputFormat")]
    pub output_format: Option<String>,
    /// Start timestamp of the query range.
    #[serde(rename="startTimestamp")]
    pub start_timestamp: Option<String>,
    /// Query GroupBy time unit.
    #[serde(rename="timeUnit")]
    pub time_unit: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1QueryMetadata {}


/// More info about Metric: https://docs.apigee.com/api-platform/analytics/analytics-reference#metrics
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1QueryMetric {
    /// Alias for the metric. Alias will be used to replace metric name in query results.
    pub alias: Option<String>,
    /// Aggregation function: avg, min, max, or sum.
    pub function: Option<String>,
    /// Required. Metric name.
    pub name: Option<String>,
    /// One of `+`, `-`, `/`, `%`, `*`.
    pub operator: Option<String>,
    /// Operand value should be provided when operator is set.
    pub value: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1QueryMetric {}


/// Quota contains the essential parameters needed that can be applied on the resources, methods, API source combination associated with this API product. While Quota is optional, setting it prevents requests from exceeding the provisioned parameters.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Quota {
    /// Required. Time interval over which the number of request messages is calculated.
    pub interval: Option<String>,
    /// Required. Upper limit allowed for the time interval and time unit specified. Requests exceeding this limit will be rejected.
    pub limit: Option<String>,
    /// Time unit defined for the `interval`. Valid values include `minute`, `hour`, `day`, or `month`. If `limit` and `interval` are valid, the default value is `hour`; otherwise, the default is null.
    #[serde(rename="timeUnit")]
    pub time_unit: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Quota {}


/// Rate plan details.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [apiproducts rateplans create organizations](OrganizationApiproductRateplanCreateCall) (request|response)
/// * [apiproducts rateplans delete organizations](OrganizationApiproductRateplanDeleteCall) (response)
/// * [apiproducts rateplans get organizations](OrganizationApiproductRateplanGetCall) (response)
/// * [apiproducts rateplans update organizations](OrganizationApiproductRateplanUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RatePlan {
    /// Name of the API product that the rate plan is associated with.
    pub apiproduct: Option<String>,
    /// Frequency at which the customer will be billed.
    #[serde(rename="billingPeriod")]
    pub billing_period: Option<String>,
    /// API call volume ranges and the fees charged when the total number of API calls is within a given range. The method used to calculate the final fee depends on the selected pricing model. For example, if the pricing model is `STAIRSTEP` and the ranges are defined as follows: ``` { "start": 1, "end": 100, "fee": 75 }, { "start": 101, "end": 200, "fee": 100 }, } ``` Then the following fees would be charged based on the total number of API calls (assuming the currency selected is `USD`): * 1 call costs $75 * 50 calls cost $75 * 150 calls cost $100 The number of API calls cannot exceed 200.
    #[serde(rename="consumptionPricingRates")]
    pub consumption_pricing_rates: Option<Vec<GoogleCloudApigeeV1RateRange>>,
    /// Pricing model used for consumption-based charges.
    #[serde(rename="consumptionPricingType")]
    pub consumption_pricing_type: Option<String>,
    /// Output only. Time that the rate plan was created in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Currency to be used for billing. Consists of a three-letter code as defined by the [ISO 4217](https://en.wikipedia.org/wiki/ISO_4217) standard.
    #[serde(rename="currencyCode")]
    pub currency_code: Option<String>,
    /// Description of the rate plan.
    pub description: Option<String>,
    /// Display name of the rate plan.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// Time when the rate plan will expire in milliseconds since epoch. Set to 0 or `null` to indicate that the rate plan should never expire.
    #[serde(rename="endTime")]
    pub end_time: Option<String>,
    /// Frequency at which the fixed fee is charged.
    #[serde(rename="fixedFeeFrequency")]
    pub fixed_fee_frequency: Option<i32>,
    /// Fixed amount that is charged at a defined interval and billed in advance of use of the API product. The fee will be prorated for the first billing period.
    #[serde(rename="fixedRecurringFee")]
    pub fixed_recurring_fee: Option<GoogleTypeMoney>,
    /// Output only. Time the rate plan was last modified in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// Output only. Name of the rate plan.
    pub name: Option<String>,
    /// DEPRECATED: This field is no longer supported and will eventually be removed when Apigee Hybrid 1.5/1.6 is no longer supported. Instead, use the `billingType` field inside `DeveloperMonetizationConfig` resource. Flag that specifies the billing account type, prepaid or postpaid.
    #[serde(rename="paymentFundingModel")]
    pub payment_funding_model: Option<String>,
    /// Details of the revenue sharing model.
    #[serde(rename="revenueShareRates")]
    pub revenue_share_rates: Option<Vec<GoogleCloudApigeeV1RevenueShareRange>>,
    /// Method used to calculate the revenue that is shared with developers.
    #[serde(rename="revenueShareType")]
    pub revenue_share_type: Option<String>,
    /// Initial, one-time fee paid when purchasing the API product.
    #[serde(rename="setupFee")]
    pub setup_fee: Option<GoogleTypeMoney>,
    /// Time when the rate plan becomes active in milliseconds since epoch.
    #[serde(rename="startTime")]
    pub start_time: Option<String>,
    /// Current state of the rate plan (draft or published).
    pub state: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1RatePlan {}
impl client::ResponseResult for GoogleCloudApigeeV1RatePlan {}


/// API call volume range and the fees charged when the total number of API calls is within the range.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RateRange {
    /// Ending value of the range. Set to 0 or `null` for the last range of values.
    pub end: Option<String>,
    /// Fee to charge when total number of API calls falls within this range.
    pub fee: Option<GoogleTypeMoney>,
    /// Starting value of the range. Set to 0 or `null` for the initial range of values.
    pub start: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1RateRange {}


/// A Reference configuration. References must refer to a keystore that also exists in the parent environment.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments references create organizations](OrganizationEnvironmentReferenceCreateCall) (request|response)
/// * [environments references delete organizations](OrganizationEnvironmentReferenceDeleteCall) (response)
/// * [environments references get organizations](OrganizationEnvironmentReferenceGetCall) (response)
/// * [environments references update organizations](OrganizationEnvironmentReferenceUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Reference {
    /// Optional. A human-readable description of this reference.
    pub description: Option<String>,
    /// Required. The resource id of this reference. Values must match the regular expression [\w\s\-.]+.
    pub name: Option<String>,
    /// Required. The id of the resource to which this reference refers. Must be the id of a resource that exists in the parent environment and is of the given resource_type.
    pub refers: Option<String>,
    /// The type of resource referred to by this reference. Valid values are 'KeyStore' or 'TrustStore'.
    #[serde(rename="resourceType")]
    pub resource_type: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Reference {}
impl client::ResponseResult for GoogleCloudApigeeV1Reference {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ReferenceConfig {
    /// Name of the reference in the following format: `organizations/{org}/environments/{env}/references/{reference}`
    pub name: Option<String>,
    /// Name of the referenced resource in the following format: `organizations/{org}/environments/{env}/keystores/{keystore}` Only references to keystore resources are supported.
    #[serde(rename="resourceName")]
    pub resource_name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ReferenceConfig {}


/// Request for ReportInstanceStatus.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances report status organizations](OrganizationInstanceReportStatuCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ReportInstanceStatusRequest {
    /// A unique ID for the instance which is guaranteed to be unique in case the user installs multiple hybrid runtimes with the same instance ID.
    #[serde(rename="instanceUid")]
    pub instance_uid: Option<String>,
    /// The time the report was generated in the runtime. Used to prevent an old status from overwriting a newer one. An instance should space out it's status reports so that clock skew does not play a factor.
    #[serde(rename="reportTime")]
    pub report_time: Option<String>,
    /// Status for config resources
    pub resources: Option<Vec<GoogleCloudApigeeV1ResourceStatus>>,
}

impl client::RequestValue for GoogleCloudApigeeV1ReportInstanceStatusRequest {}


/// Placeholder for future enhancements to status reporting protocol
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [instances report status organizations](OrganizationInstanceReportStatuCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ReportInstanceStatusResponse { _never_set: Option<bool> }

impl client::ResponseResult for GoogleCloudApigeeV1ReportInstanceStatusResponse {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ReportProperty {
    /// name of the property
    pub property: Option<String>,
    /// property values
    pub value: Option<Vec<GoogleCloudApigeeV1Attribute>>,
}

impl client::Part for GoogleCloudApigeeV1ReportProperty {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ResourceConfig {
    /// Location of the resource as a URI.
    pub location: Option<String>,
    /// Resource name in the following format: `organizations/{org}/environments/{env}/resourcefiles/{type}/{file}/revisions/{rev}` Only environment-scoped resource files are supported.
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ResourceConfig {}


/// Metadata about a resource file.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments resourcefiles create organizations](OrganizationEnvironmentResourcefileCreateCall) (response)
/// * [environments resourcefiles delete organizations](OrganizationEnvironmentResourcefileDeleteCall) (response)
/// * [environments resourcefiles update organizations](OrganizationEnvironmentResourcefileUpdateCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ResourceFile {
    /// ID of the resource file.
    pub name: Option<String>,
    /// Resource file type. {{ resource_file_type }}
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1ResourceFile {}


/// List of resource files.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ResourceFiles {
    /// List of resource files.
    #[serde(rename="resourceFile")]
    pub resource_file: Option<Vec<GoogleCloudApigeeV1ResourceFile>>,
}

impl client::Part for GoogleCloudApigeeV1ResourceFiles {}


/// The status of a resource loaded in the runtime.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ResourceStatus {
    /// The resource name. Currently only two resources are supported: EnvironmentGroup - organizations/{org}/envgroups/{envgroup} EnvironmentConfig - organizations/{org}/environments/{environment}/deployedConfig
    pub resource: Option<String>,
    /// Revisions of the resource currently deployed in the instance.
    pub revisions: Option<Vec<GoogleCloudApigeeV1RevisionStatus>>,
    /// The total number of replicas that should have this resource.
    #[serde(rename="totalReplicas")]
    pub total_replicas: Option<i32>,
    /// The uid of the resource. In the unexpected case that the instance has multiple uids for the same name, they should be reported under separate ResourceStatuses.
    pub uid: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ResourceStatus {}


/// Result is short for "action result", could be different types identified by "action_result" field. Supported types: 1. DebugInfo : generic debug info collected by runtime recorded as a list of properties. For example, the contents could be virtual host info, state change result, or execution metadata. Required fields : properties, timestamp 2. RequestMessage: information of a http request. Contains headers, request URI and http methods type.Required fields : headers, uri, verb 3. ResponseMessage: information of a http response. Contains headers, reason phrase and http status code. Required fields : headers, reasonPhrase, statusCode 4. ErrorMessage: information of a http error message. Contains detail error message, reason phrase and status code. Required fields : content, headers, reasonPhrase, statusCode 5. VariableAccess: a list of variable access actions, can be Get, Set and Remove. Required fields : accessList
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Result {
    /// Type of the action result. Can be one of the five: DebugInfo, RequestMessage, ResponseMessage, ErrorMessage, VariableAccess
    #[serde(rename="ActionResult")]
    pub action_result: Option<String>,
    /// A list of variable access actions agaist the api proxy. Supported values: Get, Set, Remove.
    #[serde(rename="accessList")]
    pub access_list: Option<Vec<GoogleCloudApigeeV1Access>>,
    /// Error message content. for example, "content" : "{\"fault\":{\"faultstring\":\"API timed out\",\"detail\":{\"errorcode\":\"flow.APITimedOut\"}}}"
    pub content: Option<String>,
    /// A list of HTTP headers. for example, '"headers" : [ { "name" : "Content-Length", "value" : "83" }, { "name" : "Content-Type", "value" : "application/json" } ]'
    pub headers: Option<Vec<GoogleCloudApigeeV1Property>>,
    /// Name value pairs used for DebugInfo ActionResult.
    pub properties: Option<GoogleCloudApigeeV1Properties>,
    /// HTTP response phrase
    #[serde(rename="reasonPhrase")]
    pub reason_phrase: Option<String>,
    /// HTTP response code
    #[serde(rename="statusCode")]
    pub status_code: Option<String>,
    /// Timestamp of when the result is recorded. Its format is dd-mm-yy hh:mm:ss:xxx. For example, `"timestamp" : "12-08-19 00:31:59:960"`
    pub timestamp: Option<String>,
    /// The relative path of the api proxy. for example, `"uRI" : "/iloveapis"`
    #[serde(rename="uRI")]
    pub u_ri: Option<String>,
    /// HTTP method verb
    pub verb: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Result {}


/// API call volume range and the percentage of revenue to share with the developer when the total number of API calls is within the range.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RevenueShareRange {
    /// Ending value of the range. Set to 0 or `null` for the last range of values.
    pub end: Option<String>,
    /// Percentage of the revenue to be shared with the developer. For example, to share 21 percent of the total revenue with the developer, set this value to 21. Specify a decimal number with a maximum of two digits following the decimal point.
    #[serde(rename="sharePercentage")]
    pub share_percentage: Option<f64>,
    /// Starting value of the range. Set to 0 or `null` for the initial range of values.
    pub start: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1RevenueShareRange {}


/// The status of a specific resource revision.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RevisionStatus {
    /// Errors reported when attempting to load this revision.
    pub errors: Option<Vec<GoogleCloudApigeeV1UpdateError>>,
    /// The json content of the resource revision. Large specs should be sent individually via the spec field to avoid hitting request size limits.
    #[serde(rename="jsonSpec")]
    pub json_spec: Option<String>,
    /// The number of replicas that have successfully loaded this revision.
    pub replicas: Option<i32>,
    /// The revision of the resource.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1RevisionStatus {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RoutingRule {
    /// URI path prefix used to route to the specified environment. May contain one or more wildcards. For example, path segments consisting of a single `*` character will match any string.
    pub basepath: Option<String>,
    /// The env group config revision_id when this rule was added or last updated. This value is set when the rule is created and will only update if the the environment_id changes. It is used to determine if the runtime is up to date with respect to this rule. This field is omitted from the IngressConfig unless the GetDeployedIngressConfig API is called with view=FULL.
    #[serde(rename="envGroupRevision")]
    pub env_group_revision: Option<String>,
    /// Name of an environment bound to the environment group in the following format: `organizations/{org}/environments/{env}`.
    pub environment: Option<String>,
    /// The resource name of the proxy revision that is receiving this basepath in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`. This field is omitted from the IngressConfig unless the GetDeployedIngressConfig API is called with view=FULL.
    pub receiver: Option<String>,
    /// The unix timestamp when this rule was updated. This is updated whenever env_group_revision is updated. This field is omitted from the IngressConfig unless the GetDeployedIngressConfig API is called with view=FULL.
    #[serde(rename="updateTime")]
    pub update_time: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1RoutingRule {}


/// Runtime configuration for the organization. Response for GetRuntimeConfig.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [get runtime config organizations](OrganizationGetRuntimeConfigCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RuntimeConfig {
    /// Cloud Storage bucket used for uploading Analytics records.
    #[serde(rename="analyticsBucket")]
    pub analytics_bucket: Option<String>,
    /// Name of the resource in the following format: `organizations/{org}/runtimeConfig`.
    pub name: Option<String>,
    /// Output only. Tenant project ID associated with the Apigee organization. The tenant project is used to host Google-managed resources that are dedicated to this Apigee organization. Clients have limited access to resources within the tenant project used to support Apigee runtime instances. Access to the tenant project is managed using SetSyncAuthorization. It can be empty if the tenant project hasn't been created yet.
    #[serde(rename="tenantProjectId")]
    pub tenant_project_id: Option<String>,
    /// Cloud Storage bucket used for uploading Trace records.
    #[serde(rename="traceBucket")]
    pub trace_bucket: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1RuntimeConfig {}


/// NEXT ID: 8 RuntimeTraceConfig defines the configurations for distributed trace in an environment.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RuntimeTraceConfig {
    /// Endpoint of the exporter.
    pub endpoint: Option<String>,
    /// Exporter that is used to view the distributed trace captured using OpenCensus. An exporter sends traces to any backend that is capable of consuming them. Recorded spans can be exported by registered exporters.
    pub exporter: Option<String>,
    /// Name of the trace config in the following format: `organizations/{org}/environment/{env}/traceConfig`
    pub name: Option<String>,
    /// List of trace configuration overrides for spicific API proxies.
    pub overrides: Option<Vec<GoogleCloudApigeeV1RuntimeTraceConfigOverride>>,
    /// The timestamp that the revision was created or updated.
    #[serde(rename="revisionCreateTime")]
    pub revision_create_time: Option<String>,
    /// Revision number which can be used by the runtime to detect if the trace config has changed between two versions.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
    /// Trace configuration for all API proxies in an environment.
    #[serde(rename="samplingConfig")]
    pub sampling_config: Option<GoogleCloudApigeeV1RuntimeTraceSamplingConfig>,
}

impl client::Part for GoogleCloudApigeeV1RuntimeTraceConfig {}


/// NEXT ID: 7 Trace configuration override for a specific API proxy in an environment.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RuntimeTraceConfigOverride {
    /// Name of the API proxy that will have its trace configuration overridden following format: `organizations/{org}/apis/{api}`
    #[serde(rename="apiProxy")]
    pub api_proxy: Option<String>,
    /// Name of the trace config override in the following format: `organizations/{org}/environment/{env}/traceConfig/overrides/{override}`
    pub name: Option<String>,
    /// The timestamp that the revision was created or updated.
    #[serde(rename="revisionCreateTime")]
    pub revision_create_time: Option<String>,
    /// Revision number which can be used by the runtime to detect if the trace config override has changed between two versions.
    #[serde(rename="revisionId")]
    pub revision_id: Option<String>,
    /// Trace configuration override for a specific API proxy in an environment.
    #[serde(rename="samplingConfig")]
    pub sampling_config: Option<GoogleCloudApigeeV1RuntimeTraceSamplingConfig>,
    /// Unique ID for the configuration override. The ID will only change if the override is deleted and recreated. Corresponds to name's "override" field.
    pub uid: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1RuntimeTraceConfigOverride {}


/// NEXT ID: 3 RuntimeTraceSamplingConfig represents the detail settings of distributed tracing. Only the fields that are defined in the distributed trace configuration can be overridden using the distribute trace configuration override APIs.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1RuntimeTraceSamplingConfig {
    /// Sampler of distributed tracing. OFF is the default value.
    pub sampler: Option<String>,
    /// Field sampling rate. This value is only applicable when using the PROBABILITY sampler. The supported values are > 0 and <= 0.5.
    #[serde(rename="samplingRate")]
    pub sampling_rate: Option<f32>,
}

impl client::Part for GoogleCloudApigeeV1RuntimeTraceSamplingConfig {}


/// Response for Schema call
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments analytics admin get schemav2 organizations](OrganizationEnvironmentAnalyticAdminGetSchemav2Call) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Schema {
    /// List of schema fields grouped as dimensions.
    pub dimensions: Option<Vec<GoogleCloudApigeeV1SchemaSchemaElement>>,
    /// Additional metadata associated with schema. This is a legacy field and usually consists of an empty array of strings.
    pub meta: Option<Vec<String>>,
    /// List of schema fields grouped as dimensions that can be used with an aggregate function such as `sum`, `avg`, `min`, and `max`.
    pub metrics: Option<Vec<GoogleCloudApigeeV1SchemaSchemaElement>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1Schema {}


/// Message type for the schema element
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SchemaSchemaElement {
    /// Name of the field.
    pub name: Option<String>,
    /// Properties for the schema field. For example: { "createTime": "2016-02-26T10:23:09.592Z", "custom": "false", "type": "string" }
    pub properties: Option<GoogleCloudApigeeV1SchemaSchemaProperty>,
}

impl client::Part for GoogleCloudApigeeV1SchemaSchemaElement {}


/// Properties for the schema field.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SchemaSchemaProperty {
    /// Time the field was created in RFC3339 string form. For example: `2016-02-26T10:23:09.592Z`.
    #[serde(rename="createTime")]
    pub create_time: Option<String>,
    /// Flag that specifies whether the field is standard in the dataset or a custom field created by the customer. `true` indicates that it is a custom field.
    pub custom: Option<String>,
    /// Data type of the field.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1SchemaSchemaProperty {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1ServiceIssuersMapping {
    /// List of trusted issuer email ids.
    #[serde(rename="emailIds")]
    pub email_ids: Option<Vec<String>>,
    /// String indicating the Apigee service name.
    pub service: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1ServiceIssuersMapping {}


/// Session carries the debug session id and its creation time.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Session {
    /// The debug session ID.
    pub id: Option<String>,
    /// The first transaction creation timestamp in millisecond, recorded by UAP.
    #[serde(rename="timestampMs")]
    pub timestamp_ms: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1Session {}


/// Request for SetAddons.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [set addons organizations](OrganizationSetAddonCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SetAddonsRequest {
    /// Required. Add-on configurations.
    #[serde(rename="addonsConfig")]
    pub addons_config: Option<GoogleCloudApigeeV1AddonsConfig>,
}

impl client::RequestValue for GoogleCloudApigeeV1SetAddonsRequest {}


/// The metadata describing a shared flow
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sharedflows delete organizations](OrganizationSharedflowDeleteCall) (response)
/// * [sharedflows get organizations](OrganizationSharedflowGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SharedFlow {
    /// The id of the most recently created revision for this shared flow.
    #[serde(rename="latestRevisionId")]
    pub latest_revision_id: Option<String>,
    /// Metadata describing the shared flow.
    #[serde(rename="metaData")]
    pub meta_data: Option<GoogleCloudApigeeV1EntityMetadata>,
    /// The ID of the shared flow.
    pub name: Option<String>,
    /// A list of revisions of this shared flow.
    pub revision: Option<Vec<String>>,
}

impl client::ResponseResult for GoogleCloudApigeeV1SharedFlow {}


/// The metadata describing a shared flow revision.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [sharedflows revisions delete organizations](OrganizationSharedflowRevisionDeleteCall) (response)
/// * [sharedflows revisions update shared flow revision organizations](OrganizationSharedflowRevisionUpdateSharedFlowRevisionCall) (response)
/// * [sharedflows create organizations](OrganizationSharedflowCreateCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SharedFlowRevision {
    /// The version of the configuration schema to which this shared flow conforms. The only supported value currently is majorVersion 4 and minorVersion 0. This setting may be used in the future to enable evolution of the shared flow format.
    #[serde(rename="configurationVersion")]
    pub configuration_version: Option<GoogleCloudApigeeV1ConfigVersion>,
    /// A textual description of the shared flow revision.
    #[serde(rename="contextInfo")]
    pub context_info: Option<String>,
    /// Time at which this shared flow revision was created, in milliseconds since epoch.
    #[serde(rename="createdAt")]
    pub created_at: Option<String>,
    /// Description of the shared flow revision.
    pub description: Option<String>,
    /// The human readable name of this shared flow.
    #[serde(rename="displayName")]
    pub display_name: Option<String>,
    /// A Key-Value map of metadata about this shared flow revision.
    #[serde(rename="entityMetaDataAsProperties")]
    pub entity_meta_data_as_properties: Option<HashMap<String, String>>,
    /// Time at which this shared flow revision was most recently modified, in milliseconds since epoch.
    #[serde(rename="lastModifiedAt")]
    pub last_modified_at: Option<String>,
    /// The resource ID of the parent shared flow.
    pub name: Option<String>,
    /// A list of policy names included in this shared flow revision.
    pub policies: Option<Vec<String>>,
    /// The resource files included in this shared flow revision.
    #[serde(rename="resourceFiles")]
    pub resource_files: Option<GoogleCloudApigeeV1ResourceFiles>,
    /// A list of the resources included in this shared flow revision formatted as "{type}://{name}".
    pub resources: Option<Vec<String>>,
    /// The resource ID of this revision.
    pub revision: Option<String>,
    /// A list of the shared flow names included in this shared flow revision.
    #[serde(rename="sharedFlows")]
    pub shared_flows: Option<Vec<String>>,
    /// The string "Application"
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1SharedFlowRevision {}


/// Encapsulates a `stats` response.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments stats get organizations](OrganizationEnvironmentStatGetCall) (response)
/// * [host stats get organizations](OrganizationHostStatGetCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Stats {
    /// List of query results on the environment level.
    pub environments: Option<Vec<GoogleCloudApigeeV1StatsEnvironmentStats>>,
    /// List of query results grouped by host.
    pub hosts: Option<Vec<GoogleCloudApigeeV1StatsHostStats>>,
    /// Metadata information.
    #[serde(rename="metaData")]
    pub meta_data: Option<GoogleCloudApigeeV1Metadata>,
}

impl client::ResponseResult for GoogleCloudApigeeV1Stats {}


/// Encapsulates the environment wrapper: ``` "environments": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.52056245E8" ] } ], "name": "prod" } ]```
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1StatsEnvironmentStats {
    /// List of metrics grouped under dimensions.
    pub dimensions: Option<Vec<GoogleCloudApigeeV1DimensionMetric>>,
    /// In the final response, only one of the following fields will be present based on the dimensions provided. If no dimensions are provided, then only top-level metrics is provided. If dimensions are included, then there will be a top-level dimensions field under environments which will contain metrics values and the dimension name. Example: ``` "environments": [ { "dimensions": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.14049521E8" ] } ], "name": "nit_proxy" } ], "name": "prod" } ]``` or ```"environments": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.19026331E8" ] } ], "name": "prod" } ]``` List of metric values.
    pub metrics: Option<Vec<GoogleCloudApigeeV1Metric>>,
    /// Name of the environment.
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1StatsEnvironmentStats {}


/// Encapsulates the hostname wrapper: ``` "hosts": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.52056245E8" ] } ], "name": "example.com" } ]```
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1StatsHostStats {
    /// List of metrics grouped under dimensions.
    pub dimensions: Option<Vec<GoogleCloudApigeeV1DimensionMetric>>,
    /// In the final response, only one of the following fields will be present based on the dimensions provided. If no dimensions are provided, then only the top-level metrics are provided. If dimensions are included, then there will be a top-level dimensions field under hostnames which will contain metrics values and the dimension name. Example: ``` "hosts": [ { "dimensions": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.14049521E8" ] } ], "name": "nit_proxy" } ], "name": "example.com" } ]``` OR ```"hosts": [ { "metrics": [ { "name": "sum(message_count)", "values": [ "2.19026331E8" ] } ], "name": "example.com" } ]``` List of metric values.
    pub metrics: Option<Vec<GoogleCloudApigeeV1Metric>>,
    /// Hostname used in query.
    pub name: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1StatsHostStats {}


/// Pub/Sub subscription of an environment.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments subscribe organizations](OrganizationEnvironmentSubscribeCall) (response)
/// * [environments unsubscribe organizations](OrganizationEnvironmentUnsubscribeCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1Subscription {
    /// Full name of the Pub/Sub subcription. Use the following structure in your request: `subscription "projects/foo/subscription/bar"`
    pub name: Option<String>,
}

impl client::RequestValue for GoogleCloudApigeeV1Subscription {}
impl client::ResponseResult for GoogleCloudApigeeV1Subscription {}


/// There is no detailed description.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [get sync authorization organizations](OrganizationGetSyncAuthorizationCall) (response)
/// * [set sync authorization organizations](OrganizationSetSyncAuthorizationCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1SyncAuthorization {
    /// Entity tag (ETag) used for optimistic concurrency control as a way to help prevent simultaneous updates from overwriting each other. For example, when you call [getSyncAuthorization](organizations/getSyncAuthorization) an ETag is returned in the response. Pass that ETag when calling the [setSyncAuthorization](organizations/setSyncAuthorization) to ensure that you are updating the correct version. If you don't pass the ETag in the call to `setSyncAuthorization`, then the existing authorization is overwritten indiscriminately. **Note**: We strongly recommend that you use the ETag in the read-modify-write cycle to avoid race conditions.
    pub etag: Option<String>,
    /// Required. Array of service accounts to grant access to control plane resources, each specified using the following format: `serviceAccount:` service-account-name. The service-account-name is formatted like an email address. For example: `my-synchronizer-manager-service_account@my_project_id.iam.gserviceaccount.com` You might specify multiple service accounts, for example, if you have multiple environments and wish to assign a unique service account to each one. The service accounts must have **Apigee Synchronizer Manager** role. See also [Create service accounts](https://cloud.google.com/apigee/docs/hybrid/latest/sa-about#create-the-service-accounts).
    pub identities: Option<Vec<String>>,
}

impl client::RequestValue for GoogleCloudApigeeV1SyncAuthorization {}
impl client::ResponseResult for GoogleCloudApigeeV1SyncAuthorization {}


/// TargetServer configuration. TargetServers are used to decouple a proxy's TargetEndpoint HTTPTargetConnections from concrete URLs for backend services.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments targetservers create organizations](OrganizationEnvironmentTargetserverCreateCall) (request|response)
/// * [environments targetservers delete organizations](OrganizationEnvironmentTargetserverDeleteCall) (response)
/// * [environments targetservers get organizations](OrganizationEnvironmentTargetserverGetCall) (response)
/// * [environments targetservers update organizations](OrganizationEnvironmentTargetserverUpdateCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TargetServer {
    /// Optional. A human-readable description of this TargetServer.
    pub description: Option<String>,
    /// Required. The host name this target connects to. Value must be a valid hostname as described by RFC-1123.
    pub host: Option<String>,
    /// Optional. Enabling/disabling a TargetServer is useful when TargetServers are used in load balancing configurations, and one or more TargetServers need to taken out of rotation periodically. Defaults to true.
    #[serde(rename="isEnabled")]
    pub is_enabled: Option<bool>,
    /// Required. The resource id of this target server. Values must match the regular expression 
    pub name: Option<String>,
    /// Required. The port number this target connects to on the given host. Value must be between 1 and 65535, inclusive.
    pub port: Option<i32>,
    /// Immutable. The protocol used by this TargetServer.
    pub protocol: Option<String>,
    /// Optional. Specifies TLS configuration info for this TargetServer. The JSON name is `sSLInfo` for legacy/backwards compatibility reasons -- Edge originally supported SSL, and the name is still used for TLS configuration.
    #[serde(rename="sSLInfo")]
    pub s_sl_info: Option<GoogleCloudApigeeV1TlsInfo>,
}

impl client::RequestValue for GoogleCloudApigeeV1TargetServer {}
impl client::ResponseResult for GoogleCloudApigeeV1TargetServer {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TargetServerConfig {
    /// Whether the target server is enabled. An empty/omitted value for this field should be interpreted as true.
    pub enabled: Option<bool>,
    /// Host name of the target server.
    pub host: Option<String>,
    /// Target server revision name in the following format: `organizations/{org}/environments/{env}/targetservers/{targetserver}/revisions/{rev}`
    pub name: Option<String>,
    /// Port number for the target server.
    pub port: Option<i32>,
    /// The protocol used by this target server.
    pub protocol: Option<String>,
    /// TLS settings for the target server.
    #[serde(rename="tlsInfo")]
    pub tls_info: Option<GoogleCloudApigeeV1TlsInfoConfig>,
}

impl client::Part for GoogleCloudApigeeV1TargetServerConfig {}


/// The response for TestDatastore
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [analytics datastores test organizations](OrganizationAnalyticDatastoreTestCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TestDatastoreResponse {
    /// Output only. Error message of test connection failure
    pub error: Option<String>,
    /// Output only. It could be `completed` or `failed`
    pub state: Option<String>,
}

impl client::ResponseResult for GoogleCloudApigeeV1TestDatastoreResponse {}


/// TLS configuration information for virtual hosts and TargetServers.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TlsInfo {
    /// The SSL/TLS cipher suites to be used. Must be one of the cipher suite names listed in: http://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#ciphersuites
    pub ciphers: Option<Vec<String>>,
    /// Optional. Enables two-way TLS.
    #[serde(rename="clientAuthEnabled")]
    pub client_auth_enabled: Option<bool>,
    /// The TLS Common Name of the certificate.
    #[serde(rename="commonName")]
    pub common_name: Option<GoogleCloudApigeeV1TlsInfoCommonName>,
    /// Required. Enables TLS. If false, neither one-way nor two-way TLS will be enabled.
    pub enabled: Option<bool>,
    /// If true, Edge ignores TLS certificate errors. Valid when configuring TLS for target servers and target endpoints, and when configuring virtual hosts that use 2-way TLS. When used with a target endpoint/target server, if the backend system uses SNI and returns a cert with a subject Distinguished Name (DN) that does not match the hostname, there is no way to ignore the error and the connection fails.
    #[serde(rename="ignoreValidationErrors")]
    pub ignore_validation_errors: Option<bool>,
    /// Required if `client_auth_enabled` is true. The resource ID for the alias containing the private key and cert.
    #[serde(rename="keyAlias")]
    pub key_alias: Option<String>,
    /// Required if `client_auth_enabled` is true. The resource ID of the keystore.
    #[serde(rename="keyStore")]
    pub key_store: Option<String>,
    /// The TLS versioins to be used.
    pub protocols: Option<Vec<String>>,
    /// The resource ID of the truststore.
    #[serde(rename="trustStore")]
    pub trust_store: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1TlsInfo {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TlsInfoCommonName {
    /// The TLS Common Name string of the certificate.
    pub value: Option<String>,
    /// Indicates whether the cert should be matched against as a wildcard cert.
    #[serde(rename="wildcardMatch")]
    pub wildcard_match: Option<bool>,
}

impl client::Part for GoogleCloudApigeeV1TlsInfoCommonName {}


/// There is no detailed description.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TlsInfoConfig {
    /// List of ciphers that are granted access.
    pub ciphers: Option<Vec<String>>,
    /// Flag that specifies whether client-side authentication is enabled for the target server. Enables two-way TLS.
    #[serde(rename="clientAuthEnabled")]
    pub client_auth_enabled: Option<bool>,
    /// Common name to validate the target server against.
    #[serde(rename="commonName")]
    pub common_name: Option<GoogleCloudApigeeV1CommonNameConfig>,
    /// Flag that specifies whether one-way TLS is enabled. Set to `true` to enable one-way TLS.
    pub enabled: Option<bool>,
    /// Flag that specifies whether to ignore TLS certificate validation errors. Set to `true` to ignore errors.
    #[serde(rename="ignoreValidationErrors")]
    pub ignore_validation_errors: Option<bool>,
    /// Name of the alias used for client-side authentication in the following format: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`
    #[serde(rename="keyAlias")]
    pub key_alias: Option<String>,
    /// Reference name and alias pair to use for client-side authentication.
    #[serde(rename="keyAliasReference")]
    pub key_alias_reference: Option<GoogleCloudApigeeV1KeyAliasReference>,
    /// List of TLS protocols that are granted access.
    pub protocols: Option<Vec<String>>,
    /// Name of the keystore or keystore reference containing trusted certificates for the server in the following format: `organizations/{org}/environments/{env}/keystores/{keystore}` or `organizations/{org}/environments/{env}/references/{reference}`
    #[serde(rename="trustStore")]
    pub trust_store: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1TlsInfoConfig {}


/// TraceConfig defines the configurations in an environment of distributed trace.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments get trace config organizations](OrganizationEnvironmentGetTraceConfigCall) (response)
/// * [environments update trace config organizations](OrganizationEnvironmentUpdateTraceConfigCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TraceConfig {
    /// Required. Endpoint of the exporter.
    pub endpoint: Option<String>,
    /// Required. Exporter that is used to view the distributed trace captured using OpenCensus. An exporter sends traces to any backend that is capable of consuming them. Recorded spans can be exported by registered exporters.
    pub exporter: Option<String>,
    /// Distributed trace configuration for all API proxies in an environment. You can also override the configuration for a specific API proxy using the distributed trace configuration overrides API.
    #[serde(rename="samplingConfig")]
    pub sampling_config: Option<GoogleCloudApigeeV1TraceSamplingConfig>,
}

impl client::RequestValue for GoogleCloudApigeeV1TraceConfig {}
impl client::ResponseResult for GoogleCloudApigeeV1TraceConfig {}


/// A representation of a configuration override.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments trace config overrides create organizations](OrganizationEnvironmentTraceConfigOverrideCreateCall) (request|response)
/// * [environments trace config overrides get organizations](OrganizationEnvironmentTraceConfigOverrideGetCall) (response)
/// * [environments trace config overrides patch organizations](OrganizationEnvironmentTraceConfigOverridePatchCall) (request|response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TraceConfigOverride {
    /// ID of the API proxy that will have its trace configuration overridden.
    #[serde(rename="apiProxy")]
    pub api_proxy: Option<String>,
    /// ID of the trace configuration override specified as a system-generated UUID.
    pub name: Option<String>,
    /// Trace configuration to override.
    #[serde(rename="samplingConfig")]
    pub sampling_config: Option<GoogleCloudApigeeV1TraceSamplingConfig>,
}

impl client::RequestValue for GoogleCloudApigeeV1TraceConfigOverride {}
impl client::ResponseResult for GoogleCloudApigeeV1TraceConfigOverride {}


/// TraceSamplingConfig represents the detail settings of distributed tracing. Only the fields that are defined in the distributed trace configuration can be overridden using the distribute trace configuration override APIs.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1TraceSamplingConfig {
    /// Sampler of distributed tracing. OFF is the default value.
    pub sampler: Option<String>,
    /// Field sampling rate. This value is only applicable when using the PROBABILITY sampler. The supported values are > 0 and <= 0.5.
    #[serde(rename="samplingRate")]
    pub sampling_rate: Option<f32>,
}

impl client::Part for GoogleCloudApigeeV1TraceSamplingConfig {}


/// Details on why a resource update failed in the runtime.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleCloudApigeeV1UpdateError {
    /// Status code.
    pub code: Option<String>,
    /// User-friendly error message.
    pub message: Option<String>,
    /// The sub resource specific to this error (e.g. a proxy deployed within the EnvironmentConfig). If empty the error refers to the top level resource.
    pub resource: Option<String>,
    /// A string that uniquely identifies the type of error. This provides a more reliable means to deduplicate errors across revisions and instances.
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::Part for GoogleCloudApigeeV1UpdateError {}


/// Specifies the audit configuration for a service. The configuration determines which permission types are logged, and what identities, if any, are exempted from logging. An AuditConfig must have one or more AuditLogConfigs. If there are AuditConfigs for both `allServices` and a specific service, the union of the two AuditConfigs is used for that service: the log_types specified in each AuditConfig are enabled, and the exempted_members in each AuditLogConfig are exempted. Example Policy with multiple AuditConfigs: { "audit_configs": [ { "service": "allServices", "audit_log_configs": [ { "log_type": "DATA_READ", "exempted_members": [ "user:jose@example.com" ] }, { "log_type": "DATA_WRITE" }, { "log_type": "ADMIN_READ" } ] }, { "service": "sampleservice.googleapis.com", "audit_log_configs": [ { "log_type": "DATA_READ" }, { "log_type": "DATA_WRITE", "exempted_members": [ "user:aliya@example.com" ] } ] } ] } For sampleservice, this policy enables DATA_READ, DATA_WRITE and ADMIN_READ logging. It also exempts jose@example.com from DATA_READ logging, and aliya@example.com from DATA_WRITE logging.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1AuditConfig {
    /// The configuration for logging of each type of permission.
    #[serde(rename="auditLogConfigs")]
    pub audit_log_configs: Option<Vec<GoogleIamV1AuditLogConfig>>,
    /// Specifies a service that will be enabled for audit logging. For example, `storage.googleapis.com`, `cloudsql.googleapis.com`. `allServices` is a special value that covers all services.
    pub service: Option<String>,
}

impl client::Part for GoogleIamV1AuditConfig {}


/// Provides the configuration for logging a type of permissions. Example: { "audit_log_configs": [ { "log_type": "DATA_READ", "exempted_members": [ "user:jose@example.com" ] }, { "log_type": "DATA_WRITE" } ] } This enables 'DATA_READ' and 'DATA_WRITE' logging, while exempting jose@example.com from DATA_READ logging.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1AuditLogConfig {
    /// Specifies the identities that do not cause logging for this type of permission. Follows the same format of Binding.members.
    #[serde(rename="exemptedMembers")]
    pub exempted_members: Option<Vec<String>>,
    /// The log type that this config enables.
    #[serde(rename="logType")]
    pub log_type: Option<String>,
}

impl client::Part for GoogleIamV1AuditLogConfig {}


/// Associates `members`, or principals, with a `role`.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1Binding {
    /// The condition that is associated with this binding. If the condition evaluates to `true`, then this binding applies to the current request. If the condition evaluates to `false`, then this binding does not apply to the current request. However, a different role binding might grant the same role to one or more of the principals in this binding. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies).
    pub condition: Option<GoogleTypeExpr>,
    /// Specifies the principals requesting access for a Cloud Platform resource. `members` can have the following values: * `allUsers`: A special identifier that represents anyone who is on the internet; with or without a Google account. * `allAuthenticatedUsers`: A special identifier that represents anyone who is authenticated with a Google account or a service account. * `user:{emailid}`: An email address that represents a specific Google account. For example, `alice@example.com` . * `serviceAccount:{emailid}`: An email address that represents a service account. For example, `my-other-app@appspot.gserviceaccount.com`. * `group:{emailid}`: An email address that represents a Google group. For example, `admins@example.com`. * `deleted:user:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a user that has been recently deleted. For example, `alice@example.com?uid=123456789012345678901`. If the user is recovered, this value reverts to `user:{emailid}` and the recovered user retains the role in the binding. * `deleted:serviceAccount:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a service account that has been recently deleted. For example, `my-other-app@appspot.gserviceaccount.com?uid=123456789012345678901`. If the service account is undeleted, this value reverts to `serviceAccount:{emailid}` and the undeleted service account retains the role in the binding. * `deleted:group:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a Google group that has been recently deleted. For example, `admins@example.com?uid=123456789012345678901`. If the group is recovered, this value reverts to `group:{emailid}` and the recovered group retains the role in the binding. * `domain:{domain}`: The G Suite domain (primary) that represents all the users of that domain. For example, `google.com` or `example.com`. 
    pub members: Option<Vec<String>>,
    /// Role that is assigned to the list of `members`, or principals. For example, `roles/viewer`, `roles/editor`, or `roles/owner`.
    pub role: Option<String>,
}

impl client::Part for GoogleIamV1Binding {}


/// An Identity and Access Management (IAM) policy, which specifies access controls for Google Cloud resources. A `Policy` is a collection of `bindings`. A `binding` binds one or more `members`, or principals, to a single `role`. Principals can be user accounts, service accounts, Google groups, and domains (such as G Suite). A `role` is a named list of permissions; each `role` can be an IAM predefined role or a user-created custom role. For some types of Google Cloud resources, a `binding` can also specify a `condition`, which is a logical expression that allows access to a resource only if the expression evaluates to `true`. A condition can add constraints based on attributes of the request, the resource, or both. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies). **JSON example:** { "bindings": [ { "role": "roles/resourcemanager.organizationAdmin", "members": [ "user:mike@example.com", "group:admins@example.com", "domain:google.com", "serviceAccount:my-project-id@appspot.gserviceaccount.com" ] }, { "role": "roles/resourcemanager.organizationViewer", "members": [ "user:eve@example.com" ], "condition": { "title": "expirable access", "description": "Does not grant access after Sep 2020", "expression": "request.time < timestamp('2020-10-01T00:00:00.000Z')", } } ], "etag": "BwWWja0YfJA=", "version": 3 } **YAML example:** bindings: - members: - user:mike@example.com - group:admins@example.com - domain:google.com - serviceAccount:my-project-id@appspot.gserviceaccount.com role: roles/resourcemanager.organizationAdmin - members: - user:eve@example.com role: roles/resourcemanager.organizationViewer condition: title: expirable access description: Does not grant access after Sep 2020 expression: request.time < timestamp('2020-10-01T00:00:00.000Z') etag: BwWWja0YfJA= version: 3 For a description of IAM and its features, see the [IAM documentation](https://cloud.google.com/iam/docs/).
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments get iam policy organizations](OrganizationEnvironmentGetIamPolicyCall) (response)
/// * [environments set iam policy organizations](OrganizationEnvironmentSetIamPolicyCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1Policy {
    /// Specifies cloud audit logging configuration for this policy.
    #[serde(rename="auditConfigs")]
    pub audit_configs: Option<Vec<GoogleIamV1AuditConfig>>,
    /// Associates a list of `members`, or principals, with a `role`. Optionally, may specify a `condition` that determines how and when the `bindings` are applied. Each of the `bindings` must contain at least one principal. The `bindings` in a `Policy` can refer to up to 1,500 principals; up to 250 of these principals can be Google groups. Each occurrence of a principal counts towards these limits. For example, if the `bindings` grant 50 different roles to `user:alice@example.com`, and not to any other principal, then you can add another 1,450 principals to the `bindings` in the `Policy`.
    pub bindings: Option<Vec<GoogleIamV1Binding>>,
    /// `etag` is used for optimistic concurrency control as a way to help prevent simultaneous updates of a policy from overwriting each other. It is strongly suggested that systems make use of the `etag` in the read-modify-write cycle to perform policy updates in order to avoid race conditions: An `etag` is returned in the response to `getIamPolicy`, and systems are expected to put that etag in the request to `setIamPolicy` to ensure that their change will be applied to the same version of the policy. **Important:** If you use IAM Conditions, you must include the `etag` field whenever you call `setIamPolicy`. If you omit this field, then IAM allows you to overwrite a version `3` policy with a version `1` policy, and all of the conditions in the version `3` policy are lost.
    pub etag: Option<String>,
    /// Specifies the format of the policy. Valid values are `0`, `1`, and `3`. Requests that specify an invalid value are rejected. Any operation that affects conditional role bindings must specify version `3`. This requirement applies to the following operations: * Getting a policy that includes a conditional role binding * Adding a conditional role binding to a policy * Changing a conditional role binding in a policy * Removing any role binding, with or without a condition, from a policy that includes conditions **Important:** If you use IAM Conditions, you must include the `etag` field whenever you call `setIamPolicy`. If you omit this field, then IAM allows you to overwrite a version `3` policy with a version `1` policy, and all of the conditions in the version `3` policy are lost. If a policy does not include any conditions, operations on that policy may specify any valid version or leave the field unset. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies).
    pub version: Option<i32>,
}

impl client::ResponseResult for GoogleIamV1Policy {}


/// Request message for `SetIamPolicy` method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments set iam policy organizations](OrganizationEnvironmentSetIamPolicyCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1SetIamPolicyRequest {
    /// REQUIRED: The complete policy to be applied to the `resource`. The size of the policy is limited to a few 10s of KB. An empty policy is a valid policy but certain Cloud Platform services (such as Projects) might reject them.
    pub policy: Option<GoogleIamV1Policy>,
    /// OPTIONAL: A FieldMask specifying which fields of the policy to modify. Only the fields in the mask will be modified. If no mask is provided, the following default mask is used: `paths: "bindings, etag"`
    #[serde(rename="updateMask")]
    pub update_mask: Option<String>,
}

impl client::RequestValue for GoogleIamV1SetIamPolicyRequest {}


/// Request message for `TestIamPermissions` method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments test iam permissions organizations](OrganizationEnvironmentTestIamPermissionCall) (request)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1TestIamPermissionsRequest {
    /// The set of permissions to check for the `resource`. Permissions with wildcards (such as '*' or 'storage.*') are not allowed. For more information see [IAM Overview](https://cloud.google.com/iam/docs/overview#permissions).
    pub permissions: Option<Vec<String>>,
}

impl client::RequestValue for GoogleIamV1TestIamPermissionsRequest {}


/// Response message for `TestIamPermissions` method.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [environments test iam permissions organizations](OrganizationEnvironmentTestIamPermissionCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleIamV1TestIamPermissionsResponse {
    /// A subset of `TestPermissionsRequest.permissions` that the caller is allowed.
    pub permissions: Option<Vec<String>>,
}

impl client::ResponseResult for GoogleIamV1TestIamPermissionsResponse {}


/// The response message for Operations.ListOperations.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [operations list organizations](OrganizationOperationListCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleLongrunningListOperationsResponse {
    /// The standard List next-page token.
    #[serde(rename="nextPageToken")]
    pub next_page_token: Option<String>,
    /// A list of operations that matches the specified filter in the request.
    pub operations: Option<Vec<GoogleLongrunningOperation>>,
}

impl client::ResponseResult for GoogleLongrunningListOperationsResponse {}


/// This resource represents a long-running operation that is the result of a network API call.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [endpoint attachments create organizations](OrganizationEndpointAttachmentCreateCall) (response)
/// * [endpoint attachments delete organizations](OrganizationEndpointAttachmentDeleteCall) (response)
/// * [envgroups attachments create organizations](OrganizationEnvgroupAttachmentCreateCall) (response)
/// * [envgroups attachments delete organizations](OrganizationEnvgroupAttachmentDeleteCall) (response)
/// * [envgroups create organizations](OrganizationEnvgroupCreateCall) (response)
/// * [envgroups delete organizations](OrganizationEnvgroupDeleteCall) (response)
/// * [envgroups patch organizations](OrganizationEnvgroupPatchCall) (response)
/// * [environments archive deployments create organizations](OrganizationEnvironmentArchiveDeploymentCreateCall) (response)
/// * [environments create organizations](OrganizationEnvironmentCreateCall) (response)
/// * [environments delete organizations](OrganizationEnvironmentDeleteCall) (response)
/// * [instances attachments create organizations](OrganizationInstanceAttachmentCreateCall) (response)
/// * [instances attachments delete organizations](OrganizationInstanceAttachmentDeleteCall) (response)
/// * [instances canaryevaluations create organizations](OrganizationInstanceCanaryevaluationCreateCall) (response)
/// * [instances nat addresses activate organizations](OrganizationInstanceNatAddresseActivateCall) (response)
/// * [instances nat addresses create organizations](OrganizationInstanceNatAddresseCreateCall) (response)
/// * [instances nat addresses delete organizations](OrganizationInstanceNatAddresseDeleteCall) (response)
/// * [instances create organizations](OrganizationInstanceCreateCall) (response)
/// * [instances delete organizations](OrganizationInstanceDeleteCall) (response)
/// * [instances patch organizations](OrganizationInstancePatchCall) (response)
/// * [operations get organizations](OrganizationOperationGetCall) (response)
/// * [create organizations](OrganizationCreateCall) (response)
/// * [delete organizations](OrganizationDeleteCall) (response)
/// * [set addons organizations](OrganizationSetAddonCall) (response)
/// * [provision organization projects](ProjectProvisionOrganizationCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleLongrunningOperation {
    /// If the value is `false`, it means the operation is still in progress. If `true`, the operation is completed, and either `error` or `response` is available.
    pub done: Option<bool>,
    /// The error result of the operation in case of failure or cancellation.
    pub error: Option<GoogleRpcStatus>,
    /// Service-specific metadata associated with the operation. It typically contains progress information and common metadata such as create time. Some services might not provide such metadata. Any method that returns a long-running operation should document the metadata type, if any.
    pub metadata: Option<HashMap<String, String>>,
    /// The server-assigned name, which is only unique within the same service that originally returns it. If you use the default HTTP mapping, the `name` should be a resource name ending with `operations/{unique_id}`.
    pub name: Option<String>,
    /// The normal response of the operation in case of success. If the original method returns no data on success, such as `Delete`, the response is `google.protobuf.Empty`. If the original method is standard `Get`/`Create`/`Update`, the response should be the resource. For other methods, the response should have the type `XxxResponse`, where `Xxx` is the original method name. For example, if the original method name is `TakeSnapshot()`, the inferred response type is `TakeSnapshotResponse`.
    pub response: Option<HashMap<String, String>>,
}

impl client::ResponseResult for GoogleLongrunningOperation {}


/// A generic empty message that you can re-use to avoid defining duplicated empty messages in your APIs. A typical example is to use it as the request or the response type of an API method. For instance: service Foo { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); } The JSON representation for `Empty` is empty JSON object `{}`.
/// 
/// # Activities
/// 
/// This type is used in activities, which are methods you may call on this type or where this type is involved in. 
/// The list links the activity name, along with information about where it is used (one of *request* and *response*).
/// 
/// * [analytics datastores delete organizations](OrganizationAnalyticDatastoreDeleteCall) (response)
/// * [datacollectors delete organizations](OrganizationDatacollectorDeleteCall) (response)
/// * [developers apps keys apiproducts update developer app key api product organizations](OrganizationDeveloperAppKeyApiproductUpdateDeveloperAppKeyApiProductCall) (response)
/// * [developers set developer status organizations](OrganizationDeveloperSetDeveloperStatuCall) (response)
/// * [environments apis revisions debugsessions delete data organizations](OrganizationEnvironmentApiRevisionDebugsessionDeleteDataCall) (response)
/// * [environments apis revisions undeploy organizations](OrganizationEnvironmentApiRevisionUndeployCall) (response)
/// * [environments archive deployments delete organizations](OrganizationEnvironmentArchiveDeploymentDeleteCall) (response)
/// * [environments caches delete organizations](OrganizationEnvironmentCacheDeleteCall) (response)
/// * [environments sharedflows revisions undeploy organizations](OrganizationEnvironmentSharedflowRevisionUndeployCall) (response)
/// * [environments trace config overrides delete organizations](OrganizationEnvironmentTraceConfigOverrideDeleteCall) (response)
/// * [environments unsubscribe organizations](OrganizationEnvironmentUnsubscribeCall) (response)
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleProtobufEmpty { _never_set: Option<bool> }

impl client::ResponseResult for GoogleProtobufEmpty {}


/// Describes what preconditions have failed. For example, if an RPC failed because it required the Terms of Service to be acknowledged, it could list the terms of service violation in the PreconditionFailure message.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleRpcPreconditionFailure {
    /// Describes all precondition violations.
    pub violations: Option<Vec<GoogleRpcPreconditionFailureViolation>>,
}

impl client::Part for GoogleRpcPreconditionFailure {}


/// A message type used to describe a single precondition failure.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleRpcPreconditionFailureViolation {
    /// A description of how the precondition failed. Developers can use this description to understand how to fix the failure. For example: "Terms of service not accepted".
    pub description: Option<String>,
    /// The subject, relative to the type, that failed. For example, "google.com/cloud" relative to the "TOS" type would indicate which terms of service is being referenced.
    pub subject: Option<String>,
    /// The type of PreconditionFailure. We recommend using a service-specific enum type to define the supported precondition violation subjects. For example, "TOS" for "Terms of Service violation".
    #[serde(rename="type")]
    pub type_: Option<String>,
}

impl client::Part for GoogleRpcPreconditionFailureViolation {}


/// The `Status` type defines a logical error model that is suitable for different programming environments, including REST APIs and RPC APIs. It is used by [gRPC](https://github.com/grpc). Each `Status` message contains three pieces of data: error code, error message, and error details. You can find out more about this error model and how to work with it in the [API Design Guide](https://cloud.google.com/apis/design/errors).
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleRpcStatus {
    /// The status code, which should be an enum value of google.rpc.Code.
    pub code: Option<i32>,
    /// A list of messages that carry the error details. There is a common set of message types for APIs to use.
    pub details: Option<Vec<HashMap<String, String>>>,
    /// A developer-facing error message, which should be in English. Any user-facing error message should be localized and sent in the google.rpc.Status.details field, or localized by the client.
    pub message: Option<String>,
}

impl client::Part for GoogleRpcStatus {}


/// Represents a textual expression in the Common Expression Language (CEL) syntax. CEL is a C-like expression language. The syntax and semantics of CEL are documented at https://github.com/google/cel-spec. Example (Comparison): title: "Summary size limit" description: "Determines if a summary is less than 100 chars" expression: "document.summary.size() < 100" Example (Equality): title: "Requestor is owner" description: "Determines if requestor is the document owner" expression: "document.owner == request.auth.claims.email" Example (Logic): title: "Public documents" description: "Determine whether the document should be publicly visible" expression: "document.type != 'private' && document.type != 'internal'" Example (Data Manipulation): title: "Notification string" description: "Create a notification string with a timestamp." expression: "'New message received at ' + string(document.create_time)" The exact variables and functions that may be referenced within an expression are determined by the service that evaluates it. See the service documentation for additional information.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleTypeExpr {
    /// Optional. Description of the expression. This is a longer text which describes the expression, e.g. when hovered over it in a UI.
    pub description: Option<String>,
    /// Textual representation of an expression in Common Expression Language syntax.
    pub expression: Option<String>,
    /// Optional. String indicating the location of the expression for error reporting, e.g. a file name and a position in the file.
    pub location: Option<String>,
    /// Optional. Title for the expression, i.e. a short string describing its purpose. This can be used e.g. in UIs which allow to enter the expression.
    pub title: Option<String>,
}

impl client::Part for GoogleTypeExpr {}


/// Represents an amount of money with its currency type.
/// 
/// This type is not used in any activity, and only used as *part* of another schema.
/// 
#[derive(Default, Clone, Debug, Serialize, Deserialize)]
pub struct GoogleTypeMoney {
    /// The three-letter currency code defined in ISO 4217.
    #[serde(rename="currencyCode")]
    pub currency_code: Option<String>,
    /// Number of nano (10^-9) units of the amount. The value must be between -999,999,999 and +999,999,999 inclusive. If `units` is positive, `nanos` must be positive or zero. If `units` is zero, `nanos` can be positive, zero, or negative. If `units` is negative, `nanos` must be negative or zero. For example $-1.75 is represented as `units`=-1 and `nanos`=-750,000,000.
    pub nanos: Option<i32>,
    /// The whole units of the amount. For example if `currencyCode` is `"USD"`, then 1 unit is one US dollar.
    pub units: Option<String>,
}

impl client::Part for GoogleTypeMoney {}



// ###################
// MethodBuilders ###
// #################

/// A builder providing access to all methods supported on *hybrid* resources.
/// It is not used directly, but through the `Apigee` hub.
///
/// # Example
///
/// Instantiate a resource builder
///
/// ```test_harness,no_run
/// extern crate hyper;
/// extern crate hyper_rustls;
/// extern crate google_apigee1 as apigee1;
/// 
/// # async fn dox() {
/// use std::default::Default;
/// use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// let secret: oauth2::ApplicationSecret = Default::default();
/// let auth = oauth2::InstalledFlowAuthenticator::builder(
///         secret,
///         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
///     ).build().await.unwrap();
/// let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // Usually you wouldn't bind this to a variable, but keep calling *CallBuilders*
/// // like `issuers_list(...)`
/// // to build up your call.
/// let rb = hub.hybrid();
/// # }
/// ```
pub struct HybridMethods<'a>
    where  {

    hub: &'a Apigee<>,
}

impl<'a> client::MethodsBuilder for HybridMethods<'a> {}

impl<'a> HybridMethods<'a> {
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists hybrid services and its trusted issuers service account ids. This api is authenticated and unauthorized(allow all the users) and used by runtime authn-authz service to query control plane's issuer service account ids.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Must be of the form `hybrid/issuers`.
    pub fn issuers_list(&self, name: &str) -> HybridIssuerListCall<'a> {
        HybridIssuerListCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
}



/// A builder providing access to all methods supported on *organization* resources.
/// It is not used directly, but through the `Apigee` hub.
///
/// # Example
///
/// Instantiate a resource builder
///
/// ```test_harness,no_run
/// extern crate hyper;
/// extern crate hyper_rustls;
/// extern crate google_apigee1 as apigee1;
/// 
/// # async fn dox() {
/// use std::default::Default;
/// use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// let secret: oauth2::ApplicationSecret = Default::default();
/// let auth = oauth2::InstalledFlowAuthenticator::builder(
///         secret,
///         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
///     ).build().await.unwrap();
/// let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // Usually you wouldn't bind this to a variable, but keep calling *CallBuilders*
/// // like `analytics_datastores_create(...)`, `analytics_datastores_delete(...)`, `analytics_datastores_get(...)`, `analytics_datastores_list(...)`, `analytics_datastores_test(...)`, `analytics_datastores_update(...)`, `apiproducts_attributes(...)`, `apiproducts_attributes_delete(...)`, `apiproducts_attributes_get(...)`, `apiproducts_attributes_list(...)`, `apiproducts_attributes_update_api_product_attribute(...)`, `apiproducts_create(...)`, `apiproducts_delete(...)`, `apiproducts_get(...)`, `apiproducts_list(...)`, `apiproducts_rateplans_create(...)`, `apiproducts_rateplans_delete(...)`, `apiproducts_rateplans_get(...)`, `apiproducts_rateplans_list(...)`, `apiproducts_rateplans_update(...)`, `apiproducts_update(...)`, `apis_create(...)`, `apis_delete(...)`, `apis_deployments_list(...)`, `apis_get(...)`, `apis_keyvaluemaps_create(...)`, `apis_keyvaluemaps_delete(...)`, `apis_list(...)`, `apis_patch(...)`, `apis_revisions_delete(...)`, `apis_revisions_deployments_list(...)`, `apis_revisions_get(...)`, `apis_revisions_update_api_proxy_revision(...)`, `apps_get(...)`, `apps_list(...)`, `create(...)`, `datacollectors_create(...)`, `datacollectors_delete(...)`, `datacollectors_get(...)`, `datacollectors_list(...)`, `datacollectors_patch(...)`, `delete(...)`, `deployments_list(...)`, `developers_apps_attributes(...)`, `developers_apps_attributes_delete(...)`, `developers_apps_attributes_get(...)`, `developers_apps_attributes_list(...)`, `developers_apps_attributes_update_developer_app_attribute(...)`, `developers_apps_create(...)`, `developers_apps_delete(...)`, `developers_apps_generate_key_pair_or_update_developer_app_status(...)`, `developers_apps_get(...)`, `developers_apps_keys_apiproducts_delete(...)`, `developers_apps_keys_apiproducts_update_developer_app_key_api_product(...)`, `developers_apps_keys_create(...)`, `developers_apps_keys_create_create(...)`, `developers_apps_keys_delete(...)`, `developers_apps_keys_get(...)`, `developers_apps_keys_replace_developer_app_key(...)`, `developers_apps_keys_update_developer_app_key(...)`, `developers_apps_list(...)`, `developers_apps_update(...)`, `developers_attributes(...)`, `developers_attributes_delete(...)`, `developers_attributes_get(...)`, `developers_attributes_list(...)`, `developers_attributes_update_developer_attribute(...)`, `developers_balance_adjust(...)`, `developers_balance_credit(...)`, `developers_create(...)`, `developers_delete(...)`, `developers_get(...)`, `developers_get_balance(...)`, `developers_get_monetization_config(...)`, `developers_list(...)`, `developers_set_developer_status(...)`, `developers_subscriptions_create(...)`, `developers_subscriptions_expire(...)`, `developers_subscriptions_get(...)`, `developers_subscriptions_list(...)`, `developers_update(...)`, `developers_update_monetization_config(...)`, `endpoint_attachments_create(...)`, `endpoint_attachments_delete(...)`, `endpoint_attachments_get(...)`, `endpoint_attachments_list(...)`, `envgroups_attachments_create(...)`, `envgroups_attachments_delete(...)`, `envgroups_attachments_get(...)`, `envgroups_attachments_list(...)`, `envgroups_create(...)`, `envgroups_delete(...)`, `envgroups_get(...)`, `envgroups_list(...)`, `envgroups_patch(...)`, `environments_analytics_admin_get_schemav2(...)`, `environments_analytics_exports_create(...)`, `environments_analytics_exports_get(...)`, `environments_analytics_exports_list(...)`, `environments_apis_deployments_list(...)`, `environments_apis_revisions_debugsessions_create(...)`, `environments_apis_revisions_debugsessions_data_get(...)`, `environments_apis_revisions_debugsessions_delete_data(...)`, `environments_apis_revisions_debugsessions_get(...)`, `environments_apis_revisions_debugsessions_list(...)`, `environments_apis_revisions_deploy(...)`, `environments_apis_revisions_deployments_generate_deploy_change_report(...)`, `environments_apis_revisions_deployments_generate_undeploy_change_report(...)`, `environments_apis_revisions_get_deployments(...)`, `environments_apis_revisions_undeploy(...)`, `environments_archive_deployments_create(...)`, `environments_archive_deployments_delete(...)`, `environments_archive_deployments_generate_download_url(...)`, `environments_archive_deployments_generate_upload_url(...)`, `environments_archive_deployments_get(...)`, `environments_archive_deployments_list(...)`, `environments_archive_deployments_patch(...)`, `environments_caches_delete(...)`, `environments_create(...)`, `environments_delete(...)`, `environments_deployments_list(...)`, `environments_flowhooks_attach_shared_flow_to_flow_hook(...)`, `environments_flowhooks_detach_shared_flow_from_flow_hook(...)`, `environments_flowhooks_get(...)`, `environments_get(...)`, `environments_get_debugmask(...)`, `environments_get_deployed_config(...)`, `environments_get_iam_policy(...)`, `environments_get_trace_config(...)`, `environments_keystores_aliases_create(...)`, `environments_keystores_aliases_csr(...)`, `environments_keystores_aliases_delete(...)`, `environments_keystores_aliases_get(...)`, `environments_keystores_aliases_get_certificate(...)`, `environments_keystores_aliases_update(...)`, `environments_keystores_create(...)`, `environments_keystores_delete(...)`, `environments_keystores_get(...)`, `environments_keyvaluemaps_create(...)`, `environments_keyvaluemaps_delete(...)`, `environments_optimized_stats_get(...)`, `environments_queries_create(...)`, `environments_queries_get(...)`, `environments_queries_get_result(...)`, `environments_queries_list(...)`, `environments_references_create(...)`, `environments_references_delete(...)`, `environments_references_get(...)`, `environments_references_update(...)`, `environments_resourcefiles_create(...)`, `environments_resourcefiles_delete(...)`, `environments_resourcefiles_get(...)`, `environments_resourcefiles_list(...)`, `environments_resourcefiles_list_environment_resources(...)`, `environments_resourcefiles_update(...)`, `environments_set_iam_policy(...)`, `environments_sharedflows_deployments_list(...)`, `environments_sharedflows_revisions_deploy(...)`, `environments_sharedflows_revisions_get_deployments(...)`, `environments_sharedflows_revisions_undeploy(...)`, `environments_stats_get(...)`, `environments_subscribe(...)`, `environments_targetservers_create(...)`, `environments_targetservers_delete(...)`, `environments_targetservers_get(...)`, `environments_targetservers_update(...)`, `environments_test_iam_permissions(...)`, `environments_trace_config_overrides_create(...)`, `environments_trace_config_overrides_delete(...)`, `environments_trace_config_overrides_get(...)`, `environments_trace_config_overrides_list(...)`, `environments_trace_config_overrides_patch(...)`, `environments_unsubscribe(...)`, `environments_update(...)`, `environments_update_debugmask(...)`, `environments_update_environment(...)`, `environments_update_trace_config(...)`, `get(...)`, `get_deployed_ingress_config(...)`, `get_runtime_config(...)`, `get_sync_authorization(...)`, `host_queries_create(...)`, `host_queries_get(...)`, `host_queries_get_result(...)`, `host_queries_get_result_view(...)`, `host_queries_list(...)`, `host_stats_get(...)`, `instances_attachments_create(...)`, `instances_attachments_delete(...)`, `instances_attachments_get(...)`, `instances_attachments_list(...)`, `instances_canaryevaluations_create(...)`, `instances_canaryevaluations_get(...)`, `instances_create(...)`, `instances_delete(...)`, `instances_get(...)`, `instances_list(...)`, `instances_nat_addresses_activate(...)`, `instances_nat_addresses_create(...)`, `instances_nat_addresses_delete(...)`, `instances_nat_addresses_get(...)`, `instances_nat_addresses_list(...)`, `instances_patch(...)`, `instances_report_status(...)`, `keyvaluemaps_create(...)`, `keyvaluemaps_delete(...)`, `list(...)`, `operations_get(...)`, `operations_list(...)`, `optimized_host_stats_get(...)`, `reports_create(...)`, `reports_delete(...)`, `reports_get(...)`, `reports_list(...)`, `reports_update(...)`, `set_addons(...)`, `set_sync_authorization(...)`, `sharedflows_create(...)`, `sharedflows_delete(...)`, `sharedflows_deployments_list(...)`, `sharedflows_get(...)`, `sharedflows_list(...)`, `sharedflows_revisions_delete(...)`, `sharedflows_revisions_deployments_list(...)`, `sharedflows_revisions_get(...)`, `sharedflows_revisions_update_shared_flow_revision(...)`, `sites_apicategories_create(...)`, `sites_apicategories_delete(...)`, `sites_apicategories_get(...)`, `sites_apicategories_list(...)`, `sites_apicategories_patch(...)` and `update(...)`
/// // to build up your call.
/// let rb = hub.organizations();
/// # }
/// ```
pub struct OrganizationMethods<'a>
    where  {

    hub: &'a Apigee<>,
}

impl<'a> client::MethodsBuilder for OrganizationMethods<'a> {}

impl<'a> OrganizationMethods<'a> {
    
    /// Create a builder to help you perform the following task:
    ///
    /// Create a Datastore for an org
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent organization name. Must be of the form `organizations/{org}`.
    pub fn analytics_datastores_create(&self, request: GoogleCloudApigeeV1Datastore, parent: &str) -> OrganizationAnalyticDatastoreCreateCall<'a> {
        OrganizationAnalyticDatastoreCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Delete a Datastore from an org.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name of the Datastore to be deleted. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    pub fn analytics_datastores_delete(&self, name: &str) -> OrganizationAnalyticDatastoreDeleteCall<'a> {
        OrganizationAnalyticDatastoreDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Get a Datastore
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name of the Datastore to be get. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    pub fn analytics_datastores_get(&self, name: &str) -> OrganizationAnalyticDatastoreGetCall<'a> {
        OrganizationAnalyticDatastoreGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// List Datastores
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The parent organization name. Must be of the form `organizations/{org}`.
    pub fn analytics_datastores_list(&self, parent: &str) -> OrganizationAnalyticDatastoreListCall<'a> {
        OrganizationAnalyticDatastoreListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _target_type: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Test if Datastore configuration is correct. This includes checking if credentials provided by customer have required permissions in target destination storage
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent organization name Must be of the form `organizations/{org}`
    pub fn analytics_datastores_test(&self, request: GoogleCloudApigeeV1Datastore, parent: &str) -> OrganizationAnalyticDatastoreTestCall<'a> {
        OrganizationAnalyticDatastoreTestCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Update a Datastore
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. The resource name of datastore to be updated. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    pub fn analytics_datastores_update(&self, request: GoogleCloudApigeeV1Datastore, name: &str) -> OrganizationAnalyticDatastoreUpdateCall<'a> {
        OrganizationAnalyticDatastoreUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an API product attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API product attribute. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/attributes/{attribute}`
    pub fn apiproducts_attributes_delete(&self, name: &str) -> OrganizationApiproductAttributeDeleteCall<'a> {
        OrganizationApiproductAttributeDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the value of an API product attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API product attribute. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/attributes/{attribute}`
    pub fn apiproducts_attributes_get(&self, name: &str) -> OrganizationApiproductAttributeGetCall<'a> {
        OrganizationApiproductAttributeGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all API product attributes.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_attributes_list(&self, parent: &str) -> OrganizationApiproductAttributeListCall<'a> {
        OrganizationApiproductAttributeListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the value of an API product attribute. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with entities also get cached for at least 180 seconds after entity is accessed during runtime. In this case, the `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_attributes_update_api_product_attribute(&self, request: GoogleCloudApigeeV1Attribute, name: &str) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {
        OrganizationApiproductAttributeUpdateApiProductAttributeCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Create a rate plan that is associated with an API product in an organization. Using rate plans, API product owners can monetize their API products by configuring one or more of the following: - Billing frequency - Initial setup fees for using an API product - Payment funding model (postpaid only) - Fixed recurring or consumption-based charges for using an API product - Revenue sharing with developer partners An API product can have multiple rate plans associated with it but *only one* rate plan can be active at any point of time. **Note: From the developer's perspective, they purchase API products not rate plans.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the API product that is associated with the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_rateplans_create(&self, request: GoogleCloudApigeeV1RatePlan, parent: &str) -> OrganizationApiproductRateplanCreateCall<'a> {
        OrganizationApiproductRateplanCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a rate plan.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. ID of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    pub fn apiproducts_rateplans_delete(&self, name: &str) -> OrganizationApiproductRateplanDeleteCall<'a> {
        OrganizationApiproductRateplanDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the details of a rate plan.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    pub fn apiproducts_rateplans_get(&self, name: &str) -> OrganizationApiproductRateplanGetCall<'a> {
        OrganizationApiproductRateplanGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all the rate plans for an API product.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}` Use `organizations/{org}/apiproducts/-` to return rate plans for all API products within the organization.
    pub fn apiproducts_rateplans_list(&self, parent: &str) -> OrganizationApiproductRateplanListCall<'a> {
        OrganizationApiproductRateplanListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _state: Default::default(),
            _start_key: Default::default(),
            _order_by: Default::default(),
            _expand: Default::default(),
            _count: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing rate plan.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    pub fn apiproducts_rateplans_update(&self, request: GoogleCloudApigeeV1RatePlan, name: &str) -> OrganizationApiproductRateplanUpdateCall<'a> {
        OrganizationApiproductRateplanUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates or creates API product attributes. This API **replaces** the current list of attributes with the attributes specified in the request body. In this way, you can update existing attributes, add new attributes, or delete existing attributes by omitting them from the request body. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with entities also get cached for at least 180 seconds after entity is accessed during runtime. In this case, the `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_attributes(&self, request: GoogleCloudApigeeV1Attributes, name: &str) -> OrganizationApiproductAttributeCall<'a> {
        OrganizationApiproductAttributeCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an API product in an organization. You create API products after you have proxied backend services using API proxies. An API product is a collection of API resources combined with quota settings and metadata that you can use to deliver customized and productized API bundles to your developer community. This metadata can include: - Scope - Environments - API proxies - Extensible profile API products enable you repackage APIs on the fly, without having to do any additional coding or configuration. Apigee recommends that you start with a simple API product including only required elements. You then provision credentials to apps to enable them to start testing your APIs. After you have authentication and authorization working against a simple API product, you can iterate to create finer-grained API products, defining different sets of API resources for each API product. **WARNING:** - If you don't specify an API proxy in the request body, *any* app associated with the product can make calls to *any* API in your entire organization. - If you don't specify an environment in the request body, the product allows access to all environments. For more information, see What is an API product?
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in which the API product will be created. Use the following structure in your request: `organizations/{org}`
    pub fn apiproducts_create(&self, request: GoogleCloudApigeeV1ApiProduct, parent: &str) -> OrganizationApiproductCreateCall<'a> {
        OrganizationApiproductCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an API product from an organization. Deleting an API product causes app requests to the resource URIs defined in the API product to fail. Ensure that you create a new API product to serve existing apps, unless your intention is to disable access to the resources defined in the API product. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via the UI or the API. View the list of API products to verify the internal name.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_delete(&self, name: &str) -> OrganizationApiproductDeleteCall<'a> {
        OrganizationApiproductDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets configuration details for an API product. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via the UI or the API. View the list of API products to verify the internal name.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_get(&self, name: &str) -> OrganizationApiproductGetCall<'a> {
        OrganizationApiproductGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all API product names for an organization. Filter the list by passing an `attributename` and `attibutevalue`. The maximum number of API products returned is 1000. You can paginate the list of API products returned using the `startKey` and `count` query parameters.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}`
    pub fn apiproducts_list(&self, parent: &str) -> OrganizationApiproductListCall<'a> {
        OrganizationApiproductListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _start_key: Default::default(),
            _expand: Default::default(),
            _count: Default::default(),
            _attributevalue: Default::default(),
            _attributename: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing API product. You must include all required values, whether or not you are updating them, as well as any optional values that you are updating. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via UI or API. View the list of API products to identify their internal names.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    pub fn apiproducts_update(&self, request: GoogleCloudApigeeV1ApiProduct, name: &str) -> OrganizationApiproductUpdateCall<'a> {
        OrganizationApiproductUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of an API proxy.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the API proxy for which to return deployment information in the following format: `organizations/{org}/apis/{api}`
    pub fn apis_deployments_list(&self, parent: &str) -> OrganizationApiDeploymentListCall<'a> {
        OrganizationApiDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a key value map in an API proxy.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment in which to create the key value map. Use the following structure in your request: `organizations/{org}/apis/{api}`
    pub fn apis_keyvaluemaps_create(&self, request: GoogleCloudApigeeV1KeyValueMap, parent: &str) -> OrganizationApiKeyvaluemapCreateCall<'a> {
        OrganizationApiKeyvaluemapCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a key value map from an API proxy.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the key value map. Use the following structure in your request: `organizations/{org}/apis/{api}/keyvaluemaps/{keyvaluemap}`
    pub fn apis_keyvaluemaps_delete(&self, name: &str) -> OrganizationApiKeyvaluemapDeleteCall<'a> {
        OrganizationApiKeyvaluemapDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of an API proxy revision.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the API proxy revision for which to return deployment information in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`.
    pub fn apis_revisions_deployments_list(&self, parent: &str) -> OrganizationApiRevisionDeploymentListCall<'a> {
        OrganizationApiRevisionDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an API proxy revision and all policies, resources, endpoints, and revisions associated with it. The API proxy revision must be undeployed before you can delete it.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. API proxy revision in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    pub fn apis_revisions_delete(&self, name: &str) -> OrganizationApiRevisionDeleteCall<'a> {
        OrganizationApiRevisionDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an API proxy revision. To download the API proxy configuration bundle for the specified revision as a zip file, set the `format` query parameter to `bundle`. If you are using curl, specify `-o filename.zip` to save the output to a file; otherwise, it displays to `stdout`. Then, develop the API proxy configuration locally and upload the updated API proxy configuration revision, as described in [updateApiProxyRevision](updateApiProxyRevision).
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. API proxy revision in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    pub fn apis_revisions_get(&self, name: &str) -> OrganizationApiRevisionGetCall<'a> {
        OrganizationApiRevisionGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _format: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing API proxy revision by uploading the API proxy configuration bundle as a zip file from your local machine. You can update only API proxy revisions that have never been deployed. After deployment, an API proxy revision becomes immutable, even if it is undeployed. Set the `Content-Type` header to either `multipart/form-data` or `application/octet-stream`.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. API proxy revision to update in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    pub fn apis_revisions_update_api_proxy_revision(&self, request: GoogleApiHttpBody, name: &str) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {
        OrganizationApiRevisionUpdateApiProxyRevisionCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _validate: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an API proxy. The API proxy created will not be accessible at runtime until it is deployed to an environment. Create a new API proxy by setting the `name` query parameter to the name of the API proxy. Import an API proxy configuration bundle stored in zip format on your local machine to your organization by doing the following: * Set the `name` query parameter to the name of the API proxy. * Set the `action` query parameter to `import`. * Set the `Content-Type` header to `multipart/form-data`. * Pass as a file the name of API proxy configuration bundle stored in zip format on your local machine using the `file` form field. **Note**: To validate the API proxy configuration bundle only without importing it, set the `action` query parameter to `validate`. When importing an API proxy configuration bundle, if the API proxy does not exist, it will be created. If the API proxy exists, then a new revision is created. Invalid API proxy configurations are rejected, and a list of validation errors is returned to the client.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in the following format: `organizations/{org}`
    pub fn apis_create(&self, request: GoogleApiHttpBody, parent: &str) -> OrganizationApiCreateCall<'a> {
        OrganizationApiCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _validate: Default::default(),
            _name: Default::default(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an API proxy and all associated endpoints, policies, resources, and revisions. The API proxy must be undeployed before you can delete it.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API proxy in the following format: `organizations/{org}/apis/{api}`
    pub fn apis_delete(&self, name: &str) -> OrganizationApiDeleteCall<'a> {
        OrganizationApiDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an API proxy including a list of existing revisions.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API proxy in the following format: `organizations/{org}/apis/{api}`
    pub fn apis_get(&self, name: &str) -> OrganizationApiGetCall<'a> {
        OrganizationApiGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the names of all API proxies in an organization. The names returned correspond to the names defined in the configuration files for each API proxy.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization in the following format: `organizations/{org}`
    pub fn apis_list(&self, parent: &str) -> OrganizationApiListCall<'a> {
        OrganizationApiListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _include_revisions: Default::default(),
            _include_meta_data: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing API proxy.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. API proxy to update in the following format: `organizations/{org}/apis/{api}`
    pub fn apis_patch(&self, request: GoogleCloudApigeeV1ApiProxy, name: &str) -> OrganizationApiPatchCall<'a> {
        OrganizationApiPatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the app profile for the specified app ID.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. App ID in the following format: `organizations/{org}/apps/{app}`
    pub fn apps_get(&self, name: &str) -> OrganizationAppGetCall<'a> {
        OrganizationAppGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists IDs of apps within an organization that have the specified app status (approved or revoked) or are of the specified app type (developer or company).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Resource path of the parent in the following format: `organizations/{org}`
    pub fn apps_list(&self, parent: &str) -> OrganizationAppListCall<'a> {
        OrganizationAppListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _status: Default::default(),
            _start_key: Default::default(),
            _rows: Default::default(),
            _key_status: Default::default(),
            _include_cred: Default::default(),
            _ids: Default::default(),
            _expand: Default::default(),
            _apptype: Default::default(),
            _api_product: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new data collector.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in which to create the data collector in the following format: `organizations/{org}`.
    pub fn datacollectors_create(&self, request: GoogleCloudApigeeV1DataCollector, parent: &str) -> OrganizationDatacollectorCreateCall<'a> {
        OrganizationDatacollectorCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _data_collector_id: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a data collector.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the data collector in the following format: `organizations/{org}/datacollectors/{data_collector_id}`.
    pub fn datacollectors_delete(&self, name: &str) -> OrganizationDatacollectorDeleteCall<'a> {
        OrganizationDatacollectorDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a data collector.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the data collector in the following format: `organizations/{org}/datacollectors/{data_collector_id}`.
    pub fn datacollectors_get(&self, name: &str) -> OrganizationDatacollectorGetCall<'a> {
        OrganizationDatacollectorGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all data collectors.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization for which to list data collectors in the following format: `organizations/{org}`.
    pub fn datacollectors_list(&self, parent: &str) -> OrganizationDatacollectorListCall<'a> {
        OrganizationDatacollectorListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a data collector.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the data collector in the following format: `organizations/{org}/datacollectors/{data_collector_id}`.
    pub fn datacollectors_patch(&self, request: GoogleCloudApigeeV1DataCollector, name: &str) -> OrganizationDatacollectorPatchCall<'a> {
        OrganizationDatacollectorPatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of API proxies or shared flows.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization for which to return deployment information in the following format: `organizations/{org}`
    pub fn deployments_list(&self, parent: &str) -> OrganizationDeploymentListCall<'a> {
        OrganizationDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _shared_flows: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a developer app attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer app attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/attributes/{attribute}`
    pub fn developers_apps_attributes_delete(&self, name: &str) -> OrganizationDeveloperAppAttributeDeleteCall<'a> {
        OrganizationDeveloperAppAttributeDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns a developer app attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer app attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/attributes/{attribute}`
    pub fn developers_apps_attributes_get(&self, name: &str) -> OrganizationDeveloperAppAttributeGetCall<'a> {
        OrganizationDeveloperAppAttributeGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns a list of all developer app attributes.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_attributes_list(&self, parent: &str) -> OrganizationDeveloperAppAttributeListCall<'a> {
        OrganizationDeveloperAppAttributeListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a developer app attribute. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with these entities are cached for at least 180 seconds after the entity is accessed at runtime. Therefore, an `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the developer app attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/attributes/{attribute}`
    pub fn developers_apps_attributes_update_developer_app_attribute(&self, request: GoogleCloudApigeeV1Attribute, name: &str) -> OrganizationDeveloperAppAttributeUpdateDeveloperAppAttributeCall<'a> {
        OrganizationDeveloperAppAttributeUpdateDeveloperAppAttributeCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Removes an API product from an app's consumer key. After the API product is removed, the app cannot access the API resources defined in that API product. **Note**: The consumer key is not removed, only its association with the API product.
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the API product in the developer app key in the following format: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}/apiproducts/{apiproduct}`
    pub fn developers_apps_keys_apiproducts_delete(&self, name: &str) -> OrganizationDeveloperAppKeyApiproductDeleteCall<'a> {
        OrganizationDeveloperAppKeyApiproductDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Approves or revokes the consumer key for an API product. After a consumer key is approved, the app can use it to access APIs. A consumer key that is revoked or pending cannot be used to access an API. Any access tokens associated with a revoked consumer key will remain active. However, Apigee checks the status of the consumer key and if set to `revoked` will not allow access to the API.
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the API product in the developer app key in the following format: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}/apiproducts/{apiproduct}`
    pub fn developers_apps_keys_apiproducts_update_developer_app_key_api_product(&self, name: &str) -> OrganizationDeveloperAppKeyApiproductUpdateDeveloperAppKeyApiProductCall<'a> {
        OrganizationDeveloperAppKeyApiproductUpdateDeveloperAppKeyApiProductCall {
            hub: self.hub,
            _name: name.to_string(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a custom consumer key and secret for a developer app. This is particularly useful if you want to migrate existing consumer keys and secrets to Apigee from another system. Consumer keys and secrets can contain letters, numbers, underscores, and hyphens. No other special characters are allowed. To avoid service disruptions, a consumer key and secret should not exceed 2 KBs each. **Note**: When creating the consumer key and secret, an association to API products will not be made. Therefore, you should not specify the associated API products in your request. Instead, use the UpdateDeveloperAppKey API to make the association after the consumer key and secret are created. If a consumer key and secret already exist, you can keep them or delete them using the DeleteDeveloperAppKey API.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Parent of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps`
    pub fn developers_apps_keys_create_create(&self, request: GoogleCloudApigeeV1DeveloperAppKey, parent: &str) -> OrganizationDeveloperAppKeyCreateCreateCall<'a> {
        OrganizationDeveloperAppKeyCreateCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a custom consumer key and secret for a developer app. This is particularly useful if you want to migrate existing consumer keys and secrets to Apigee from another system. Consumer keys and secrets can contain letters, numbers, underscores, and hyphens. No other special characters are allowed. To avoid service disruptions, a consumer key and secret should not exceed 2 KBs each. **Note**: When creating the consumer key and secret, an association to API products will not be made. Therefore, you should not specify the associated API products in your request. Instead, use the UpdateDeveloperAppKey API to make the association after the consumer key and secret are created. If a consumer key and secret already exist, you can keep them or delete them using the DeleteDeveloperAppKey API.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Parent of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps`
    pub fn developers_apps_keys_create(&self, request: GoogleCloudApigeeV1DeveloperAppKey, parent: &str) -> OrganizationDeveloperAppKeyCreateCall<'a> {
        OrganizationDeveloperAppKeyCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an app's consumer key and removes all API products associated with the app. After the consumer key is deleted, it cannot be used to access any APIs. **Note**: After you delete a consumer key, you may want to: 1. Create a new consumer key and secret for the developer app using the CreateDeveloperAppKey API, and subsequently add an API product to the key using the UpdateDeveloperAppKey API. 2. Delete the developer app, if it is no longer required.
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}`
    pub fn developers_apps_keys_delete(&self, name: &str) -> OrganizationDeveloperAppKeyDeleteCall<'a> {
        OrganizationDeveloperAppKeyDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets details for a consumer key for a developer app, including the key and secret value, associated API products, and other information.
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}`
    pub fn developers_apps_keys_get(&self, name: &str) -> OrganizationDeveloperAppKeyGetCall<'a> {
        OrganizationDeveloperAppKeyGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the scope of an app. This API replaces the existing scopes with those specified in the request. Include or exclude any existing scopes that you want to retain or delete, respectively. The specified scopes must already be defined for the API products associated with the app. This API sets the `scopes` element under the `apiProducts` element in the attributes of the app.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Name of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}`
    pub fn developers_apps_keys_replace_developer_app_key(&self, request: GoogleCloudApigeeV1DeveloperAppKey, name: &str) -> OrganizationDeveloperAppKeyReplaceDeveloperAppKeyCall<'a> {
        OrganizationDeveloperAppKeyReplaceDeveloperAppKeyCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Adds an API product to a developer app key, enabling the app that holds the key to access the API resources bundled in the API product. In addition, you can add attributes to a developer app key. This API replaces the existing attributes with those specified in the request. Include or exclude any existing attributes that you want to retain or delete, respectively. You can use the same key to access all API products associated with the app.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Name of the developer app key. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}/keys/{key}`
    pub fn developers_apps_keys_update_developer_app_key(&self, request: GoogleCloudApigeeV1DeveloperAppKey, name: &str) -> OrganizationDeveloperAppKeyUpdateDeveloperAppKeyCall<'a> {
        OrganizationDeveloperAppKeyUpdateDeveloperAppKeyCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates attributes for a developer app. This API replaces the current attributes with those specified in the request.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_attributes(&self, request: GoogleCloudApigeeV1Attributes, name: &str) -> OrganizationDeveloperAppAttributeCall<'a> {
        OrganizationDeveloperAppAttributeCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an app associated with a developer. This API associates the developer app with the specified API product and auto-generates an API key for the app to use in calls to API proxies inside that API product. The `name` is the unique ID of the app that you can use in API calls. The `DisplayName` (set as an attribute) appears in the UI. If you don't set the `DisplayName` attribute, the `name` appears in the UI.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_apps_create(&self, request: GoogleCloudApigeeV1DeveloperApp, parent: &str) -> OrganizationDeveloperAppCreateCall<'a> {
        OrganizationDeveloperAppCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a developer app. **Note**: The delete operation is asynchronous. The developer app is deleted immediately, but its associated resources, such as app keys or access tokens, may take anywhere from a few seconds to a few minutes to be deleted.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_delete(&self, name: &str) -> OrganizationDeveloperAppDeleteCall<'a> {
        OrganizationDeveloperAppDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Manages access to a developer app by enabling you to: * Approve or revoke a developer app * Generate a new consumer key and secret for a developer app To approve or revoke a developer app, set the `action` query parameter to `approved` or `revoked`, respectively, and the `Content-Type` header to `application/octet-stream`. If a developer app is revoked, none of its API keys are valid for API calls even though the keys are still `approved`. If successful, the API call returns the following HTTP status code: `204 No Content` To generate a new consumer key and secret for a developer app, pass the new key/secret details. Rather than replace an existing key, this API generates a new key. In this case, multiple key pairs may be associated with a single developer app. Each key pair has an independent status (`approved` or `revoked`) and expiration time. Any approved, non-expired key can be used in an API call. For example, if you're using API key rotation, you can generate new keys with expiration times that overlap keys that are going to expire. You might also generate a new consumer key/secret if the security of the original key/secret is compromised. The `keyExpiresIn` property defines the expiration time for the API key in milliseconds. If you don't set this property or set it to `-1`, the API key never expires. **Notes**: * When generating a new key/secret, this API replaces the existing attributes, notes, and callback URLs with those specified in the request. Include or exclude any existing information that you want to retain or delete, respectively. * To migrate existing consumer keys and secrets to hybrid from another system, see the CreateDeveloperAppKey API.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_generate_key_pair_or_update_developer_app_status(&self, request: GoogleCloudApigeeV1DeveloperApp, name: &str) -> OrganizationDeveloperAppGenerateKeyPairOrUpdateDeveloperAppStatuCall<'a> {
        OrganizationDeveloperAppGenerateKeyPairOrUpdateDeveloperAppStatuCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns the details for a developer app.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_get(&self, name: &str) -> OrganizationDeveloperAppGetCall<'a> {
        OrganizationDeveloperAppGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _query: Default::default(),
            _entity: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all apps created by a developer in an Apigee organization. Optionally, you can request an expanded view of the developer apps. A maximum of 100 developer apps are returned per API call. You can paginate the list of deveoper apps returned using the `startKey` and `count` query parameters.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_apps_list(&self, parent: &str) -> OrganizationDeveloperAppListCall<'a> {
        OrganizationDeveloperAppListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _start_key: Default::default(),
            _shallow_expand: Default::default(),
            _expand: Default::default(),
            _count: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the details for a developer app. In addition, you can add an API product to a developer app and automatically generate an API key for the app to use when calling APIs in the API product. If you want to use an existing API key for the API product, add the API product to the API key using the UpdateDeveloperAppKey API. Using this API, you cannot update the following: * App name as it is the primary key used to identify the app and cannot be changed. * Scopes associated with the app. Instead, use the ReplaceDeveloperAppKey API. This API replaces the existing attributes with those specified in the request. Include or exclude any existing attributes that you want to retain or delete, respectively.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the developer app. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/apps/{app}`
    pub fn developers_apps_update(&self, request: GoogleCloudApigeeV1DeveloperApp, name: &str) -> OrganizationDeveloperAppUpdateCall<'a> {
        OrganizationDeveloperAppUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a developer attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/attributes/{attribute}`
    pub fn developers_attributes_delete(&self, name: &str) -> OrganizationDeveloperAttributeDeleteCall<'a> {
        OrganizationDeveloperAttributeDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns the value of the specified developer attribute.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/attributes/{attribute}`
    pub fn developers_attributes_get(&self, name: &str) -> OrganizationDeveloperAttributeGetCall<'a> {
        OrganizationDeveloperAttributeGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns a list of all developer attributes.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Email address of the developer for which attributes are being listed. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_attributes_list(&self, parent: &str) -> OrganizationDeveloperAttributeListCall<'a> {
        OrganizationDeveloperAttributeListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a developer attribute. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (default). Any custom attributes associated with these entities are cached for at least 180 seconds after the entity is accessed at runtime. Therefore, an `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the developer attribute. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/attributes/{attribute}`
    pub fn developers_attributes_update_developer_attribute(&self, request: GoogleCloudApigeeV1Attribute, name: &str) -> OrganizationDeveloperAttributeUpdateDeveloperAttributeCall<'a> {
        OrganizationDeveloperAttributeUpdateDeveloperAttributeCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Adjust the prepaid balance for the developer. This API will be used in scenarios where the developer has been under-charged or over-charged.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Account balance for the developer. Use the following structure in your request: `organizations/{org}/developers/{developer}/balance`
    pub fn developers_balance_adjust(&self, request: GoogleCloudApigeeV1AdjustDeveloperBalanceRequest, name: &str) -> OrganizationDeveloperBalanceAdjustCall<'a> {
        OrganizationDeveloperBalanceAdjustCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Credits the account balance for the developer.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Account balance for the developer. Use the following structure in your request: `organizations/{org}/developers/{developer}/balance`
    pub fn developers_balance_credit(&self, request: GoogleCloudApigeeV1CreditDeveloperBalanceRequest, name: &str) -> OrganizationDeveloperBalanceCreditCall<'a> {
        OrganizationDeveloperBalanceCreditCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a subscription to an API product. 
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Email address of the developer that is purchasing a subscription to the API product. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_subscriptions_create(&self, request: GoogleCloudApigeeV1DeveloperSubscription, parent: &str) -> OrganizationDeveloperSubscriptionCreateCall<'a> {
        OrganizationDeveloperSubscriptionCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Expires an API product subscription immediately.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the API product subscription. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/subscriptions/{subscription}`
    pub fn developers_subscriptions_expire(&self, request: GoogleCloudApigeeV1ExpireDeveloperSubscriptionRequest, name: &str) -> OrganizationDeveloperSubscriptionExpireCall<'a> {
        OrganizationDeveloperSubscriptionExpireCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets details for an API product subscription.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API product subscription. Use the following structure in your request: `organizations/{org}/developers/{developer_email}/subscriptions/{subscription}`
    pub fn developers_subscriptions_get(&self, name: &str) -> OrganizationDeveloperSubscriptionGetCall<'a> {
        OrganizationDeveloperSubscriptionGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all API product subscriptions for a developer.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Email address of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_subscriptions_list(&self, parent: &str) -> OrganizationDeveloperSubscriptionListCall<'a> {
        OrganizationDeveloperSubscriptionListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _start_key: Default::default(),
            _count: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates developer attributes. This API replaces the existing attributes with those specified in the request. Add new attributes, and include or exclude any existing attributes that you want to retain or remove, respectively. The custom attribute limit is 18. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (default). Any custom attributes associated with these entities are cached for at least 180 seconds after the entity is accessed at runtime. Therefore, an `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Email address of the developer for which attributes are being updated. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_attributes(&self, request: GoogleCloudApigeeV1Attributes, parent: &str) -> OrganizationDeveloperAttributeCall<'a> {
        OrganizationDeveloperAttributeCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a developer. Once created, the developer can register an app and obtain an API key. At creation time, a developer is set as `active`. To change the developer status, use the SetDeveloperStatus API.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the Apigee organization in which the developer is created. Use the following structure in your request: `organizations/{org}`.
    pub fn developers_create(&self, request: GoogleCloudApigeeV1Developer, parent: &str) -> OrganizationDeveloperCreateCall<'a> {
        OrganizationDeveloperCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a developer. All apps and API keys associated with the developer are also removed. **Warning**: This API will permanently delete the developer and related artifacts. To avoid permanently deleting developers and their artifacts, set the developer status to `inactive` using the SetDeveloperStatus API. **Note**: The delete operation is asynchronous. The developer app is deleted immediately, but its associated resources, such as apps and API keys, may take anywhere from a few seconds to a few minutes to be deleted.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Email address of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_delete(&self, name: &str) -> OrganizationDeveloperDeleteCall<'a> {
        OrganizationDeveloperDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns the developer details, including the developer's name, email address, apps, and other information. **Note**: The response includes only the first 100 developer apps.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Email address of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_get(&self, name: &str) -> OrganizationDeveloperGetCall<'a> {
        OrganizationDeveloperGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the account balance for the developer.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Account balance for the developer. Use the following structure in your request: `organizations/{org}/developers/{developer}/balance`
    pub fn developers_get_balance(&self, name: &str) -> OrganizationDeveloperGetBalanceCall<'a> {
        OrganizationDeveloperGetBalanceCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the monetization configuration for the developer.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Monetization configuration for the developer. Use the following structure in your request: `organizations/{org}/developers/{developer}/monetizationConfig`
    pub fn developers_get_monetization_config(&self, name: &str) -> OrganizationDeveloperGetMonetizationConfigCall<'a> {
        OrganizationDeveloperGetMonetizationConfigCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all developers in an organization by email address. By default, the response does not include company developers. Set the `includeCompany` query parameter to `true` to include company developers. **Note**: A maximum of 1000 developers are returned in the response. You paginate the list of developers returned using the `startKey` and `count` query parameters.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the Apigee organization. Use the following structure in your request: `organizations/{org}`.
    pub fn developers_list(&self, parent: &str) -> OrganizationDeveloperListCall<'a> {
        OrganizationDeveloperListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _start_key: Default::default(),
            _include_company: Default::default(),
            _ids: Default::default(),
            _expand: Default::default(),
            _count: Default::default(),
            _app: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Sets the status of a developer. A developer is `active` by default. If you set a developer's status to `inactive`, the API keys assigned to the developer apps are no longer valid even though the API keys are set to `approved`. Inactive developers can still sign in to the developer portal and create apps; however, any new API keys generated during app creation won't work. To set the status of a developer, set the `action` query parameter to `active` or `inactive`, and the `Content-Type` header to `application/octet-stream`. If successful, the API call returns the following HTTP status code: `204 No Content`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_id}`
    pub fn developers_set_developer_status(&self, name: &str) -> OrganizationDeveloperSetDeveloperStatuCall<'a> {
        OrganizationDeveloperSetDeveloperStatuCall {
            hub: self.hub,
            _name: name.to_string(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a developer. This API replaces the existing developer details with those specified in the request. Include or exclude any existing details that you want to retain or delete, respectively. The custom attribute limit is 18. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with these entities are cached for at least 180 seconds after the entity is accessed at runtime. Therefore, an `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Email address of the developer. Use the following structure in your request: `organizations/{org}/developers/{developer_email}`
    pub fn developers_update(&self, request: GoogleCloudApigeeV1Developer, name: &str) -> OrganizationDeveloperUpdateCall<'a> {
        OrganizationDeveloperUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the monetization configuration for the developer.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Monetization configuration for the developer. Use the following structure in your request: `organizations/{org}/developers/{developer}/monetizationConfig`
    pub fn developers_update_monetization_config(&self, request: GoogleCloudApigeeV1DeveloperMonetizationConfig, name: &str) -> OrganizationDeveloperUpdateMonetizationConfigCall<'a> {
        OrganizationDeveloperUpdateMonetizationConfigCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an endpoint attachment. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Organization the endpoint attachment will be created in.
    pub fn endpoint_attachments_create(&self, request: GoogleCloudApigeeV1EndpointAttachment, parent: &str) -> OrganizationEndpointAttachmentCreateCall<'a> {
        OrganizationEndpointAttachmentCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _endpoint_attachment_id: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an endpoint attachment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the endpoint attachment. Use the following structure in your request: `organizations/{org}/endpointAttachments/{endpoint_attachment}`
    pub fn endpoint_attachments_delete(&self, name: &str) -> OrganizationEndpointAttachmentDeleteCall<'a> {
        OrganizationEndpointAttachmentDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the endpoint attachment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the endpoint attachment. Use the following structure in your request: `organizations/{org}/endpointAttachments/{endpoint_attachment}`
    pub fn endpoint_attachments_get(&self, name: &str) -> OrganizationEndpointAttachmentGetCall<'a> {
        OrganizationEndpointAttachmentGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the endpoint attachments in an organization.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization for which to list endpoint attachments. Use the following structure in your request: `organizations/{org}`
    pub fn endpoint_attachments_list(&self, parent: &str) -> OrganizationEndpointAttachmentListCall<'a> {
        OrganizationEndpointAttachmentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new attachment of an environment to an environment group.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. EnvironmentGroup under which to create the attachment in the following format: `organizations/{org}/envgroups/{envgroup}`.
    pub fn envgroups_attachments_create(&self, request: GoogleCloudApigeeV1EnvironmentGroupAttachment, parent: &str) -> OrganizationEnvgroupAttachmentCreateCall<'a> {
        OrganizationEnvgroupAttachmentCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an environment group attachment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment group attachment to delete in the following format: `organizations/{org}/envgroups/{envgroup}/attachments/{attachment}`.
    pub fn envgroups_attachments_delete(&self, name: &str) -> OrganizationEnvgroupAttachmentDeleteCall<'a> {
        OrganizationEnvgroupAttachmentDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an environment group attachment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment group attachment in the following format: `organizations/{org}/envgroups/{envgroup}/attachments/{attachment}`
    pub fn envgroups_attachments_get(&self, name: &str) -> OrganizationEnvgroupAttachmentGetCall<'a> {
        OrganizationEnvgroupAttachmentGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all attachments of an environment group.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment group in the following format: `organizations/{org}/envgroups/{envgroup}`.
    pub fn envgroups_attachments_list(&self, parent: &str) -> OrganizationEnvgroupAttachmentListCall<'a> {
        OrganizationEnvgroupAttachmentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new environment group.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in which to create the environment group in the following format: `organizations/{org}`.
    pub fn envgroups_create(&self, request: GoogleCloudApigeeV1EnvironmentGroup, parent: &str) -> OrganizationEnvgroupCreateCall<'a> {
        OrganizationEnvgroupCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _name: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an environment group.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment group in the following format: `organizations/{org}/envgroups/{envgroup}`.
    pub fn envgroups_delete(&self, name: &str) -> OrganizationEnvgroupDeleteCall<'a> {
        OrganizationEnvgroupDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an environment group.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment group in the following format: `organizations/{org}/envgroups/{envgroup}`.
    pub fn envgroups_get(&self, name: &str) -> OrganizationEnvgroupGetCall<'a> {
        OrganizationEnvgroupGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all environment groups.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization for which to list environment groups in the following format: `organizations/{org}`.
    pub fn envgroups_list(&self, parent: &str) -> OrganizationEnvgroupListCall<'a> {
        OrganizationEnvgroupListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an environment group.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the environment group to update in the format: `organizations/{org}/envgroups/{envgroup}.
    pub fn envgroups_patch(&self, request: GoogleCloudApigeeV1EnvironmentGroup, name: &str) -> OrganizationEnvgroupPatchCall<'a> {
        OrganizationEnvgroupPatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a list of metrics and dimensions that can be used to create analytics queries and reports. Each schema element contains the name of the field, its associated type, and a flag indicating whether it is a standard or custom field.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Path to the schema. Use the following structure in your request: `organizations/{org}/environments/{env}/analytics/admin/schemav2`.
    pub fn environments_analytics_admin_get_schemav2(&self, name: &str) -> OrganizationEnvironmentAnalyticAdminGetSchemav2Call<'a> {
        OrganizationEnvironmentAnalyticAdminGetSchemav2Call {
            hub: self.hub,
            _name: name.to_string(),
            _type_: Default::default(),
            _disable_cache: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Submit a data export job to be processed in the background. If the request is successful, the API returns a 201 status, a URI that can be used to retrieve the status of the export job, and the `state` value of "enqueued".
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Names of the parent organization and environment. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_analytics_exports_create(&self, request: GoogleCloudApigeeV1ExportRequest, parent: &str) -> OrganizationEnvironmentAnalyticExportCreateCall<'a> {
        OrganizationEnvironmentAnalyticExportCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the details and status of an analytics export job. If the export job is still in progress, its `state` is set to "running". After the export job has completed successfully, its `state` is set to "completed". If the export job fails, its `state` is set to `failed`.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name of the export to get.
    pub fn environments_analytics_exports_get(&self, name: &str) -> OrganizationEnvironmentAnalyticExportGetCall<'a> {
        OrganizationEnvironmentAnalyticExportGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the details and status of all analytics export jobs belonging to the parent organization and environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Names of the parent organization and environment. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_analytics_exports_list(&self, parent: &str) -> OrganizationEnvironmentAnalyticExportListCall<'a> {
        OrganizationEnvironmentAnalyticExportListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of an API proxy in an environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name representing an API proxy in an environment in the following format: `organizations/{org}/environments/{env}/apis/{api}`
    pub fn environments_apis_deployments_list(&self, parent: &str) -> OrganizationEnvironmentApiDeploymentListCall<'a> {
        OrganizationEnvironmentApiDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the debug data from a transaction.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the debug session transaction. Must be of the form: `organizations/{organization}/environments/{environment}/apis/{api}/revisions/{revision}/debugsessions/{session}/data/{transaction}`.
    pub fn environments_apis_revisions_debugsessions_data_get(&self, name: &str) -> OrganizationEnvironmentApiRevisionDebugsessionDataGetCall<'a> {
        OrganizationEnvironmentApiRevisionDebugsessionDataGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a debug session for a deployed API Proxy revision.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The resource name of the API Proxy revision deployment for which to create the DebugSession. Must be of the form `organizations/{organization}/environments/{environment}/apis/{api}/revisions/{revision}`.
    pub fn environments_apis_revisions_debugsessions_create(&self, request: GoogleCloudApigeeV1DebugSession, parent: &str) -> OrganizationEnvironmentApiRevisionDebugsessionCreateCall<'a> {
        OrganizationEnvironmentApiRevisionDebugsessionCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _timeout: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes the data from a debug session. This does not cancel the debug session or prevent further data from being collected if the session is still active in runtime pods.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the debug session to delete. Must be of the form: `organizations/{organization}/environments/{environment}/apis/{api}/revisions/{revision}/debugsessions/{debugsession}`.
    pub fn environments_apis_revisions_debugsessions_delete_data(&self, name: &str) -> OrganizationEnvironmentApiRevisionDebugsessionDeleteDataCall<'a> {
        OrganizationEnvironmentApiRevisionDebugsessionDeleteDataCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Retrieves a debug session.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the debug session to retrieve. Must be of the form: `organizations/{organization}/environments/{environment}/apis/{api}/revisions/{revision}/debugsessions/{session}`.
    pub fn environments_apis_revisions_debugsessions_get(&self, name: &str) -> OrganizationEnvironmentApiRevisionDebugsessionGetCall<'a> {
        OrganizationEnvironmentApiRevisionDebugsessionGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists debug sessions that are currently active in the given API Proxy revision.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The name of the API Proxy revision deployment for which to list debug sessions. Must be of the form: `organizations/{organization}/environments/{environment}/apis/{api}/revisions/{revision}`.
    pub fn environments_apis_revisions_debugsessions_list(&self, parent: &str) -> OrganizationEnvironmentApiRevisionDebugsessionListCall<'a> {
        OrganizationEnvironmentApiRevisionDebugsessionListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Generates a report for a dry run analysis of a DeployApiProxy request without committing the deployment. In addition to the standard validations performed when adding deployments, additional analysis will be done to detect possible traffic routing changes that would result from this deployment being created. Any potential routing conflicts or unsafe changes will be reported in the response. This routing analysis is not performed for a non-dry-run DeployApiProxy request. For a request path `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}/deployments:generateDeployChangeReport`, two permissions are required: * `apigee.deployments.create` on the resource `organizations/{org}/environments/{env}` * `apigee.proxyrevisions.deploy` on the resource `organizations/{org}/apis/{api}/revisions/{rev}`
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the API proxy revision deployment in the following format: `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}`
    pub fn environments_apis_revisions_deployments_generate_deploy_change_report(&self, name: &str) -> OrganizationEnvironmentApiRevisionDeploymentGenerateDeployChangeReportCall<'a> {
        OrganizationEnvironmentApiRevisionDeploymentGenerateDeployChangeReportCall {
            hub: self.hub,
            _name: name.to_string(),
            _override_: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Generates a report for a dry run analysis of an UndeployApiProxy request without committing the undeploy. In addition to the standard validations performed when removing deployments, additional analysis will be done to detect possible traffic routing changes that would result from this deployment being removed. Any potential routing conflicts or unsafe changes will be reported in the response. This routing analysis is not performed for a non-dry-run UndeployApiProxy request. For a request path `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}/deployments:generateUndeployChangeReport`, two permissions are required: * `apigee.deployments.delete` on the resource `organizations/{org}/environments/{env}` * `apigee.proxyrevisions.undeploy` on the resource `organizations/{org}/apis/{api}/revisions/{rev}`
    /// 
    /// # Arguments
    ///
    /// * `name` - Name of the API proxy revision deployment in the following format: `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}`
    pub fn environments_apis_revisions_deployments_generate_undeploy_change_report(&self, name: &str) -> OrganizationEnvironmentApiRevisionDeploymentGenerateUndeployChangeReportCall<'a> {
        OrganizationEnvironmentApiRevisionDeploymentGenerateUndeployChangeReportCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deploys a revision of an API proxy. If another revision of the same API proxy revision is currently deployed, set the `override` parameter to `true` to have this revision replace the currently deployed revision. You cannot invoke an API proxy until it has been deployed to an environment. After you deploy an API proxy revision, you cannot edit it. To edit the API proxy, you must create and deploy a new revision. For a request path `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}/deployments`, two permissions are required: * `apigee.deployments.create` on the resource `organizations/{org}/environments/{env}` * `apigee.proxyrevisions.deploy` on the resource `organizations/{org}/apis/{api}/revisions/{rev}` 
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API proxy revision deployment in the following format: `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}`
    pub fn environments_apis_revisions_deploy(&self, name: &str) -> OrganizationEnvironmentApiRevisionDeployCall<'a> {
        OrganizationEnvironmentApiRevisionDeployCall {
            hub: self.hub,
            _name: name.to_string(),
            _service_account: Default::default(),
            _sequenced_rollout: Default::default(),
            _override_: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the deployment of an API proxy revision and actual state reported by runtime pods.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name representing an API proxy revision in an environment in the following format: `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}`
    pub fn environments_apis_revisions_get_deployments(&self, name: &str) -> OrganizationEnvironmentApiRevisionGetDeploymentCall<'a> {
        OrganizationEnvironmentApiRevisionGetDeploymentCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Undeploys an API proxy revision from an environment. For a request path `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}/deployments`, two permissions are required: * `apigee.deployments.delete` on the resource `organizations/{org}/environments/{env}` * `apigee.proxyrevisions.undeploy` on the resource `organizations/{org}/apis/{api}/revisions/{rev}`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the API proxy revision deployment in the following format: `organizations/{org}/environments/{env}/apis/{api}/revisions/{rev}`
    pub fn environments_apis_revisions_undeploy(&self, name: &str) -> OrganizationEnvironmentApiRevisionUndeployCall<'a> {
        OrganizationEnvironmentApiRevisionUndeployCall {
            hub: self.hub,
            _name: name.to_string(),
            _sequenced_rollout: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new ArchiveDeployment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The Environment this Archive Deployment will be created in.
    pub fn environments_archive_deployments_create(&self, request: GoogleCloudApigeeV1ArchiveDeployment, parent: &str) -> OrganizationEnvironmentArchiveDeploymentCreateCall<'a> {
        OrganizationEnvironmentArchiveDeploymentCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an archive deployment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the Archive Deployment in the following format: `organizations/{org}/environments/{env}/archiveDeployments/{id}`.
    pub fn environments_archive_deployments_delete(&self, name: &str) -> OrganizationEnvironmentArchiveDeploymentDeleteCall<'a> {
        OrganizationEnvironmentArchiveDeploymentDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Generates a signed URL for downloading the original zip file used to create an Archive Deployment. The URL is only valid for a limited period and should be used within minutes after generation. Each call returns a new upload URL.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. The name of the Archive Deployment you want to download.
    pub fn environments_archive_deployments_generate_download_url(&self, request: GoogleCloudApigeeV1GenerateDownloadUrlRequest, name: &str) -> OrganizationEnvironmentArchiveDeploymentGenerateDownloadUrlCall<'a> {
        OrganizationEnvironmentArchiveDeploymentGenerateDownloadUrlCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Generates a signed URL for uploading an Archive zip file to Google Cloud Storage. Once the upload is complete, the signed URL should be passed to CreateArchiveDeployment. When uploading to the generated signed URL, please follow these restrictions: * Source file type should be a zip file. * Source file size should not exceed 1GB limit. * No credentials should be attached - the signed URLs provide access to the target bucket using internal service identity; if credentials were attached, the identity from the credentials would be used, but that identity does not have permissions to upload files to the URL. When making a HTTP PUT request, these two headers need to be specified: * `content-type: application/zip` * `x-goog-content-length-range: 0,1073741824` And this header SHOULD NOT be specified: * `Authorization: Bearer YOUR_TOKEN`
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The organization and environment to upload to.
    pub fn environments_archive_deployments_generate_upload_url(&self, request: GoogleCloudApigeeV1GenerateUploadUrlRequest, parent: &str) -> OrganizationEnvironmentArchiveDeploymentGenerateUploadUrlCall<'a> {
        OrganizationEnvironmentArchiveDeploymentGenerateUploadUrlCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the specified ArchiveDeployment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the Archive Deployment in the following format: `organizations/{org}/environments/{env}/archiveDeployments/{id}`.
    pub fn environments_archive_deployments_get(&self, name: &str) -> OrganizationEnvironmentArchiveDeploymentGetCall<'a> {
        OrganizationEnvironmentArchiveDeploymentGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the ArchiveDeployments in the specified Environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the Environment for which to list Archive Deployments in the format: `organizations/{org}/environments/{env}`.
    pub fn environments_archive_deployments_list(&self, parent: &str) -> OrganizationEnvironmentArchiveDeploymentListCall<'a> {
        OrganizationEnvironmentArchiveDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _filter: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing ArchiveDeployment. Labels can modified but most of the other fields are not modifiable.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Name of the Archive Deployment in the following format: `organizations/{org}/environments/{env}/archiveDeployments/{id}`.
    pub fn environments_archive_deployments_patch(&self, request: GoogleCloudApigeeV1ArchiveDeployment, name: &str) -> OrganizationEnvironmentArchiveDeploymentPatchCall<'a> {
        OrganizationEnvironmentArchiveDeploymentPatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a cache.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Cache resource name of the form: `organizations/{organization_id}/environments/{environment_id}/caches/{cache_id}`
    pub fn environments_caches_delete(&self, name: &str) -> OrganizationEnvironmentCacheDeleteCall<'a> {
        OrganizationEnvironmentCacheDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of API proxies or shared flows in an environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment for which to return deployment information in the following format: `organizations/{org}/environments/{env}`
    pub fn environments_deployments_list(&self, parent: &str) -> OrganizationEnvironmentDeploymentListCall<'a> {
        OrganizationEnvironmentDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _shared_flows: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Attaches a shared flow to a flow hook.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the flow hook to which the shared flow should be attached in the following format: `organizations/{org}/environments/{env}/flowhooks/{flowhook}`
    pub fn environments_flowhooks_attach_shared_flow_to_flow_hook(&self, request: GoogleCloudApigeeV1FlowHook, name: &str) -> OrganizationEnvironmentFlowhookAttachSharedFlowToFlowHookCall<'a> {
        OrganizationEnvironmentFlowhookAttachSharedFlowToFlowHookCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Detaches a shared flow from a flow hook.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the flow hook to detach in the following format: `organizations/{org}/environments/{env}/flowhooks/{flowhook}`
    pub fn environments_flowhooks_detach_shared_flow_from_flow_hook(&self, name: &str) -> OrganizationEnvironmentFlowhookDetachSharedFlowFromFlowHookCall<'a> {
        OrganizationEnvironmentFlowhookDetachSharedFlowFromFlowHookCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Returns the name of the shared flow attached to the specified flow hook. If there's no shared flow attached to the flow hook, the API does not return an error; it simply does not return a name in the response.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the flow hook in the following format: `organizations/{org}/environments/{env}/flowhooks/{flowhook}`
    pub fn environments_flowhooks_get(&self, name: &str) -> OrganizationEnvironmentFlowhookGetCall<'a> {
        OrganizationEnvironmentFlowhookGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an alias from a key/certificate pair. The structure of the request is controlled by the `format` query parameter: - `keycertfile` - Separate PEM-encoded key and certificate files are uploaded. Set `Content-Type: multipart/form-data` and include the `keyFile`, `certFile`, and `password` (if keys are encrypted) fields in the request body. If uploading to a truststore, omit `keyFile`. - `pkcs12` - A PKCS12 file is uploaded. Set `Content-Type: multipart/form-data`, provide the file in the `file` field, and include the `password` field if the file is encrypted in the request body. - `selfsignedcert` - A new private key and certificate are generated. Set `Content-Type: application/json` and include CertificateGenerationSpec in the request body.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the keystore. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}`.
    pub fn environments_keystores_aliases_create(&self, request: GoogleApiHttpBody, parent: &str) -> OrganizationEnvironmentKeystoreAliaseCreateCall<'a> {
        OrganizationEnvironmentKeystoreAliaseCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _ignore_newline_validation: Default::default(),
            _ignore_expiry_validation: Default::default(),
            _format: Default::default(),
            _alias: Default::default(),
            __password: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Generates a PKCS #10 Certificate Signing Request for the private key in an alias.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the alias. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`.
    pub fn environments_keystores_aliases_csr(&self, name: &str) -> OrganizationEnvironmentKeystoreAliaseCsrCall<'a> {
        OrganizationEnvironmentKeystoreAliaseCsrCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an alias.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the alias. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`.
    pub fn environments_keystores_aliases_delete(&self, name: &str) -> OrganizationEnvironmentKeystoreAliaseDeleteCall<'a> {
        OrganizationEnvironmentKeystoreAliaseDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an alias.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the alias. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`.
    pub fn environments_keystores_aliases_get(&self, name: &str) -> OrganizationEnvironmentKeystoreAliaseGetCall<'a> {
        OrganizationEnvironmentKeystoreAliaseGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the certificate from an alias in PEM-encoded form.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the alias. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`.
    pub fn environments_keystores_aliases_get_certificate(&self, name: &str) -> OrganizationEnvironmentKeystoreAliaseGetCertificateCall<'a> {
        OrganizationEnvironmentKeystoreAliaseGetCertificateCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the certificate in an alias.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the alias. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}/aliases/{alias}`
    pub fn environments_keystores_aliases_update(&self, request: GoogleApiHttpBody, name: &str) -> OrganizationEnvironmentKeystoreAliaseUpdateCall<'a> {
        OrganizationEnvironmentKeystoreAliaseUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _ignore_newline_validation: Default::default(),
            _ignore_expiry_validation: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a keystore or truststore. - Keystore: Contains certificates and their associated keys. - Truststore: Contains trusted certificates used to validate a server's certificate. These certificates are typically self-signed certificates or certificates that are not signed by a trusted CA.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment in which to create the keystore. Use the following format in your request: `organizations/{org}/environments/{env}`
    pub fn environments_keystores_create(&self, request: GoogleCloudApigeeV1Keystore, parent: &str) -> OrganizationEnvironmentKeystoreCreateCall<'a> {
        OrganizationEnvironmentKeystoreCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _name: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a keystore or truststore.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the keystore. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}`
    pub fn environments_keystores_delete(&self, name: &str) -> OrganizationEnvironmentKeystoreDeleteCall<'a> {
        OrganizationEnvironmentKeystoreDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a keystore or truststore.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the keystore. Use the following format in your request: `organizations/{org}/environments/{env}/keystores/{keystore}`.
    pub fn environments_keystores_get(&self, name: &str) -> OrganizationEnvironmentKeystoreGetCall<'a> {
        OrganizationEnvironmentKeystoreGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a key value map in an environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment in which to create the key value map. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_keyvaluemaps_create(&self, request: GoogleCloudApigeeV1KeyValueMap, parent: &str) -> OrganizationEnvironmentKeyvaluemapCreateCall<'a> {
        OrganizationEnvironmentKeyvaluemapCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a key value map from an environment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the key value map. Use the following structure in your request: `organizations/{org}/environments/{env}/keyvaluemaps/{keyvaluemap}`
    pub fn environments_keyvaluemaps_delete(&self, name: &str) -> OrganizationEnvironmentKeyvaluemapDeleteCall<'a> {
        OrganizationEnvironmentKeyvaluemapDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Similar to GetStats except that the response is less verbose.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name for which the interactive query will be executed. Use the following format in your request: `organizations/{org}/environments/{env}/optimizedStats/{dimensions}` Dimensions let you view metrics in meaningful groupings, such as `apiproxy`, `target_host`. The value of `dimensions` should be a comma-separated list as shown below: `organizations/{org}/environments/{env}/optimizedStats/apiproxy,request_verb`
    pub fn environments_optimized_stats_get(&self, name: &str) -> OrganizationEnvironmentOptimizedStatGetCall<'a> {
        OrganizationEnvironmentOptimizedStatGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _tzo: Default::default(),
            _ts_ascending: Default::default(),
            _topk: Default::default(),
            _time_unit: Default::default(),
            _time_range: Default::default(),
            _sortby: Default::default(),
            _sort: Default::default(),
            _sonar: Default::default(),
            _select: Default::default(),
            _realtime: Default::default(),
            _offset: Default::default(),
            _limit: Default::default(),
            _filter: Default::default(),
            _agg_table: Default::default(),
            _accuracy: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Submit a query to be processed in the background. If the submission of the query succeeds, the API returns a 201 status and an ID that refer to the query. In addition to the HTTP status 201, the `state` of "enqueued" means that the request succeeded.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent resource name. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_queries_create(&self, request: GoogleCloudApigeeV1Query, parent: &str) -> OrganizationEnvironmentQueryCreateCall<'a> {
        OrganizationEnvironmentQueryCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Get query status If the query is still in progress, the `state` is set to "running" After the query has completed successfully, `state` is set to "completed"
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the asynchronous query to get. Must be of the form `organizations/{org}/environments/{env}/queries/{queryId}`.
    pub fn environments_queries_get(&self, name: &str) -> OrganizationEnvironmentQueryGetCall<'a> {
        OrganizationEnvironmentQueryGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// After the query is completed, use this API to retrieve the results. If the request succeeds, and there is a non-zero result set, the result is downloaded to the client as a zipped JSON file. The name of the downloaded file will be: OfflineQueryResult-.zip Example: `OfflineQueryResult-9cfc0d85-0f30-46d6-ae6f-318d0cb961bd.zip`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the asynchronous query result to get. Must be of the form `organizations/{org}/environments/{env}/queries/{queryId}/result`.
    pub fn environments_queries_get_result(&self, name: &str) -> OrganizationEnvironmentQueryGetResultCall<'a> {
        OrganizationEnvironmentQueryGetResultCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Return a list of Asynchronous Queries
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The parent resource name. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_queries_list(&self, parent: &str) -> OrganizationEnvironmentQueryListCall<'a> {
        OrganizationEnvironmentQueryListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _to: Default::default(),
            _submitted_by: Default::default(),
            _status: Default::default(),
            _incl_queries_without_report: Default::default(),
            _from: Default::default(),
            _dataset: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a Reference in the specified environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent environment name under which the Reference will be created. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_references_create(&self, request: GoogleCloudApigeeV1Reference, parent: &str) -> OrganizationEnvironmentReferenceCreateCall<'a> {
        OrganizationEnvironmentReferenceCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a Reference from an environment. Returns the deleted Reference resource.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the Reference to delete. Must be of the form `organizations/{org}/environments/{env}/references/{ref}`.
    pub fn environments_references_delete(&self, name: &str) -> OrganizationEnvironmentReferenceDeleteCall<'a> {
        OrganizationEnvironmentReferenceDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a Reference resource.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the Reference to get. Must be of the form `organizations/{org}/environments/{env}/references/{ref}`.
    pub fn environments_references_get(&self, name: &str) -> OrganizationEnvironmentReferenceGetCall<'a> {
        OrganizationEnvironmentReferenceGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing Reference. Note that this operation has PUT semantics; it will replace the entirety of the existing Reference with the resource in the request body.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. The name of the Reference to update. Must be of the form `organizations/{org}/environments/{env}/references/{ref}`.
    pub fn environments_references_update(&self, request: GoogleCloudApigeeV1Reference, name: &str) -> OrganizationEnvironmentReferenceUpdateCall<'a> {
        OrganizationEnvironmentReferenceUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a resource file. Specify the `Content-Type` as `application/octet-stream` or `multipart/form-data`. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment in which to create the resource file in the following format: `organizations/{org}/environments/{env}`.
    pub fn environments_resourcefiles_create(&self, request: GoogleApiHttpBody, parent: &str) -> OrganizationEnvironmentResourcefileCreateCall<'a> {
        OrganizationEnvironmentResourcefileCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _type_: Default::default(),
            _name: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a resource file. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment in the following format: `organizations/{org}/environments/{env}`.
    /// * `type` - Required. Resource file type. {{ resource_file_type }}
    /// * `name` - Required. ID of the resource file to delete. Must match the regular expression: [a-zA-Z0-9:/\\!@#$%^&{}\[\]()+\-=,.~'` ]{1,255}
    pub fn environments_resourcefiles_delete(&self, parent: &str, type_: &str, name: &str) -> OrganizationEnvironmentResourcefileDeleteCall<'a> {
        OrganizationEnvironmentResourcefileDeleteCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _type_: type_.to_string(),
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the contents of a resource file. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment in the following format: `organizations/{org}/environments/{env}`.
    /// * `type` - Required. Resource file type. {{ resource_file_type }}
    /// * `name` - Required. ID of the resource file. Must match the regular expression: [a-zA-Z0-9:/\\!@#$%^&{}\[\]()+\-=,.~'` ]{1,255}
    pub fn environments_resourcefiles_get(&self, parent: &str, type_: &str, name: &str) -> OrganizationEnvironmentResourcefileGetCall<'a> {
        OrganizationEnvironmentResourcefileGetCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _type_: type_.to_string(),
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all resource files, optionally filtering by type. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment in which to list resource files in the following format: `organizations/{org}/environments/{env}`.
    pub fn environments_resourcefiles_list(&self, parent: &str) -> OrganizationEnvironmentResourcefileListCall<'a> {
        OrganizationEnvironmentResourcefileListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _type_: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all resource files, optionally filtering by type. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment in which to list resource files in the following format: `organizations/{org}/environments/{env}`.
    /// * `type` - Optional. Type of resource files to list. {{ resource_file_type }}
    pub fn environments_resourcefiles_list_environment_resources(&self, parent: &str, type_: &str) -> OrganizationEnvironmentResourcefileListEnvironmentResourceCall<'a> {
        OrganizationEnvironmentResourcefileListEnvironmentResourceCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _type_: type_.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a resource file. Specify the `Content-Type` as `application/octet-stream` or `multipart/form-data`. For more information about resource files, see [Resource files](https://cloud.google.com/apigee/docs/api-platform/develop/resource-files).
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment in the following format: `organizations/{org}/environments/{env}`.
    /// * `type` - Required. Resource file type. {{ resource_file_type }}
    /// * `name` - Required. ID of the resource file to update. Must match the regular expression: [a-zA-Z0-9:/\\!@#$%^&{}\[\]()+\-=,.~'` ]{1,255}
    pub fn environments_resourcefiles_update(&self, request: GoogleApiHttpBody, parent: &str, type_: &str, name: &str) -> OrganizationEnvironmentResourcefileUpdateCall<'a> {
        OrganizationEnvironmentResourcefileUpdateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _type_: type_.to_string(),
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of a shared flow in an environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name representing a shared flow in an environment in the following format: `organizations/{org}/environments/{env}/sharedflows/{sharedflow}`
    pub fn environments_sharedflows_deployments_list(&self, parent: &str) -> OrganizationEnvironmentSharedflowDeploymentListCall<'a> {
        OrganizationEnvironmentSharedflowDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deploys a revision of a shared flow. If another revision of the same shared flow is currently deployed, set the `override` parameter to `true` to have this revision replace the currently deployed revision. You cannot use a shared flow until it has been deployed to an environment. For a request path `organizations/{org}/environments/{env}/sharedflows/{sf}/revisions/{rev}/deployments`, two permissions are required: * `apigee.deployments.create` on the resource `organizations/{org}/environments/{env}` * `apigee.sharedflowrevisions.deploy` on the resource `organizations/{org}/sharedflows/{sf}/revisions/{rev}`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the shared flow revision to deploy in the following format: `organizations/{org}/environments/{env}/sharedflows/{sharedflow}/revisions/{rev}`
    pub fn environments_sharedflows_revisions_deploy(&self, name: &str) -> OrganizationEnvironmentSharedflowRevisionDeployCall<'a> {
        OrganizationEnvironmentSharedflowRevisionDeployCall {
            hub: self.hub,
            _name: name.to_string(),
            _service_account: Default::default(),
            _override_: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the deployment of a shared flow revision and actual state reported by runtime pods.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name representing a shared flow in an environment in the following format: `organizations/{org}/environments/{env}/sharedflows/{sharedflow}/revisions/{rev}`
    pub fn environments_sharedflows_revisions_get_deployments(&self, name: &str) -> OrganizationEnvironmentSharedflowRevisionGetDeploymentCall<'a> {
        OrganizationEnvironmentSharedflowRevisionGetDeploymentCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Undeploys a shared flow revision from an environment. For a request path `organizations/{org}/environments/{env}/sharedflows/{sf}/revisions/{rev}/deployments`, two permissions are required: * `apigee.deployments.delete` on the resource `organizations/{org}/environments/{env}` * `apigee.sharedflowrevisions.undeploy` on the resource `organizations/{org}/sharedflows/{sf}/revisions/{rev}`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the shared flow revision to undeploy in the following format: `organizations/{org}/environments/{env}/sharedflows/{sharedflow}/revisions/{rev}`
    pub fn environments_sharedflows_revisions_undeploy(&self, name: &str) -> OrganizationEnvironmentSharedflowRevisionUndeployCall<'a> {
        OrganizationEnvironmentSharedflowRevisionUndeployCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Retrieve metrics grouped by dimensions. The types of metrics you can retrieve include traffic, message counts, API call latency, response size, and cache hits and counts. Dimensions let you view metrics in meaningful groups. You can optionally pass dimensions as path parameters to the `stats` API. If dimensions are not specified, the metrics are computed on the entire set of data for the given time range.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name for which the interactive query will be executed. Use the following format in your request: `organizations/{org}/environments/{env}/stats/{dimensions}` Dimensions let you view metrics in meaningful groupings, such as `apiproxy` or `target_host`. The value of dimensions should be a comma-separated list, as shown below: `organizations/{org}/environments/{env}/stats/apiproxy,request_verb`
    pub fn environments_stats_get(&self, name: &str) -> OrganizationEnvironmentStatGetCall<'a> {
        OrganizationEnvironmentStatGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _tzo: Default::default(),
            _ts_ascending: Default::default(),
            _topk: Default::default(),
            _time_unit: Default::default(),
            _time_range: Default::default(),
            _sortby: Default::default(),
            _sort: Default::default(),
            _sonar: Default::default(),
            _select: Default::default(),
            _realtime: Default::default(),
            _offset: Default::default(),
            _limit: Default::default(),
            _filter: Default::default(),
            _agg_table: Default::default(),
            _accuracy: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a TargetServer in the specified environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent environment name under which the TargetServer will be created. Must be of the form `organizations/{org}/environments/{env}`.
    pub fn environments_targetservers_create(&self, request: GoogleCloudApigeeV1TargetServer, parent: &str) -> OrganizationEnvironmentTargetserverCreateCall<'a> {
        OrganizationEnvironmentTargetserverCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _name: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a TargetServer from an environment. Returns the deleted TargetServer resource.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the TargetServer to delete. Must be of the form `organizations/{org}/environments/{env}/targetservers/{target_server_id}`.
    pub fn environments_targetservers_delete(&self, name: &str) -> OrganizationEnvironmentTargetserverDeleteCall<'a> {
        OrganizationEnvironmentTargetserverDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a TargetServer resource.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the TargetServer to get. Must be of the form `organizations/{org}/environments/{env}/targetservers/{target_server_id}`.
    pub fn environments_targetservers_get(&self, name: &str) -> OrganizationEnvironmentTargetserverGetCall<'a> {
        OrganizationEnvironmentTargetserverGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing TargetServer. Note that this operation has PUT semantics; it will replace the entirety of the existing TargetServer with the resource in the request body.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. The name of the TargetServer to replace. Must be of the form `organizations/{org}/environments/{env}/targetservers/{target_server_id}`.
    pub fn environments_targetservers_update(&self, request: GoogleCloudApigeeV1TargetServer, name: &str) -> OrganizationEnvironmentTargetserverUpdateCall<'a> {
        OrganizationEnvironmentTargetserverUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a trace configuration override. The response contains a system-generated UUID, that can be used to view, update, or delete the configuration override. Use the List API to view the existing trace configuration overrides.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Parent resource of the trace configuration override. Use the following structure in your request. "organizations/*/environments/*/traceConfig".
    pub fn environments_trace_config_overrides_create(&self, request: GoogleCloudApigeeV1TraceConfigOverride, parent: &str) -> OrganizationEnvironmentTraceConfigOverrideCreateCall<'a> {
        OrganizationEnvironmentTraceConfigOverrideCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a distributed trace configuration override.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the trace configuration override. Use the following structure in your request: "organizations/*/environments/*/traceConfig/overrides/*".
    pub fn environments_trace_config_overrides_delete(&self, name: &str) -> OrganizationEnvironmentTraceConfigOverrideDeleteCall<'a> {
        OrganizationEnvironmentTraceConfigOverrideDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a trace configuration override.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the trace configuration override. Use the following structure in your request: "organizations/*/environments/*/traceConfig/overrides/*".
    pub fn environments_trace_config_overrides_get(&self, name: &str) -> OrganizationEnvironmentTraceConfigOverrideGetCall<'a> {
        OrganizationEnvironmentTraceConfigOverrideGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all of the distributed trace configuration overrides in an environment.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Parent resource of the trace configuration override. Use the following structure in your request: "organizations/*/environments/*/traceConfig".
    pub fn environments_trace_config_overrides_list(&self, parent: &str) -> OrganizationEnvironmentTraceConfigOverrideListCall<'a> {
        OrganizationEnvironmentTraceConfigOverrideListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a distributed trace configuration override. Note that the repeated fields have replace semantics when included in the field mask and that they will be overwritten by the value of the fields in the request body.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the trace configuration override. Use the following structure in your request: "organizations/*/environments/*/traceConfig/overrides/*".
    pub fn environments_trace_config_overrides_patch(&self, request: GoogleCloudApigeeV1TraceConfigOverride, name: &str) -> OrganizationEnvironmentTraceConfigOverridePatchCall<'a> {
        OrganizationEnvironmentTraceConfigOverridePatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an environment in an organization.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in which the environment will be created. Use the following structure in your request: `organizations/{org}`
    pub fn environments_create(&self, request: GoogleCloudApigeeV1Environment, parent: &str) -> OrganizationEnvironmentCreateCall<'a> {
        OrganizationEnvironmentCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _name: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an environment from an organization.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_delete(&self, name: &str) -> OrganizationEnvironmentDeleteCall<'a> {
        OrganizationEnvironmentDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets environment details.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_get(&self, name: &str) -> OrganizationEnvironmentGetCall<'a> {
        OrganizationEnvironmentGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the debug mask singleton resource for an environment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the debug mask. Use the following structure in your request: `organizations/{org}/environments/{env}/debugmask`.
    pub fn environments_get_debugmask(&self, name: &str) -> OrganizationEnvironmentGetDebugmaskCall<'a> {
        OrganizationEnvironmentGetDebugmaskCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the deployed configuration for an environment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the environment deployed configuration resource. Use the following structure in your request: `organizations/{org}/environments/{env}/deployedConfig`
    pub fn environments_get_deployed_config(&self, name: &str) -> OrganizationEnvironmentGetDeployedConfigCall<'a> {
        OrganizationEnvironmentGetDeployedConfigCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the IAM policy on an environment. For more information, see [Manage users, roles, and permissions using the API](https://cloud.google.com/apigee/docs/api-platform/system-administration/manage-users-roles). You must have the `apigee.environments.getIamPolicy` permission to call this API.
    /// 
    /// # Arguments
    ///
    /// * `resource` - REQUIRED: The resource for which the policy is being requested. See the operation documentation for the appropriate value for this field.
    pub fn environments_get_iam_policy(&self, resource: &str) -> OrganizationEnvironmentGetIamPolicyCall<'a> {
        OrganizationEnvironmentGetIamPolicyCall {
            hub: self.hub,
            _resource: resource.to_string(),
            _options_requested_policy_version: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Get distributed trace configuration in an environment.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the trace configuration. Use the following structure in your request: "organizations/*/environments/*/traceConfig".
    pub fn environments_get_trace_config(&self, name: &str) -> OrganizationEnvironmentGetTraceConfigCall<'a> {
        OrganizationEnvironmentGetTraceConfigCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Sets the IAM policy on an environment, if the policy already exists it will be replaced. For more information, see [Manage users, roles, and permissions using the API](https://cloud.google.com/apigee/docs/api-platform/system-administration/manage-users-roles). You must have the `apigee.environments.setIamPolicy` permission to call this API.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `resource` - REQUIRED: The resource for which the policy is being specified. See the operation documentation for the appropriate value for this field.
    pub fn environments_set_iam_policy(&self, request: GoogleIamV1SetIamPolicyRequest, resource: &str) -> OrganizationEnvironmentSetIamPolicyCall<'a> {
        OrganizationEnvironmentSetIamPolicyCall {
            hub: self.hub,
            _request: request,
            _resource: resource.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a subscription for the environment's Pub/Sub topic. The server will assign a random name for this subscription. The "name" and "push_config" must *not* be specified.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_subscribe(&self, parent: &str) -> OrganizationEnvironmentSubscribeCall<'a> {
        OrganizationEnvironmentSubscribeCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Tests the permissions of a user on an environment, and returns a subset of permissions that the user has on the environment. If the environment does not exist, an empty permission set is returned (a NOT_FOUND error is not returned).
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `resource` - REQUIRED: The resource for which the policy detail is being requested. See the operation documentation for the appropriate value for this field.
    pub fn environments_test_iam_permissions(&self, request: GoogleIamV1TestIamPermissionsRequest, resource: &str) -> OrganizationEnvironmentTestIamPermissionCall<'a> {
        OrganizationEnvironmentTestIamPermissionCall {
            hub: self.hub,
            _request: request,
            _resource: resource.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a subscription for the environment's Pub/Sub topic.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_unsubscribe(&self, request: GoogleCloudApigeeV1Subscription, parent: &str) -> OrganizationEnvironmentUnsubscribeCall<'a> {
        OrganizationEnvironmentUnsubscribeCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing environment. When updating properties, you must pass all existing properties to the API, even if they are not being changed. If you omit properties from the payload, the properties are removed. To get the current list of properties for the environment, use the [Get Environment API](get). **Note**: Both `PUT` and `POST` methods are supported for updating an existing environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_update(&self, request: GoogleCloudApigeeV1Environment, name: &str) -> OrganizationEnvironmentUpdateCall<'a> {
        OrganizationEnvironmentUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the debug mask singleton resource for an environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Name of the debug mask.
    pub fn environments_update_debugmask(&self, request: GoogleCloudApigeeV1DebugMask, name: &str) -> OrganizationEnvironmentUpdateDebugmaskCall<'a> {
        OrganizationEnvironmentUpdateDebugmaskCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _replace_repeated_fields: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an existing environment. When updating properties, you must pass all existing properties to the API, even if they are not being changed. If you omit properties from the payload, the properties are removed. To get the current list of properties for the environment, use the [Get Environment API](get). **Note**: Both `PUT` and `POST` methods are supported for updating an existing environment.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the environment. Use the following structure in your request: `organizations/{org}/environments/{env}`
    pub fn environments_update_environment(&self, request: GoogleCloudApigeeV1Environment, name: &str) -> OrganizationEnvironmentUpdateEnvironmentCall<'a> {
        OrganizationEnvironmentUpdateEnvironmentCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the trace configurations in an environment. Note that the repeated fields have replace semantics when included in the field mask and that they will be overwritten by the value of the fields in the request body.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the trace configuration. Use the following structure in your request: "organizations/*/environments/*/traceConfig".
    pub fn environments_update_trace_config(&self, request: GoogleCloudApigeeV1TraceConfig, name: &str) -> OrganizationEnvironmentUpdateTraceConfigCall<'a> {
        OrganizationEnvironmentUpdateTraceConfigCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Submit a query at host level to be processed in the background. If the submission of the query succeeds, the API returns a 201 status and an ID that refer to the query. In addition to the HTTP status 201, the `state` of "enqueued" means that the request succeeded.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent resource name. Must be of the form `organizations/{org}`.
    pub fn host_queries_create(&self, request: GoogleCloudApigeeV1Query, parent: &str) -> OrganizationHostQueryCreateCall<'a> {
        OrganizationHostQueryCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Get status of a query submitted at host level. If the query is still in progress, the `state` is set to "running" After the query has completed successfully, `state` is set to "completed"
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the asynchronous query to get. Must be of the form `organizations/{org}/queries/{queryId}`.
    pub fn host_queries_get(&self, name: &str) -> OrganizationHostQueryGetCall<'a> {
        OrganizationHostQueryGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// After the query is completed, use this API to retrieve the results. If the request succeeds, and there is a non-zero result set, the result is downloaded to the client as a zipped JSON file. The name of the downloaded file will be: OfflineQueryResult-.zip Example: `OfflineQueryResult-9cfc0d85-0f30-46d6-ae6f-318d0cb961bd.zip`
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the asynchronous query result to get. Must be of the form `organizations/{org}/queries/{queryId}/result`.
    pub fn host_queries_get_result(&self, name: &str) -> OrganizationHostQueryGetResultCall<'a> {
        OrganizationHostQueryGetResultCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// 
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the asynchronous query result view to get. Must be of the form `organizations/{org}/queries/{queryId}/resultView`.
    pub fn host_queries_get_result_view(&self, name: &str) -> OrganizationHostQueryGetResultViewCall<'a> {
        OrganizationHostQueryGetResultViewCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Return a list of Asynchronous Queries at host level.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The parent resource name. Must be of the form `organizations/{org}`.
    pub fn host_queries_list(&self, parent: &str) -> OrganizationHostQueryListCall<'a> {
        OrganizationHostQueryListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _to: Default::default(),
            _submitted_by: Default::default(),
            _status: Default::default(),
            _incl_queries_without_report: Default::default(),
            _from: Default::default(),
            _envgroup_hostname: Default::default(),
            _dataset: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Retrieve metrics grouped by dimensions in host level. The types of metrics you can retrieve include traffic, message counts, API call latency, response size, and cache hits and counts. Dimensions let you view metrics in meaningful groups. You can optionally pass dimensions as path parameters to the `stats` API. If dimensions are not specified, the metrics are computed on the entire set of data for the given time range.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name for which the interactive query will be executed. Use the following format in your request: `organizations/{org}/hostStats/{dimensions}` Dimensions let you view metrics in meaningful groupings, such as `apiproxy`, `target_host`. The value of dimensions should be a comma-separated list as shown below `organizations/{org}/hostStats/apiproxy,request_verb`
    pub fn host_stats_get(&self, name: &str) -> OrganizationHostStatGetCall<'a> {
        OrganizationHostStatGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _tzo: Default::default(),
            _ts_ascending: Default::default(),
            _topk: Default::default(),
            _time_unit: Default::default(),
            _time_range: Default::default(),
            _sortby: Default::default(),
            _sort: Default::default(),
            _select: Default::default(),
            _realtime: Default::default(),
            _offset: Default::default(),
            _limit: Default::default(),
            _filter: Default::default(),
            _envgroup_hostname: Default::default(),
            _accuracy: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new attachment of an environment to an instance. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`.
    pub fn instances_attachments_create(&self, request: GoogleCloudApigeeV1InstanceAttachment, parent: &str) -> OrganizationInstanceAttachmentCreateCall<'a> {
        OrganizationInstanceAttachmentCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an attachment. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the attachment. Use the following structure in your request: `organizations/{org}/instances/{instance}/attachments/{attachment}`.
    pub fn instances_attachments_delete(&self, name: &str) -> OrganizationInstanceAttachmentDeleteCall<'a> {
        OrganizationInstanceAttachmentDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets an attachment. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the attachment. Use the following structure in your request: `organizations/{org}/instances/{instance}/attachments/{attachment}`
    pub fn instances_attachments_get(&self, name: &str) -> OrganizationInstanceAttachmentGetCall<'a> {
        OrganizationInstanceAttachmentGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all attachments to an instance. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}/instances/{instance}`
    pub fn instances_attachments_list(&self, parent: &str) -> OrganizationInstanceAttachmentListCall<'a> {
        OrganizationInstanceAttachmentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new canary evaluation for an organization.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}/instances/{instance}`.
    pub fn instances_canaryevaluations_create(&self, request: GoogleCloudApigeeV1CanaryEvaluation, parent: &str) -> OrganizationInstanceCanaryevaluationCreateCall<'a> {
        OrganizationInstanceCanaryevaluationCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a CanaryEvaluation for an organization.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the CanaryEvaluation. Use the following structure in your request: `organizations/{org}/instances/*/canaryevaluations/{evaluation}`
    pub fn instances_canaryevaluations_get(&self, name: &str) -> OrganizationInstanceCanaryevaluationGetCall<'a> {
        OrganizationInstanceCanaryevaluationGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Activates the NAT address. The Apigee instance can now use this for Internet egress traffic. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the nat address. Use the following structure in your request: `organizations/{org}/instances/{instances}/natAddresses/{nataddress}``
    pub fn instances_nat_addresses_activate(&self, request: GoogleCloudApigeeV1ActivateNatAddressRequest, name: &str) -> OrganizationInstanceNatAddresseActivateCall<'a> {
        OrganizationInstanceNatAddresseActivateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a NAT address. The address is created in the RESERVED state and a static external IP address will be provisioned. At this time, the instance will not use this IP address for Internet egress traffic. The address can be activated for use once any required firewall IP whitelisting has been completed. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`
    pub fn instances_nat_addresses_create(&self, request: GoogleCloudApigeeV1NatAddress, parent: &str) -> OrganizationInstanceNatAddresseCreateCall<'a> {
        OrganizationInstanceNatAddresseCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes the NAT address. Connections that are actively using the address are drained before it is removed. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the nat address. Use the following structure in your request: `organizations/{org}/instances/{instances}/natAddresses/{nataddress}``
    pub fn instances_nat_addresses_delete(&self, name: &str) -> OrganizationInstanceNatAddresseDeleteCall<'a> {
        OrganizationInstanceNatAddresseDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the details of a NAT address. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the nat address. Use the following structure in your request: `organizations/{org}/instances/{instances}/natAddresses/{nataddress}`
    pub fn instances_nat_addresses_get(&self, name: &str) -> OrganizationInstanceNatAddresseGetCall<'a> {
        OrganizationInstanceNatAddresseGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the NAT addresses for an Apigee instance. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`
    pub fn instances_nat_addresses_list(&self, parent: &str) -> OrganizationInstanceNatAddresseListCall<'a> {
        OrganizationInstanceNatAddresseListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an Apigee runtime instance. The instance is accessible from the authorized network configured on the organization. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}`.
    pub fn instances_create(&self, request: GoogleCloudApigeeV1Instance, parent: &str) -> OrganizationInstanceCreateCall<'a> {
        OrganizationInstanceCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an Apigee runtime instance. The instance stops serving requests and the runtime data is deleted. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`.
    pub fn instances_delete(&self, name: &str) -> OrganizationInstanceDeleteCall<'a> {
        OrganizationInstanceDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the details for an Apigee runtime instance. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`.
    pub fn instances_get(&self, name: &str) -> OrganizationInstanceGetCall<'a> {
        OrganizationInstanceGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all Apigee runtime instances for the organization. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}`.
    pub fn instances_list(&self, parent: &str) -> OrganizationInstanceListCall<'a> {
        OrganizationInstanceListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates an Apigee runtime instance. You can update the fields described in NodeConfig. No other fields will be updated. **Note:** Not supported for Apigee hybrid.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the instance. Use the following structure in your request: `organizations/{org}/instances/{instance}`.
    pub fn instances_patch(&self, request: GoogleCloudApigeeV1Instance, name: &str) -> OrganizationInstancePatchCall<'a> {
        OrganizationInstancePatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _update_mask: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Reports the latest status for a runtime instance.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `instance` - The name of the instance reporting this status. For SaaS the request will be rejected if no instance exists under this name. Format is organizations/{org}/instances/{instance}
    pub fn instances_report_status(&self, request: GoogleCloudApigeeV1ReportInstanceStatusRequest, instance: &str) -> OrganizationInstanceReportStatuCall<'a> {
        OrganizationInstanceReportStatuCall {
            hub: self.hub,
            _request: request,
            _instance: instance.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a key value map in an organization.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the organization in which to create the key value map file. Use the following structure in your request: `organizations/{org}`
    pub fn keyvaluemaps_create(&self, request: GoogleCloudApigeeV1KeyValueMap, parent: &str) -> OrganizationKeyvaluemapCreateCall<'a> {
        OrganizationKeyvaluemapCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a key value map from an organization.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the key value map. Use the following structure in your request: `organizations/{org}/keyvaluemaps/{keyvaluemap}`
    pub fn keyvaluemaps_delete(&self, name: &str) -> OrganizationKeyvaluemapDeleteCall<'a> {
        OrganizationKeyvaluemapDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the latest state of a long-running operation. Clients can use this method to poll the operation result at intervals as recommended by the API service.
    /// 
    /// # Arguments
    ///
    /// * `name` - The name of the operation resource.
    pub fn operations_get(&self, name: &str) -> OrganizationOperationGetCall<'a> {
        OrganizationOperationGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists operations that match the specified filter in the request. If the server doesn't support this method, it returns `UNIMPLEMENTED`. NOTE: the `name` binding allows API services to override the binding to use different resource name schemes, such as `users/*/operations`. To override the binding, API services can add a binding such as `"/v1/{name=users/*}/operations"` to their service configuration. For backwards compatibility, the default name includes the operations collection id, however overriding users must ensure the name binding is the parent resource, without the operations collection id.
    /// 
    /// # Arguments
    ///
    /// * `name` - The name of the operation's parent resource.
    pub fn operations_list(&self, name: &str) -> OrganizationOperationListCall<'a> {
        OrganizationOperationListCall {
            hub: self.hub,
            _name: name.to_string(),
            _page_token: Default::default(),
            _page_size: Default::default(),
            _filter: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Similar to GetHostStats except that the response is less verbose.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Resource name for which the interactive query will be executed. Use the following format in your request: `organizations/{organization_id}/optimizedHostStats/{dimensions}` Dimensions let you view metrics in meaningful groupings, such as `apiproxy`, `target_host`. The value of dimensions should be a comma-separated list as shown below: `organizations/{org}/optimizedHostStats/apiproxy,request_verb`
    pub fn optimized_host_stats_get(&self, name: &str) -> OrganizationOptimizedHostStatGetCall<'a> {
        OrganizationOptimizedHostStatGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _tzo: Default::default(),
            _ts_ascending: Default::default(),
            _topk: Default::default(),
            _time_unit: Default::default(),
            _time_range: Default::default(),
            _sortby: Default::default(),
            _sort: Default::default(),
            _select: Default::default(),
            _realtime: Default::default(),
            _offset: Default::default(),
            _limit: Default::default(),
            _filter: Default::default(),
            _envgroup_hostname: Default::default(),
            _accuracy: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a Custom Report for an Organization. A Custom Report provides Apigee Customers to create custom dashboards in addition to the standard dashboards which are provided. The Custom Report in its simplest form contains specifications about metrics, dimensions and filters. It is important to note that the custom report by itself does not provide an executable entity. The Edge UI converts the custom report definition into an analytics query and displays the result in a chart.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The parent organization name under which the Custom Report will be created. Must be of the form: `organizations/{organization_id}/reports`
    pub fn reports_create(&self, request: GoogleCloudApigeeV1CustomReport, parent: &str) -> OrganizationReportCreateCall<'a> {
        OrganizationReportCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes an existing custom report definition
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Custom Report name of the form: `organizations/{organization_id}/reports/{report_name}`
    pub fn reports_delete(&self, name: &str) -> OrganizationReportDeleteCall<'a> {
        OrganizationReportDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Retrieve a custom report definition.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Custom Report name of the form: `organizations/{organization_id}/reports/{report_name}`
    pub fn reports_get(&self, name: &str) -> OrganizationReportGetCall<'a> {
        OrganizationReportGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Return a list of Custom Reports
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The parent organization name under which the API product will be listed `organizations/{organization_id}/reports`
    pub fn reports_list(&self, parent: &str) -> OrganizationReportListCall<'a> {
        OrganizationReportListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _expand: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Update an existing custom report definition
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Custom Report name of the form: `organizations/{organization_id}/reports/{report_name}`
    pub fn reports_update(&self, request: GoogleCloudApigeeV1CustomReport, name: &str) -> OrganizationReportUpdateCall<'a> {
        OrganizationReportUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of a shared flow.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the shared flow for which to return deployment information in the following format: `organizations/{org}/sharedflows/{sharedflow}`
    pub fn sharedflows_deployments_list(&self, parent: &str) -> OrganizationSharedflowDeploymentListCall<'a> {
        OrganizationSharedflowDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all deployments of a shared flow revision.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the API proxy revision for which to return deployment information in the following format: `organizations/{org}/sharedflows/{sharedflow}/revisions/{rev}`.
    pub fn sharedflows_revisions_deployments_list(&self, parent: &str) -> OrganizationSharedflowRevisionDeploymentListCall<'a> {
        OrganizationSharedflowRevisionDeploymentListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a shared flow and all associated policies, resources, and revisions. You must undeploy the shared flow before deleting it.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the shared flow revision to delete. Must be of the form: `organizations/{organization_id}/sharedflows/{shared_flow_id}/revisions/{revision_id}`
    pub fn sharedflows_revisions_delete(&self, name: &str) -> OrganizationSharedflowRevisionDeleteCall<'a> {
        OrganizationSharedflowRevisionDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a revision of a shared flow. To download the shared flow configuration bundle for the specified revision as a zip file, set the `format` query parameter to `bundle`. If you are using curl, specify `-o filename.zip` to save the output to a file; otherwise, it displays to `stdout`. Then, develop the shared flow configuration locally and upload the updated sharedFlow configuration revision, as described in [updateSharedFlowRevision](updateSharedFlowRevision).
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the shared flow revision to get. Must be of the form: `organizations/{organization_id}/sharedflows/{shared_flow_id}/revisions/{revision_id}`
    pub fn sharedflows_revisions_get(&self, name: &str) -> OrganizationSharedflowRevisionGetCall<'a> {
        OrganizationSharedflowRevisionGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _format: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a shared flow revision. This operation is only allowed on revisions which have never been deployed. After deployment a revision becomes immutable, even if it becomes undeployed. The payload is a ZIP-formatted shared flow. Content type must be either multipart/form-data or application/octet-stream.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. The name of the shared flow revision to update. Must be of the form: `organizations/{organization_id}/sharedflows/{shared_flow_id}/revisions/{revision_id}`
    pub fn sharedflows_revisions_update_shared_flow_revision(&self, request: GoogleApiHttpBody, name: &str) -> OrganizationSharedflowRevisionUpdateSharedFlowRevisionCall<'a> {
        OrganizationSharedflowRevisionUpdateSharedFlowRevisionCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _validate: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Uploads a ZIP-formatted shared flow configuration bundle to an organization. If the shared flow already exists, this creates a new revision of it. If the shared flow does not exist, this creates it. Once imported, the shared flow revision must be deployed before it can be accessed at runtime. The size limit of a shared flow bundle is 15 MB.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. The name of the parent organization under which to create the shared flow. Must be of the form: `organizations/{organization_id}`
    pub fn sharedflows_create(&self, request: GoogleApiHttpBody, parent: &str) -> OrganizationSharedflowCreateCall<'a> {
        OrganizationSharedflowCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _name: Default::default(),
            _action: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a shared flow and all it's revisions. The shared flow must be undeployed before you can delete it.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. shared flow name of the form: `organizations/{organization_id}/sharedflows/{shared_flow_id}`
    pub fn sharedflows_delete(&self, name: &str) -> OrganizationSharedflowDeleteCall<'a> {
        OrganizationSharedflowDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a shared flow by name, including a list of its revisions.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. The name of the shared flow to get. Must be of the form: `organizations/{organization_id}/sharedflows/{shared_flow_id}`
    pub fn sharedflows_get(&self, name: &str) -> OrganizationSharedflowGetCall<'a> {
        OrganizationSharedflowGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists all shared flows in the organization.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. The name of the parent organization under which to get shared flows. Must be of the form: `organizations/{organization_id}`
    pub fn sharedflows_list(&self, parent: &str) -> OrganizationSharedflowListCall<'a> {
        OrganizationSharedflowListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _include_revisions: Default::default(),
            _include_meta_data: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates a new category on the portal.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `parent` - Required. Name of the portal. Use the following structure in your request: `organizations/{org}/sites/{site}`
    pub fn sites_apicategories_create(&self, request: GoogleCloudApigeeV1ApiCategoryData, parent: &str) -> OrganizationSiteApicategoryCreateCall<'a> {
        OrganizationSiteApicategoryCreateCall {
            hub: self.hub,
            _request: request,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Deletes a category from the portal.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the category. Use the following structure in your request: `organizations/{org}/sites/{site}/apicategories/{apicategory}`
    pub fn sites_apicategories_delete(&self, name: &str) -> OrganizationSiteApicategoryDeleteCall<'a> {
        OrganizationSiteApicategoryDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets a category on the portal.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the category. Use the following structure in your request: `organizations/{org}/sites/{site}/apicategories/{apicategory}`
    pub fn sites_apicategories_get(&self, name: &str) -> OrganizationSiteApicategoryGetCall<'a> {
        OrganizationSiteApicategoryGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the categories on the portal.
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Name of the portal. Use the following structure in your request: `organizations/{org}/sites/{site}`
    pub fn sites_apicategories_list(&self, parent: &str) -> OrganizationSiteApicategoryListCall<'a> {
        OrganizationSiteApicategoryListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates a category on the portal.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the category. Use the following structure in your request: `organizations/{org}/sites/{site}/apicategories/{apicategory}`
    pub fn sites_apicategories_patch(&self, request: GoogleCloudApigeeV1ApiCategoryData, name: &str) -> OrganizationSiteApicategoryPatchCall<'a> {
        OrganizationSiteApicategoryPatchCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Creates an Apigee organization. See [Create an Apigee organization](https://cloud.google.com/apigee/docs/api-platform/get-started/create-org).
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    pub fn create(&self, request: GoogleCloudApigeeV1Organization) -> OrganizationCreateCall<'a> {
        OrganizationCreateCall {
            hub: self.hub,
            _request: request,
            _parent: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Delete an Apigee organization. Only supported for SubscriptionType TRIAL.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}`
    pub fn delete(&self, name: &str) -> OrganizationDeleteCall<'a> {
        OrganizationDeleteCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the profile for an Apigee organization. See [Understanding organizations](https://cloud.google.com/apigee/docs/api-platform/fundamentals/organization-structure).
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Apigee organization name in the following format: `organizations/{org}`
    pub fn get(&self, name: &str) -> OrganizationGetCall<'a> {
        OrganizationGetCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Gets the deployed ingress configuration for an organization.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the deployed configuration for the organization in the following format: 'organizations/{org}/deployedIngressConfig'.
    pub fn get_deployed_ingress_config(&self, name: &str) -> OrganizationGetDeployedIngressConfigCall<'a> {
        OrganizationGetDeployedIngressConfigCall {
            hub: self.hub,
            _name: name.to_string(),
            _view: Default::default(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Get runtime config for an organization.
    /// 
    /// # Arguments
    ///
    /// * `name` - Required. Name of the runtime config for the organization in the following format: 'organizations/{org}/runtimeConfig'.
    pub fn get_runtime_config(&self, name: &str) -> OrganizationGetRuntimeConfigCall<'a> {
        OrganizationGetRuntimeConfigCall {
            hub: self.hub,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the service accounts with the permissions required to allow the Synchronizer to download environment data from the control plane. An ETag is returned in the response to `getSyncAuthorization`. Pass that ETag when calling [setSyncAuthorization](setSyncAuthorization) to ensure that you are updating the correct version. If you don't pass the ETag in the call to `setSyncAuthorization`, then the existing authorization is overwritten indiscriminately. For more information, see [Configure the Synchronizer](https://cloud.google.com/apigee/docs/hybrid/latest/synchronizer-access). **Note**: Available to Apigee hybrid only.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the Apigee organization. Use the following structure in your request: `organizations/{org}`
    pub fn get_sync_authorization(&self, request: GoogleCloudApigeeV1GetSyncAuthorizationRequest, name: &str) -> OrganizationGetSyncAuthorizationCall<'a> {
        OrganizationGetSyncAuthorizationCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Lists the Apigee organizations and associated GCP projects that you have permission to access. See [Understanding organizations](https://cloud.google.com/apigee/docs/api-platform/fundamentals/organization-structure).
    /// 
    /// # Arguments
    ///
    /// * `parent` - Required. Use the following structure in your request: `organizations`
    pub fn list(&self, parent: &str) -> OrganizationListCall<'a> {
        OrganizationListCall {
            hub: self.hub,
            _parent: parent.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Configures the add-ons for the Apigee organization. The existing add-on configuration will be fully replaced.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `org` - Required. Name of the organization. Use the following structure in your request: `organizations/{org}`
    pub fn set_addons(&self, request: GoogleCloudApigeeV1SetAddonsRequest, org: &str) -> OrganizationSetAddonCall<'a> {
        OrganizationSetAddonCall {
            hub: self.hub,
            _request: request,
            _org: org.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Sets the permissions required to allow the Synchronizer to download environment data from the control plane. You must call this API to enable proper functioning of hybrid. Pass the ETag when calling `setSyncAuthorization` to ensure that you are updating the correct version. To get an ETag, call [getSyncAuthorization](getSyncAuthorization). If you don't pass the ETag in the call to `setSyncAuthorization`, then the existing authorization is overwritten indiscriminately. For more information, see [Configure the Synchronizer](https://cloud.google.com/apigee/docs/hybrid/latest/synchronizer-access). **Note**: Available to Apigee hybrid only.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Name of the Apigee organization. Use the following structure in your request: `organizations/{org}`
    pub fn set_sync_authorization(&self, request: GoogleCloudApigeeV1SyncAuthorization, name: &str) -> OrganizationSetSyncAuthorizationCall<'a> {
        OrganizationSetSyncAuthorizationCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
    
    /// Create a builder to help you perform the following task:
    ///
    /// Updates the properties for an Apigee organization. No other fields in the organization profile will be updated.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `name` - Required. Apigee organization name in the following format: `organizations/{org}`
    pub fn update(&self, request: GoogleCloudApigeeV1Organization, name: &str) -> OrganizationUpdateCall<'a> {
        OrganizationUpdateCall {
            hub: self.hub,
            _request: request,
            _name: name.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
}



/// A builder providing access to all methods supported on *project* resources.
/// It is not used directly, but through the `Apigee` hub.
///
/// # Example
///
/// Instantiate a resource builder
///
/// ```test_harness,no_run
/// extern crate hyper;
/// extern crate hyper_rustls;
/// extern crate google_apigee1 as apigee1;
/// 
/// # async fn dox() {
/// use std::default::Default;
/// use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// let secret: oauth2::ApplicationSecret = Default::default();
/// let auth = oauth2::InstalledFlowAuthenticator::builder(
///         secret,
///         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
///     ).build().await.unwrap();
/// let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // Usually you wouldn't bind this to a variable, but keep calling *CallBuilders*
/// // like `provision_organization(...)`
/// // to build up your call.
/// let rb = hub.projects();
/// # }
/// ```
pub struct ProjectMethods<'a>
    where  {

    hub: &'a Apigee<>,
}

impl<'a> client::MethodsBuilder for ProjectMethods<'a> {}

impl<'a> ProjectMethods<'a> {
    
    /// Create a builder to help you perform the following task:
    ///
    /// Provisions a new Apigee organization with a functioning runtime. This is the standard way to create trial organizations for a free Apigee trial.
    /// 
    /// # Arguments
    ///
    /// * `request` - No description provided.
    /// * `project` - Required. Name of the GCP project with which to associate the Apigee organization.
    pub fn provision_organization(&self, request: GoogleCloudApigeeV1ProvisionOrganizationRequest, project: &str) -> ProjectProvisionOrganizationCall<'a> {
        ProjectProvisionOrganizationCall {
            hub: self.hub,
            _request: request,
            _project: project.to_string(),
            _delegate: Default::default(),
            _additional_params: Default::default(),
            _scopes: Default::default(),
        }
    }
}





// ###################
// CallBuilders   ###
// #################

/// Lists hybrid services and its trusted issuers service account ids. This api is authenticated and unauthorized(allow all the users) and used by runtime authn-authz service to query control plane's issuer service account ids.
///
/// A builder for the *issuers.list* method supported by a *hybrid* resource.
/// It is not used directly, but through a `HybridMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.hybrid().issuers_list("name")
///              .doit().await;
/// # }
/// ```
pub struct HybridIssuerListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for HybridIssuerListCall<'a> {}

impl<'a> HybridIssuerListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListHybridIssuersResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.hybrid.issuers.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Must be of the form `hybrid/issuers`.
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> HybridIssuerListCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> HybridIssuerListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> HybridIssuerListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> HybridIssuerListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Create a Datastore for an org
///
/// A builder for the *analytics.datastores.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1Datastore;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1Datastore::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_create(req, "parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1Datastore,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreCreateCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Datastore)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/analytics/datastores";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1Datastore) -> OrganizationAnalyticDatastoreCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. The parent organization name. Must be of the form `organizations/{org}`.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationAnalyticDatastoreCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Delete a Datastore from an org.
///
/// A builder for the *analytics.datastores.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreDeleteCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleProtobufEmpty)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Resource name of the Datastore to be deleted. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationAnalyticDatastoreDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Get a Datastore
///
/// A builder for the *analytics.datastores.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreGetCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Datastore)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Resource name of the Datastore to be get. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationAnalyticDatastoreGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// List Datastores
///
/// A builder for the *analytics.datastores.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_list("parent")
///              .target_type("duo")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _target_type: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreListCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListDatastoresResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._target_type {
            params.push(("targetType", value.to_string()));
        }
        for &field in ["alt", "parent", "targetType"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/analytics/datastores";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. The parent organization name. Must be of the form `organizations/{org}`.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationAnalyticDatastoreListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Optional. TargetType is used to fetch all Datastores that match the type
    ///
    /// Sets the *target type* query property to the given value.
    pub fn target_type(mut self, new_value: &str) -> OrganizationAnalyticDatastoreListCall<'a> {
        self._target_type = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Test if Datastore configuration is correct. This includes checking if credentials provided by customer have required permissions in target destination storage
///
/// A builder for the *analytics.datastores.test* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1Datastore;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1Datastore::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_test(req, "parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreTestCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1Datastore,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreTestCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreTestCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1TestDatastoreResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.test",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/analytics/datastores:test";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1Datastore) -> OrganizationAnalyticDatastoreTestCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. The parent organization name Must be of the form `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationAnalyticDatastoreTestCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreTestCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreTestCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreTestCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Update a Datastore
///
/// A builder for the *analytics.datastores.update* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1Datastore;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1Datastore::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().analytics_datastores_update(req, "name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAnalyticDatastoreUpdateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1Datastore,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAnalyticDatastoreUpdateCall<'a> {}

impl<'a> OrganizationAnalyticDatastoreUpdateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Datastore)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.analytics.datastores.update",
                               http_method: hyper::Method::PUT });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::PUT).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1Datastore) -> OrganizationAnalyticDatastoreUpdateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. The resource name of datastore to be updated. Must be of the form `organizations/{org}/analytics/datastores/{datastoreId}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationAnalyticDatastoreUpdateCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAnalyticDatastoreUpdateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAnalyticDatastoreUpdateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAnalyticDatastoreUpdateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes an API product attribute.
///
/// A builder for the *apiproducts.attributes.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_attributes_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductAttributeDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductAttributeDeleteCall<'a> {}

impl<'a> OrganizationApiproductAttributeDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Attribute)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.attributes.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product attribute. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/attributes/{attribute}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductAttributeDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductAttributeDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductAttributeDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductAttributeDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets the value of an API product attribute.
///
/// A builder for the *apiproducts.attributes.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_attributes_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductAttributeGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductAttributeGetCall<'a> {}

impl<'a> OrganizationApiproductAttributeGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Attribute)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.attributes.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product attribute. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/attributes/{attribute}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductAttributeGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductAttributeGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductAttributeGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductAttributeGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all API product attributes.
///
/// A builder for the *apiproducts.attributes.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_attributes_list("parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductAttributeListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductAttributeListCall<'a> {}

impl<'a> OrganizationApiproductAttributeListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Attributes)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.attributes.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/attributes";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiproductAttributeListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductAttributeListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductAttributeListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductAttributeListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates the value of an API product attribute. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with entities also get cached for at least 180 seconds after entity is accessed during runtime. In this case, the `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
///
/// A builder for the *apiproducts.attributes.updateApiProductAttribute* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1Attribute;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1Attribute::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_attributes_update_api_product_attribute(req, "name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1Attribute,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {}

impl<'a> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Attribute)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.attributes.updateApiProductAttribute",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1Attribute) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductAttributeUpdateApiProductAttributeCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Create a rate plan that is associated with an API product in an organization. Using rate plans, API product owners can monetize their API products by configuring one or more of the following: - Billing frequency - Initial setup fees for using an API product - Payment funding model (postpaid only) - Fixed recurring or consumption-based charges for using an API product - Revenue sharing with developer partners An API product can have multiple rate plans associated with it but *only one* rate plan can be active at any point of time. **Note: From the developer's perspective, they purchase API products not rate plans.
///
/// A builder for the *apiproducts.rateplans.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1RatePlan;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1RatePlan::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_rateplans_create(req, "parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductRateplanCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1RatePlan,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductRateplanCreateCall<'a> {}

impl<'a> OrganizationApiproductRateplanCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1RatePlan)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.rateplans.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/rateplans";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1RatePlan) -> OrganizationApiproductRateplanCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the API product that is associated with the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiproductRateplanCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductRateplanCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductRateplanCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductRateplanCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes a rate plan.
///
/// A builder for the *apiproducts.rateplans.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_rateplans_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductRateplanDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductRateplanDeleteCall<'a> {}

impl<'a> OrganizationApiproductRateplanDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1RatePlan)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.rateplans.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. ID of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductRateplanDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductRateplanDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductRateplanDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductRateplanDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets the details of a rate plan.
///
/// A builder for the *apiproducts.rateplans.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_rateplans_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductRateplanGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductRateplanGetCall<'a> {}

impl<'a> OrganizationApiproductRateplanGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1RatePlan)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.rateplans.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductRateplanGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductRateplanGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductRateplanGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductRateplanGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all the rate plans for an API product.
///
/// A builder for the *apiproducts.rateplans.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_rateplans_list("parent")
///              .state("duo")
///              .start_key("ipsum")
///              .order_by("sed")
///              .expand(true)
///              .count(-50)
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductRateplanListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _state: Option<String>,
    _start_key: Option<String>,
    _order_by: Option<String>,
    _expand: Option<bool>,
    _count: Option<i32>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductRateplanListCall<'a> {}

impl<'a> OrganizationApiproductRateplanListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListRatePlansResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.rateplans.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(8 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._state {
            params.push(("state", value.to_string()));
        }
        if let Some(value) = self._start_key {
            params.push(("startKey", value.to_string()));
        }
        if let Some(value) = self._order_by {
            params.push(("orderBy", value.to_string()));
        }
        if let Some(value) = self._expand {
            params.push(("expand", value.to_string()));
        }
        if let Some(value) = self._count {
            params.push(("count", value.to_string()));
        }
        for &field in ["alt", "parent", "state", "startKey", "orderBy", "expand", "count"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/rateplans";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}` Use `organizations/{org}/apiproducts/-` to return rate plans for all API products within the organization.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiproductRateplanListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// State of the rate plans (`DRAFT`, `PUBLISHED`) that you want to display.
    ///
    /// Sets the *state* query property to the given value.
    pub fn state(mut self, new_value: &str) -> OrganizationApiproductRateplanListCall<'a> {
        self._state = Some(new_value.to_string());
        self
    }
    /// Name of the rate plan from which to start displaying the list of rate plans. If omitted, the list starts from the first item. For example, to view the rate plans from 51-150, set the value of `startKey` to the name of the 51st rate plan and set the value of `count` to 100.
    ///
    /// Sets the *start key* query property to the given value.
    pub fn start_key(mut self, new_value: &str) -> OrganizationApiproductRateplanListCall<'a> {
        self._start_key = Some(new_value.to_string());
        self
    }
    /// Name of the attribute used for sorting. Valid values include: * `name`: Name of the rate plan. * `state`: State of the rate plan (`DRAFT`, `PUBLISHED`). * `startTime`: Time when the rate plan becomes active. * `endTime`: Time when the rate plan expires. **Note**: Not supported by Apigee at this time.
    ///
    /// Sets the *order by* query property to the given value.
    pub fn order_by(mut self, new_value: &str) -> OrganizationApiproductRateplanListCall<'a> {
        self._order_by = Some(new_value.to_string());
        self
    }
    /// Flag that specifies whether to expand the results. Set to `true` to get expanded details about each API. Defaults to `false`.
    ///
    /// Sets the *expand* query property to the given value.
    pub fn expand(mut self, new_value: bool) -> OrganizationApiproductRateplanListCall<'a> {
        self._expand = Some(new_value);
        self
    }
    /// Number of rate plans to return in the API call. Use with the `startKey` parameter to provide more targeted filtering. The maximum limit is 1000. Defaults to 100.
    ///
    /// Sets the *count* query property to the given value.
    pub fn count(mut self, new_value: i32) -> OrganizationApiproductRateplanListCall<'a> {
        self._count = Some(new_value);
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductRateplanListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductRateplanListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductRateplanListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates an existing rate plan.
///
/// A builder for the *apiproducts.rateplans.update* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1RatePlan;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1RatePlan::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_rateplans_update(req, "name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductRateplanUpdateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1RatePlan,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductRateplanUpdateCall<'a> {}

impl<'a> OrganizationApiproductRateplanUpdateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1RatePlan)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.rateplans.update",
                               http_method: hyper::Method::PUT });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::PUT).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1RatePlan) -> OrganizationApiproductRateplanUpdateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the rate plan. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}/rateplans/{rateplan}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductRateplanUpdateCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductRateplanUpdateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductRateplanUpdateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductRateplanUpdateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates or creates API product attributes. This API **replaces** the current list of attributes with the attributes specified in the request body. In this way, you can update existing attributes, add new attributes, or delete existing attributes by omitting them from the request body. **Note**: OAuth access tokens and Key Management Service (KMS) entities (apps, developers, and API products) are cached for 180 seconds (current default). Any custom attributes associated with entities also get cached for at least 180 seconds after entity is accessed during runtime. In this case, the `ExpiresIn` element on the OAuthV2 policy won't be able to expire an access token in less than 180 seconds.
///
/// A builder for the *apiproducts.attributes* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1Attributes;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1Attributes::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_attributes(req, "name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductAttributeCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1Attributes,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductAttributeCall<'a> {}

impl<'a> OrganizationApiproductAttributeCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1Attributes)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.attributes",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}/attributes";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1Attributes) -> OrganizationApiproductAttributeCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductAttributeCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductAttributeCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductAttributeCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductAttributeCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Creates an API product in an organization. You create API products after you have proxied backend services using API proxies. An API product is a collection of API resources combined with quota settings and metadata that you can use to deliver customized and productized API bundles to your developer community. This metadata can include: - Scope - Environments - API proxies - Extensible profile API products enable you repackage APIs on the fly, without having to do any additional coding or configuration. Apigee recommends that you start with a simple API product including only required elements. You then provision credentials to apps to enable them to start testing your APIs. After you have authentication and authorization working against a simple API product, you can iterate to create finer-grained API products, defining different sets of API resources for each API product. **WARNING:** - If you don't specify an API proxy in the request body, *any* app associated with the product can make calls to *any* API in your entire organization. - If you don't specify an environment in the request body, the product allows access to all environments. For more information, see What is an API product?
///
/// A builder for the *apiproducts.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1ApiProduct;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1ApiProduct::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_create(req, "parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1ApiProduct,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductCreateCall<'a> {}

impl<'a> OrganizationApiproductCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProduct)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/apiproducts";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1ApiProduct) -> OrganizationApiproductCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the organization in which the API product will be created. Use the following structure in your request: `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiproductCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes an API product from an organization. Deleting an API product causes app requests to the resource URIs defined in the API product to fail. Ensure that you create a new API product to serve existing apps, unless your intention is to disable access to the resources defined in the API product. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via the UI or the API. View the list of API products to verify the internal name.
///
/// A builder for the *apiproducts.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductDeleteCall<'a> {}

impl<'a> OrganizationApiproductDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProduct)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets configuration details for an API product. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via the UI or the API. View the list of API products to verify the internal name.
///
/// A builder for the *apiproducts.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductGetCall<'a> {}

impl<'a> OrganizationApiproductGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProduct)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all API product names for an organization. Filter the list by passing an `attributename` and `attibutevalue`. The maximum number of API products returned is 1000. You can paginate the list of API products returned using the `startKey` and `count` query parameters.
///
/// A builder for the *apiproducts.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_list("parent")
///              .start_key("eos")
///              .expand(false)
///              .count("sed")
///              .attributevalue("duo")
///              .attributename("sed")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _start_key: Option<String>,
    _expand: Option<bool>,
    _count: Option<String>,
    _attributevalue: Option<String>,
    _attributename: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductListCall<'a> {}

impl<'a> OrganizationApiproductListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListApiProductsResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(8 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._start_key {
            params.push(("startKey", value.to_string()));
        }
        if let Some(value) = self._expand {
            params.push(("expand", value.to_string()));
        }
        if let Some(value) = self._count {
            params.push(("count", value.to_string()));
        }
        if let Some(value) = self._attributevalue {
            params.push(("attributevalue", value.to_string()));
        }
        if let Some(value) = self._attributename {
            params.push(("attributename", value.to_string()));
        }
        for &field in ["alt", "parent", "startKey", "expand", "count", "attributevalue", "attributename"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/apiproducts";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the organization. Use the following structure in your request: `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiproductListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Gets a list of API products starting with a specific API product in the list. For example, if you're returning 50 API products at a time (using the `count` query parameter), you can view products 50-99 by entering the name of the 50th API product in the first API (without using `startKey`). Product name is case sensitive.
    ///
    /// Sets the *start key* query property to the given value.
    pub fn start_key(mut self, new_value: &str) -> OrganizationApiproductListCall<'a> {
        self._start_key = Some(new_value.to_string());
        self
    }
    /// Flag that specifies whether to expand the results. Set to `true` to get expanded details about each API.
    ///
    /// Sets the *expand* query property to the given value.
    pub fn expand(mut self, new_value: bool) -> OrganizationApiproductListCall<'a> {
        self._expand = Some(new_value);
        self
    }
    /// Enter the number of API products you want returned in the API call. The limit is 1000.
    ///
    /// Sets the *count* query property to the given value.
    pub fn count(mut self, new_value: &str) -> OrganizationApiproductListCall<'a> {
        self._count = Some(new_value.to_string());
        self
    }
    /// Value of the attribute used to filter the search.
    ///
    /// Sets the *attributevalue* query property to the given value.
    pub fn attributevalue(mut self, new_value: &str) -> OrganizationApiproductListCall<'a> {
        self._attributevalue = Some(new_value.to_string());
        self
    }
    /// Name of the attribute used to filter the search.
    ///
    /// Sets the *attributename* query property to the given value.
    pub fn attributename(mut self, new_value: &str) -> OrganizationApiproductListCall<'a> {
        self._attributename = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates an existing API product. You must include all required values, whether or not you are updating them, as well as any optional values that you are updating. The API product name required in the request URL is the internal name of the product, not the display name. While they may be the same, it depends on whether the API product was created via UI or API. View the list of API products to identify their internal names.
///
/// A builder for the *apiproducts.update* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1ApiProduct;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1ApiProduct::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apiproducts_update(req, "name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiproductUpdateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1ApiProduct,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiproductUpdateCall<'a> {}

impl<'a> OrganizationApiproductUpdateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProduct)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apiproducts.update",
                               http_method: hyper::Method::PUT });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::PUT).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1ApiProduct) -> OrganizationApiproductUpdateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the API product. Use the following structure in your request: `organizations/{org}/apiproducts/{apiproduct}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiproductUpdateCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiproductUpdateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiproductUpdateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiproductUpdateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all deployments of an API proxy.
///
/// A builder for the *apis.deployments.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_deployments_list("parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiDeploymentListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiDeploymentListCall<'a> {}

impl<'a> OrganizationApiDeploymentListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListDeploymentsResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.deployments.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/deployments";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API proxy for which to return deployment information in the following format: `organizations/{org}/apis/{api}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiDeploymentListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiDeploymentListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiDeploymentListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiDeploymentListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Creates a key value map in an API proxy.
///
/// A builder for the *apis.keyvaluemaps.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1KeyValueMap;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1KeyValueMap::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_keyvaluemaps_create(req, "parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiKeyvaluemapCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1KeyValueMap,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiKeyvaluemapCreateCall<'a> {}

impl<'a> OrganizationApiKeyvaluemapCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1KeyValueMap)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.keyvaluemaps.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/keyvaluemaps";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1KeyValueMap) -> OrganizationApiKeyvaluemapCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the environment in which to create the key value map. Use the following structure in your request: `organizations/{org}/apis/{api}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiKeyvaluemapCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiKeyvaluemapCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiKeyvaluemapCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiKeyvaluemapCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes a key value map from an API proxy.
///
/// A builder for the *apis.keyvaluemaps.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_keyvaluemaps_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiKeyvaluemapDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiKeyvaluemapDeleteCall<'a> {}

impl<'a> OrganizationApiKeyvaluemapDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1KeyValueMap)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.keyvaluemaps.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the key value map. Use the following structure in your request: `organizations/{org}/apis/{api}/keyvaluemaps/{keyvaluemap}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiKeyvaluemapDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiKeyvaluemapDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiKeyvaluemapDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiKeyvaluemapDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all deployments of an API proxy revision.
///
/// A builder for the *apis.revisions.deployments.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_revisions_deployments_list("parent")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiRevisionDeploymentListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiRevisionDeploymentListCall<'a> {}

impl<'a> OrganizationApiRevisionDeploymentListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListDeploymentsResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.revisions.deployments.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        for &field in ["alt", "parent"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/deployments";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API proxy revision for which to return deployment information in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiRevisionDeploymentListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiRevisionDeploymentListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiRevisionDeploymentListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiRevisionDeploymentListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes an API proxy revision and all policies, resources, endpoints, and revisions associated with it. The API proxy revision must be undeployed before you can delete it.
///
/// A builder for the *apis.revisions.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_revisions_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiRevisionDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiRevisionDeleteCall<'a> {}

impl<'a> OrganizationApiRevisionDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxyRevision)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.revisions.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. API proxy revision in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiRevisionDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiRevisionDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiRevisionDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiRevisionDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets an API proxy revision. To download the API proxy configuration bundle for the specified revision as a zip file, set the `format` query parameter to `bundle`. If you are using curl, specify `-o filename.zip` to save the output to a file; otherwise, it displays to `stdout`. Then, develop the API proxy configuration locally and upload the updated API proxy configuration revision, as described in [updateApiProxyRevision](updateApiProxyRevision).
///
/// A builder for the *apis.revisions.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_revisions_get("name")
///              .format("vero")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiRevisionGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _format: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiRevisionGetCall<'a> {}

impl<'a> OrganizationApiRevisionGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleApiHttpBody)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.revisions.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(4 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        if let Some(value) = self._format {
            params.push(("format", value.to_string()));
        }
        for &field in ["alt", "name", "format"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. API proxy revision in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiRevisionGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// Format used when downloading the API proxy configuration revision. Set to `bundle` to download the API proxy configuration revision as a zip file.
    ///
    /// Sets the *format* query property to the given value.
    pub fn format(mut self, new_value: &str) -> OrganizationApiRevisionGetCall<'a> {
        self._format = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiRevisionGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiRevisionGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiRevisionGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates an existing API proxy revision by uploading the API proxy configuration bundle as a zip file from your local machine. You can update only API proxy revisions that have never been deployed. After deployment, an API proxy revision becomes immutable, even if it is undeployed. Set the `Content-Type` header to either `multipart/form-data` or `application/octet-stream`.
///
/// A builder for the *apis.revisions.updateApiProxyRevision* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleApiHttpBody;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleApiHttpBody::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_revisions_update_api_proxy_revision(req, "name")
///              .validate(false)
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiRevisionUpdateApiProxyRevisionCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleApiHttpBody,
    _name: String,
    _validate: Option<bool>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {}

impl<'a> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxyRevision)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.revisions.updateApiProxyRevision",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(5 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        if let Some(value) = self._validate {
            params.push(("validate", value.to_string()));
        }
        for &field in ["alt", "name", "validate"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleApiHttpBody) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. API proxy revision to update in the following format: `organizations/{org}/apis/{api}/revisions/{rev}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// Ignored. All uploads are validated regardless of the value of this field. Maintained for compatibility with Apigee Edge API.
    ///
    /// Sets the *validate* query property to the given value.
    pub fn validate(mut self, new_value: bool) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {
        self._validate = Some(new_value);
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiRevisionUpdateApiProxyRevisionCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Creates an API proxy. The API proxy created will not be accessible at runtime until it is deployed to an environment. Create a new API proxy by setting the `name` query parameter to the name of the API proxy. Import an API proxy configuration bundle stored in zip format on your local machine to your organization by doing the following: * Set the `name` query parameter to the name of the API proxy. * Set the `action` query parameter to `import`. * Set the `Content-Type` header to `multipart/form-data`. * Pass as a file the name of API proxy configuration bundle stored in zip format on your local machine using the `file` form field. **Note**: To validate the API proxy configuration bundle only without importing it, set the `action` query parameter to `validate`. When importing an API proxy configuration bundle, if the API proxy does not exist, it will be created. If the API proxy exists, then a new revision is created. Invalid API proxy configurations are rejected, and a list of validation errors is returned to the client.
///
/// A builder for the *apis.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleApiHttpBody;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleApiHttpBody::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_create(req, "parent")
///              .validate(false)
///              .name("diam")
///              .action("dolor")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleApiHttpBody,
    _parent: String,
    _validate: Option<bool>,
    _name: Option<String>,
    _action: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiCreateCall<'a> {}

impl<'a> OrganizationApiCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxyRevision)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(7 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._validate {
            params.push(("validate", value.to_string()));
        }
        if let Some(value) = self._name {
            params.push(("name", value.to_string()));
        }
        if let Some(value) = self._action {
            params.push(("action", value.to_string()));
        }
        for &field in ["alt", "parent", "validate", "name", "action"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/apis";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleApiHttpBody) -> OrganizationApiCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the organization in the following format: `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Ignored. All uploads are validated regardless of the value of this field. Maintained for compatibility with Apigee Edge API.
    ///
    /// Sets the *validate* query property to the given value.
    pub fn validate(mut self, new_value: bool) -> OrganizationApiCreateCall<'a> {
        self._validate = Some(new_value);
        self
    }
    /// Name of the API proxy. Restrict the characters used to: A-Za-z0-9._-
    ///
    /// Sets the *name* query property to the given value.
    pub fn name(mut self, new_value: &str) -> OrganizationApiCreateCall<'a> {
        self._name = Some(new_value.to_string());
        self
    }
    /// Action to perform when importing an API proxy configuration bundle. Set this parameter to one of the following values: * `import` to import the API proxy configuration bundle. * `validate` to validate the API proxy configuration bundle without importing it.
    ///
    /// Sets the *action* query property to the given value.
    pub fn action(mut self, new_value: &str) -> OrganizationApiCreateCall<'a> {
        self._action = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes an API proxy and all associated endpoints, policies, resources, and revisions. The API proxy must be undeployed before you can delete it.
///
/// A builder for the *apis.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiDeleteCall<'a> {}

impl<'a> OrganizationApiDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxy)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API proxy in the following format: `organizations/{org}/apis/{api}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets an API proxy including a list of existing revisions.
///
/// A builder for the *apis.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiGetCall<'a> {}

impl<'a> OrganizationApiGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxy)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the API proxy in the following format: `organizations/{org}/apis/{api}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists the names of all API proxies in an organization. The names returned correspond to the names defined in the configuration files for each API proxy.
///
/// A builder for the *apis.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_list("parent")
///              .include_revisions(false)
///              .include_meta_data(false)
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _include_revisions: Option<bool>,
    _include_meta_data: Option<bool>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiListCall<'a> {}

impl<'a> OrganizationApiListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListApiProxiesResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(5 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._include_revisions {
            params.push(("includeRevisions", value.to_string()));
        }
        if let Some(value) = self._include_meta_data {
            params.push(("includeMetaData", value.to_string()));
        }
        for &field in ["alt", "parent", "includeRevisions", "includeMetaData"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/apis";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the organization in the following format: `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationApiListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Flag that specifies whether to include a list of revisions in the response.
    ///
    /// Sets the *include revisions* query property to the given value.
    pub fn include_revisions(mut self, new_value: bool) -> OrganizationApiListCall<'a> {
        self._include_revisions = Some(new_value);
        self
    }
    /// Flag that specifies whether to include API proxy metadata in the response.
    ///
    /// Sets the *include meta data* query property to the given value.
    pub fn include_meta_data(mut self, new_value: bool) -> OrganizationApiListCall<'a> {
        self._include_meta_data = Some(new_value);
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Updates an existing API proxy.
///
/// A builder for the *apis.patch* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1ApiProxy;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1ApiProxy::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apis_patch(req, "name")
///              .update_mask("invidunt")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationApiPatchCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1ApiProxy,
    _name: String,
    _update_mask: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationApiPatchCall<'a> {}

impl<'a> OrganizationApiPatchCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ApiProxy)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apis.patch",
                               http_method: hyper::Method::PATCH });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(5 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        if let Some(value) = self._update_mask {
            params.push(("updateMask", value.to_string()));
        }
        for &field in ["alt", "name", "updateMask"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::PATCH).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1ApiProxy) -> OrganizationApiPatchCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. API proxy to update in the following format: `organizations/{org}/apis/{api}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationApiPatchCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// Required. The list of fields to update.
    ///
    /// Sets the *update mask* query property to the given value.
    pub fn update_mask(mut self, new_value: &str) -> OrganizationApiPatchCall<'a> {
        self._update_mask = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationApiPatchCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationApiPatchCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationApiPatchCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets the app profile for the specified app ID.
///
/// A builder for the *apps.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apps_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAppGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAppGetCall<'a> {}

impl<'a> OrganizationAppGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1App)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apps.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. App ID in the following format: `organizations/{org}/apps/{app}`
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationAppGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAppGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAppGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAppGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists IDs of apps within an organization that have the specified app status (approved or revoked) or are of the specified app type (developer or company).
///
/// A builder for the *apps.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().apps_list("parent")
///              .status("elitr")
///              .start_key("Lorem")
///              .rows("diam")
///              .key_status("no")
///              .include_cred(false)
///              .ids("accusam")
///              .expand(true)
///              .apptype("consetetur")
///              .api_product("voluptua.")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationAppListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _status: Option<String>,
    _start_key: Option<String>,
    _rows: Option<String>,
    _key_status: Option<String>,
    _include_cred: Option<bool>,
    _ids: Option<String>,
    _expand: Option<bool>,
    _apptype: Option<String>,
    _api_product: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationAppListCall<'a> {}

impl<'a> OrganizationAppListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListAppsResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.apps.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(12 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._status {
            params.push(("status", value.to_string()));
        }
        if let Some(value) = self._start_key {
            params.push(("startKey", value.to_string()));
        }
        if let Some(value) = self._rows {
            params.push(("rows", value.to_string()));
        }
        if let Some(value) = self._key_status {
            params.push(("keyStatus", value.to_string()));
        }
        if let Some(value) = self._include_cred {
            params.push(("includeCred", value.to_string()));
        }
        if let Some(value) = self._ids {
            params.push(("ids", value.to_string()));
        }
        if let Some(value) = self._expand {
            params.push(("expand", value.to_string()));
        }
        if let Some(value) = self._apptype {
            params.push(("apptype", value.to_string()));
        }
        if let Some(value) = self._api_product {
            params.push(("apiProduct", value.to_string()));
        }
        for &field in ["alt", "parent", "status", "startKey", "rows", "keyStatus", "includeCred", "ids", "expand", "apptype", "apiProduct"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/apps";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Resource path of the parent in the following format: `organizations/{org}`
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Optional. Filter by the status of the app. Valid values are `approved` or `revoked`. Defaults to `approved`.
    ///
    /// Sets the *status* query property to the given value.
    pub fn status(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._status = Some(new_value.to_string());
        self
    }
    /// Returns the list of apps starting from the specified app ID.
    ///
    /// Sets the *start key* query property to the given value.
    pub fn start_key(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._start_key = Some(new_value.to_string());
        self
    }
    /// Optional. Maximum number of app IDs to return. Defaults to 10000.
    ///
    /// Sets the *rows* query property to the given value.
    pub fn rows(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._rows = Some(new_value.to_string());
        self
    }
    /// Optional. Key status of the app. Valid values include `approved` or `revoked`. Defaults to `approved`.
    ///
    /// Sets the *key status* query property to the given value.
    pub fn key_status(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._key_status = Some(new_value.to_string());
        self
    }
    /// Optional. Flag that specifies whether to include credentials in the response.
    ///
    /// Sets the *include cred* query property to the given value.
    pub fn include_cred(mut self, new_value: bool) -> OrganizationAppListCall<'a> {
        self._include_cred = Some(new_value);
        self
    }
    /// Optional. Comma-separated list of app IDs on which to filter.
    ///
    /// Sets the *ids* query property to the given value.
    pub fn ids(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._ids = Some(new_value.to_string());
        self
    }
    /// Optional. Flag that specifies whether to return an expanded list of apps for the organization. Defaults to `false`.
    ///
    /// Sets the *expand* query property to the given value.
    pub fn expand(mut self, new_value: bool) -> OrganizationAppListCall<'a> {
        self._expand = Some(new_value);
        self
    }
    /// Optional. Filter by the type of the app. Valid values are `company` or `developer`. Defaults to `developer`.
    ///
    /// Sets the *apptype* query property to the given value.
    pub fn apptype(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._apptype = Some(new_value.to_string());
        self
    }
    /// API product.
    ///
    /// Sets the *api product* query property to the given value.
    pub fn api_product(mut self, new_value: &str) -> OrganizationAppListCall<'a> {
        self._api_product = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationAppListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationAppListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationAppListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Creates a new data collector.
///
/// A builder for the *datacollectors.create* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// use apigee1::api::GoogleCloudApigeeV1DataCollector;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // As the method needs a request, you would usually fill it with the desired information
/// // into the respective structure. Some of the parts shown here might not be applicable !
/// // Values shown here are possibly random and not representative !
/// let mut req = GoogleCloudApigeeV1DataCollector::default();
/// 
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().datacollectors_create(req, "parent")
///              .data_collector_id("erat")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationDatacollectorCreateCall<'a>
    where  {

    hub: &'a Apigee<>,
    _request: GoogleCloudApigeeV1DataCollector,
    _parent: String,
    _data_collector_id: Option<String>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationDatacollectorCreateCall<'a> {}

impl<'a> OrganizationDatacollectorCreateCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1DataCollector)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.datacollectors.create",
                               http_method: hyper::Method::POST });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(5 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._data_collector_id {
            params.push(("dataCollectorId", value.to_string()));
        }
        for &field in ["alt", "parent", "dataCollectorId"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/datacollectors";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();

        let mut json_mime_type: mime::Mime = "application/json".parse().unwrap();
        let mut request_value_reader =
            {
                let mut value = json::value::to_value(&self._request).expect("serde to work");
                client::remove_json_null_values(&mut value);
                let mut dst = io::Cursor::new(Vec::with_capacity(128));
                json::to_writer(&mut dst, &value).unwrap();
                dst
            };
        let request_size = request_value_reader.seek(io::SeekFrom::End(0)).unwrap();
        request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();


        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            request_value_reader.seek(io::SeekFrom::Start(0)).unwrap();
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::POST).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .header(CONTENT_TYPE, format!("{}", json_mime_type.to_string()))
                        .header(CONTENT_LENGTH, request_size as u64)
                        .body(hyper::body::Body::from(request_value_reader.get_ref().clone()));

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    ///
    /// Sets the *request* property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn request(mut self, new_value: GoogleCloudApigeeV1DataCollector) -> OrganizationDatacollectorCreateCall<'a> {
        self._request = new_value;
        self
    }
    /// Required. Name of the organization in which to create the data collector in the following format: `organizations/{org}`.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationDatacollectorCreateCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// ID of the data collector. Overrides any ID in the data collector resource. Must be a string beginning with `dc_` that contains only letters, numbers, and underscores.
    ///
    /// Sets the *data collector id* query property to the given value.
    pub fn data_collector_id(mut self, new_value: &str) -> OrganizationDatacollectorCreateCall<'a> {
        self._data_collector_id = Some(new_value.to_string());
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationDatacollectorCreateCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationDatacollectorCreateCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationDatacollectorCreateCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Deletes a data collector.
///
/// A builder for the *datacollectors.delete* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().datacollectors_delete("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationDatacollectorDeleteCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationDatacollectorDeleteCall<'a> {}

impl<'a> OrganizationDatacollectorDeleteCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleProtobufEmpty)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.datacollectors.delete",
                               http_method: hyper::Method::DELETE });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::DELETE).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the data collector in the following format: `organizations/{org}/datacollectors/{data_collector_id}`.
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationDatacollectorDeleteCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationDatacollectorDeleteCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationDatacollectorDeleteCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationDatacollectorDeleteCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Gets a data collector.
///
/// A builder for the *datacollectors.get* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().datacollectors_get("name")
///              .doit().await;
/// # }
/// ```
pub struct OrganizationDatacollectorGetCall<'a>
    where  {

    hub: &'a Apigee<>,
    _name: String,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationDatacollectorGetCall<'a> {}

impl<'a> OrganizationDatacollectorGetCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1DataCollector)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.datacollectors.get",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(3 + self._additional_params.len());
        params.push(("name", self._name.to_string()));
        for &field in ["alt", "name"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+name}";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+name}", "name")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["name"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the data collector in the following format: `organizations/{org}/datacollectors/{data_collector_id}`.
    ///
    /// Sets the *name* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn name(mut self, new_value: &str) -> OrganizationDatacollectorGetCall<'a> {
        self._name = new_value.to_string();
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationDatacollectorGetCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationDatacollectorGetCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationDatacollectorGetCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
    }
}


/// Lists all data collectors.
///
/// A builder for the *datacollectors.list* method supported by a *organization* resource.
/// It is not used directly, but through a `OrganizationMethods` instance.
///
/// # Example
///
/// Instantiate a resource method builder
///
/// ```test_harness,no_run
/// # extern crate hyper;
/// # extern crate hyper_rustls;
/// # extern crate google_apigee1 as apigee1;
/// # async fn dox() {
/// # use std::default::Default;
/// # use apigee1::{Apigee, oauth2, hyper, hyper_rustls};
/// 
/// # let secret: oauth2::ApplicationSecret = Default::default();
/// # let auth = oauth2::InstalledFlowAuthenticator::builder(
/// #         secret,
/// #         oauth2::InstalledFlowReturnMethod::HTTPRedirect,
/// #     ).build().await.unwrap();
/// # let mut hub = Apigee::new(hyper::Client::builder().build(hyper_rustls::HttpsConnector::with_native_roots()), auth);
/// // You can configure optional parameters by calling the respective setters at will, and
/// // execute the final call using `doit()`.
/// // Values shown here are possibly random and not representative !
/// let result = hub.organizations().datacollectors_list("parent")
///              .page_token("takimata")
///              .page_size(-19)
///              .doit().await;
/// # }
/// ```
pub struct OrganizationDatacollectorListCall<'a>
    where  {

    hub: &'a Apigee<>,
    _parent: String,
    _page_token: Option<String>,
    _page_size: Option<i32>,
    _delegate: Option<&'a mut dyn client::Delegate>,
    _additional_params: HashMap<String, String>,
    _scopes: BTreeMap<String, ()>
}

impl<'a> client::CallBuilder for OrganizationDatacollectorListCall<'a> {}

impl<'a> OrganizationDatacollectorListCall<'a> {


    /// Perform the operation you have build so far.
    pub async fn doit(mut self) -> client::Result<(hyper::Response<hyper::body::Body>, GoogleCloudApigeeV1ListDataCollectorsResponse)> {
        use url::percent_encoding::{percent_encode, DEFAULT_ENCODE_SET};
        use std::io::{Read, Seek};
        use hyper::header::{CONTENT_TYPE, CONTENT_LENGTH, AUTHORIZATION, USER_AGENT, LOCATION};
        use client::ToParts;
        let mut dd = client::DefaultDelegate;
        let mut dlg: &mut dyn client::Delegate = match self._delegate {
            Some(d) => d,
            None => &mut dd
        };
        dlg.begin(client::MethodInfo { id: "apigee.organizations.datacollectors.list",
                               http_method: hyper::Method::GET });
        let mut params: Vec<(&str, String)> = Vec::with_capacity(5 + self._additional_params.len());
        params.push(("parent", self._parent.to_string()));
        if let Some(value) = self._page_token {
            params.push(("pageToken", value.to_string()));
        }
        if let Some(value) = self._page_size {
            params.push(("pageSize", value.to_string()));
        }
        for &field in ["alt", "parent", "pageToken", "pageSize"].iter() {
            if self._additional_params.contains_key(field) {
                dlg.finished(false);
                return Err(client::Error::FieldClash(field));
            }
        }
        for (name, value) in self._additional_params.iter() {
            params.push((&name, value.clone()));
        }

        params.push(("alt", "json".to_string()));

        let mut url = self.hub._base_url.clone() + "v1/{+parent}/datacollectors";
        if self._scopes.len() == 0 {
            self._scopes.insert(Scope::CloudPlatform.as_ref().to_string(), ());
        }

        for &(find_this, param_name) in [("{+parent}", "parent")].iter() {
            let mut replace_with = String::new();
            for &(name, ref value) in params.iter() {
                if name == param_name {
                    replace_with = value.to_string();
                    break;
                }
            }
            if find_this.as_bytes()[1] == '+' as u8 {
                replace_with = percent_encode(replace_with.as_bytes(), DEFAULT_ENCODE_SET).to_string();
            }
            url = url.replace(find_this, &replace_with);
        }
        {
            let mut indices_for_removal: Vec<usize> = Vec::with_capacity(1);
            for param_name in ["parent"].iter() {
                if let Some(index) = params.iter().position(|t| &t.0 == param_name) {
                    indices_for_removal.push(index);
                }
            }
            for &index in indices_for_removal.iter() {
                params.remove(index);
            }
        }

        let url = url::Url::parse_with_params(&url, params).unwrap();



        loop {
            let token = match self.hub.auth.token(&self._scopes.keys().collect::<Vec<_>>()[..]).await {
                Ok(token) => token.clone(),
                Err(err) => {
                    match  dlg.token(&err) {
                        Some(token) => token,
                        None => {
                            dlg.finished(false);
                            return Err(client::Error::MissingToken(err))
                        }
                    }
                }
            };
            let mut req_result = {
                let client = &self.hub.client;
                dlg.pre_request();
                let mut req_builder = hyper::Request::builder().method(hyper::Method::GET).uri(url.clone().into_string())
                        .header(USER_AGENT, self.hub._user_agent.clone())                            .header(AUTHORIZATION, format!("Bearer {}", token.as_str()));


                        let request = req_builder
                        .body(hyper::body::Body::empty());

                client.request(request.unwrap()).await
                
            };

            match req_result {
                Err(err) => {
                    if let client::Retry::After(d) = dlg.http_error(&err) {
                        sleep(d);
                        continue;
                    }
                    dlg.finished(false);
                    return Err(client::Error::HttpError(err))
                }
                Ok(mut res) => {
                    if !res.status().is_success() {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;
                        let (parts, _) = res.into_parts();
                        let body = hyper::Body::from(res_body_string.clone());
                        let restored_response = hyper::Response::from_parts(parts, body);

                        let server_response = json::from_str::<serde_json::Value>(&res_body_string).ok();

                        if let client::Retry::After(d) = dlg.http_failure(&restored_response, server_response.clone()) {
                            sleep(d);
                            continue;
                        }

                        dlg.finished(false);

                        return match server_response {
                            Some(error_value) => Err(client::Error::BadRequest(error_value)),
                            None => Err(client::Error::Failure(restored_response)),
                        }
                    }
                    let result_value = {
                        let res_body_string = client::get_body_as_string(res.body_mut()).await;

                        match json::from_str(&res_body_string) {
                            Ok(decoded) => (res, decoded),
                            Err(err) => {
                                dlg.response_json_decode_error(&res_body_string, &err);
                                return Err(client::Error::JsonDecodeError(res_body_string, err));
                            }
                        }
                    };

                    dlg.finished(true);
                    return Ok(result_value)
                }
            }
        }
    }


    /// Required. Name of the organization for which to list data collectors in the following format: `organizations/{org}`.
    ///
    /// Sets the *parent* path property to the given value.
    ///
    /// Even though the property as already been set when instantiating this call,
    /// we provide this method for API completeness.
    pub fn parent(mut self, new_value: &str) -> OrganizationDatacollectorListCall<'a> {
        self._parent = new_value.to_string();
        self
    }
    /// Page token, returned from a previous ListDataCollectors call, that you can use to retrieve the next page.
    ///
    /// Sets the *page token* query property to the given value.
    pub fn page_token(mut self, new_value: &str) -> OrganizationDatacollectorListCall<'a> {
        self._page_token = Some(new_value.to_string());
        self
    }
    /// Maximum number of data collectors to return. The page size defaults to 25.
    ///
    /// Sets the *page size* query property to the given value.
    pub fn page_size(mut self, new_value: i32) -> OrganizationDatacollectorListCall<'a> {
        self._page_size = Some(new_value);
        self
    }
    /// The delegate implementation is consulted whenever there is an intermediate result, or if something goes wrong
    /// while executing the actual API request.
    /// 
    /// It should be used to handle progress information, and to implement a certain level of resilience.
    ///
    /// Sets the *delegate* property to the given value.
    pub fn delegate(mut self, new_value: &'a mut dyn client::Delegate) -> OrganizationDatacollectorListCall<'a> {
        self._delegate = Some(new_value);
        self
    }

    /// Set any additional parameter of the query string used in the request.
    /// It should be used to set parameters which are not yet available through their own
    /// setters.
    ///
    /// Please note that this method must not be used to set any of the known parameters
    /// which have their own setter method. If done anyway, the request will fail.
    ///
    /// # Additional Parameters
    ///
    /// * *$.xgafv* (query-string) - V1 error format.
    /// * *access_token* (query-string) - OAuth access token.
    /// * *alt* (query-string) - Data format for response.
    /// * *callback* (query-string) - JSONP
    /// * *fields* (query-string) - Selector specifying which fields to include in a partial response.
    /// * *key* (query-string) - API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.
    /// * *oauth_token* (query-string) - OAuth 2.0 token for the current user.
    /// * *prettyPrint* (query-boolean) - Returns response with indentations and line breaks.
    /// * *quotaUser* (query-string) - Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.
    /// * *uploadType* (query-string) - Legacy upload protocol for media (e.g. "media", "multipart").
    /// * *upload_protocol* (query-string) - Upload protocol for media (e.g. "raw", "multipart").
    pub fn param<T>(mut self, name: T, value: T) -> OrganizationDatacollectorListCall<'a>
                                                        where T: AsRef<str> {
        self._additional_params.insert(name.as_ref().to_string(), value.as_ref().to_string());
        self
    }

    /// Identifies the authorization scope for the method you are building.
    ///
    /// Use this method to actively specify which scope should be used, instead the default `Scope` variant
    /// `Scope::CloudPlatform`.
    ///
    /// The `scope` will be added to a set of scopes. This is important as one can maintain access
    /// tokens for more than one scope.
    /// If `None` is specified, then all scopes will be removed and no default scope will be used either.
    /// In that case, you have to specify your API-key using the `key` parameter (see the `param()`
    /// function for details).
    ///
    /// Usually there is more than one suitable scope to authorize an operation, some of which may
    /// encompass more rights than others. For example, for listing resources, a *read-only* scope will be
    /// sufficient, a read-write scope will do as well.
    pub fn add_scope<T, S>(mut self, scope: T) -> OrganizationDatacollectorListCall<'a>
                                                        where T: Into<Option<S>>,
                                                              S: AsRef<str> {
        match scope.into() {
          Some(scope) => self._scopes.insert(scope.as_ref().to_string(), ()),
          None => None,
        };
        self
  