//! Sbot type and connection-related methods.
use async_std::net::TcpStream;

use kuska_handshake::async_std::BoxStream;
use kuska_sodiumoxide::crypto::{auth, sign::ed25519};
use kuska_ssb::{
    api::ApiCaller,
    discovery, keystore,
    keystore::OwnedIdentity,
    rpc::{RpcReader, RpcWriter},
};

use crate::error::GolgiError;

/// A struct representing a connection with a running sbot.
/// A client and an rpc_reader can together be used to make requests to the sbot
/// and read the responses.
/// Note there can be multiple SbotConnection at the same time.
pub struct SbotConnection {
    /// Client for writing requests to go-bot
    pub client: ApiCaller<TcpStream>,
    /// RpcReader object for reading responses from go-sbot
    pub rpc_reader: RpcReader<TcpStream>,
}

/// Holds the Scuttlebutt identity, keys and configuration parameters for
/// connecting to a local sbot and implements all Golgi API methods.
pub struct Sbot {
    /// The ID (public key value) of the account associated with the local sbot instance.
    pub id: String,
    public_key: ed25519::PublicKey,
    private_key: ed25519::SecretKey,
    address: String,
    // aka caps key (scuttleverse identifier)
    network_id: auth::Key,
}

impl Sbot {
    /// Initiate a connection with an sbot instance. Define the IP address,
    /// port and network key for the sbot, then retrieve the public key,
    /// private key (secret) and identity from the `.ssb-go/secret` file.
    pub async fn init(ip_port: Option<String>, net_id: Option<String>) -> Result<Sbot, GolgiError> {
        let address = if ip_port.is_none() {
            "127.0.0.1:8008".to_string()
        } else {
            ip_port.unwrap()
        };

        let network_id = if net_id.is_none() {
            discovery::ssb_net_id()
        } else {
            auth::Key::from_slice(&hex::decode(net_id.unwrap()).unwrap()).unwrap()
        };

        let OwnedIdentity { pk, sk, id } = keystore::from_gosbot_local()
            .await
            .expect("couldn't read local secret");

        Ok(Self {
            id,
            public_key: pk,
            private_key: sk,
            address,
            network_id,
        })
    }

    /// Creates a new connection with the sbot, using the address, network_id,
    /// public_key and private_key supplied when Sbot was initialized.
    ///
    /// Note that a single Sbot can have multiple SbotConnection at the same time.
    pub async fn get_sbot_connection(&self) -> Result<SbotConnection, GolgiError> {
        let address = self.address.clone();
        let network_id = self.network_id.clone();
        let public_key = self.public_key;
        let private_key = self.private_key.clone();
        Sbot::_get_sbot_connection_helper(address, network_id, public_key, private_key).await
    }

    /// Private helper function which creates a new connection with sbot,
    /// but with all variables passed as arguments.
    ///
    /// Open a TCP stream to the sbot and perform the secret handshake. If
    /// successful, create a box stream and split it into a writer and reader.
    /// Return RPC handles to the sbot as part of the `struct` output.
    async fn _get_sbot_connection_helper(
        address: String,
        network_id: auth::Key,
        public_key: ed25519::PublicKey,
        private_key: ed25519::SecretKey,
    ) -> Result<SbotConnection, GolgiError> {
        let socket = TcpStream::connect(&address)
            .await
            .map_err(|source| GolgiError::Io {
                source,
                context: "failed to initiate tcp stream connection".to_string(),
            })?;

        let handshake = kuska_handshake::async_std::handshake_client(
            &mut &socket,
            network_id.clone(),
            public_key,
            private_key.clone(),
            public_key,
        )
        .await
        .map_err(GolgiError::Handshake)?;

        let (box_stream_read, box_stream_write) =
            BoxStream::from_handshake(socket.clone(), socket, handshake, 0x8000).split_read_write();

        let rpc_reader = RpcReader::new(box_stream_read);
        let client = ApiCaller::new(RpcWriter::new(box_stream_write));
        let sbot_connection = SbotConnection { rpc_reader, client };
        Ok(sbot_connection)
    }
}
