//! Return the SSB ID of the local sbot instance.
//!
//! Implements the following methods:
//!
//! - [`Sbot::whoami`]

use crate::{error::GolgiError, sbot::Sbot, utils};

impl Sbot {
    /// Get the public key of the local identity.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn fetch_id() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let pub_key = sbot_client.whoami().await?;
    ///
    ///     println!("local ssb id: {}", pub_key);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn whoami(&mut self) -> Result<String, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let req_id = sbot_connection.client.whoami_req_send().await?;

        let result = utils::get_async(
            &mut sbot_connection.rpc_reader,
            req_id,
            utils::json_res_parse,
        )
        .await?;

        let id = result
            .get("id")
            .ok_or_else(|| GolgiError::Sbot("id key not found on whoami call".to_string()))?
            .as_str()
            .ok_or_else(|| GolgiError::Sbot("whoami returned non-string value".to_string()))?;
        Ok(id.to_string())
    }
}
