//! Publish Scuttlebutt messages.
//!
//! Implements the following methods:
//!
//! - [`Sbot::publish`]
//! - [`Sbot::publish_description`]
//! - [`Sbot::publish_name`]
//! - [`Sbot::publish_post`]

use crate::{error::GolgiError, messages::SsbMessageContent, sbot::Sbot, utils};

impl Sbot {
    /// Publish a message.
    ///
    /// # Arguments
    ///
    /// * `msg` - A `SsbMessageContent` `enum` whose variants include `Pub`,
    /// `Post`, `Contact`, `About`, `Channel` and `Vote`.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError, messages::SsbMessageContent};
    ///
    /// async fn publish_a_msg() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     // Construct an SSB message of type `post`.
    ///     let post = SsbMessageContent::Post {
    ///         text: "And then those vesicles, filled with the Golgi products, move to the rest of the cell".to_string(),
    ///         mentions: None,
    ///     };
    ///
    ///     let msg_ref = sbot_client.publish(post).await?;
    ///
    ///     println!("msg reference for the golgi post: {}", msg_ref);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn publish(&mut self, msg: SsbMessageContent) -> Result<String, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let req_id = sbot_connection.client.publish_req_send(msg).await?;

        utils::get_async(
            &mut sbot_connection.rpc_reader,
            req_id,
            utils::string_res_parse,
        )
        .await
    }

    /// Publish a post.
    ///
    /// Convenient wrapper around the `publish` method which constructs and
    /// publishes a `post` type message appropriately from a string.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn publish_a_post() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let text = "The Golgi is located right near the nucleus.";
    ///
    ///     let msg_ref = sbot_client.publish_post(text).await?;
    ///
    ///     println!("msg reference for the golgi post: {}", msg_ref);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn publish_post(&mut self, text: &str) -> Result<String, GolgiError> {
        let msg = SsbMessageContent::Post {
            text: text.to_string(),
            mentions: None,
        };
        self.publish(msg).await
    }

    /// Publish a description for the local identity.
    ///
    /// Convenient wrapper around the `publish` method which constructs and
    /// publishes an `about` type description message appropriately from a string.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn publish_a_description() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let description = "The Golgi apparatus was identified by the Italian scientist Camillo Golgi in 1897.";
    ///
    ///     let msg_ref = sbot_client.publish_description(description).await?;
    ///
    ///     println!("msg reference for the golgi description: {}", msg_ref);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn publish_description(&mut self, description: &str) -> Result<String, GolgiError> {
        let msg = SsbMessageContent::About {
            about: self.id.to_string(),
            name: None,
            title: None,
            branch: None,
            image: None,
            description: Some(description.to_string()),
            location: None,
            start_datetime: None,
        };
        self.publish(msg).await
    }

    /// Publish a name for the local identity.
    ///
    /// Convenient wrapper around the `publish` method which constructs and
    /// publishes an `about` type name message appropriately from a string.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn publish_a_name() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let name = "glyphski_golgionikus";
    ///
    ///     let msg_ref = sbot_client.publish_name(name).await?;
    ///
    ///     println!("msg reference: {}", msg_ref);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn publish_name(&mut self, name: &str) -> Result<String, GolgiError> {
        let msg = SsbMessageContent::About {
            about: self.id.to_string(),
            name: Some(name.to_string()),
            title: None,
            branch: None,
            image: None,
            description: None,
            location: None,
            start_datetime: None,
        };
        self.publish(msg).await
    }
}
