//! Create and use invite codes.
//!
//! Implements the following methods:
//!
//! - [`Sbot::invite_create`]
//! - [`Sbot::invite_use`]

use crate::{error::GolgiError, sbot::Sbot, utils};

impl Sbot {
    /// Generate an invite code.
    ///
    /// Calls the `invite.create` RPC method and returns the code.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn invite_code_generator() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let invite_code = sbot_client.invite_create(5).await?;
    ///
    ///     println!("this invite code can be used 5 times: {}", invite_code);
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn invite_create(&mut self, uses: u16) -> Result<String, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let req_id = sbot_connection.client.invite_create_req_send(uses).await?;

        utils::get_async(
            &mut sbot_connection.rpc_reader,
            req_id,
            utils::string_res_parse,
        )
        .await
    }

    /// Use an invite code.
    ///
    /// Calls the `invite.use` RPC method and returns a reference to the follow
    /// message.
    ///
    /// # Example
    ///
    /// ```rust
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn invite_code_consumer() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let invite_code = "127.0.0.1:8008:@0iMa+vP7B2aMrV3dzRxlch/iqZn/UM3S3Oo2oVeILY8=.ed25519~ZHNjeajPB/84NjjsrglZInlh46W55RcNDPcffTPgX/Q=";
    ///     
    ///     match sbot_client.invite_use(invite_code).await {
    ///         Ok(msg_ref) => println!("consumed invite code. msg reference: {}", msg_ref),
    ///         Err(e) => eprintln!("failed to consume the invite code: {}", e),
    ///     }
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn invite_use(&mut self, invite_code: &str) -> Result<String, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let req_id = sbot_connection
            .client
            .invite_use_req_send(invite_code)
            .await?;

        utils::get_async(
            &mut sbot_connection.rpc_reader,
            req_id,
            utils::string_res_parse,
        )
        .await
    }
}
