//! Return a history stream.
//!
//! Implements the following methods:
//!
//! - [`Sbot::create_history_stream`]

use async_std::stream::Stream;
use kuska_ssb::api::dto::CreateHistoryStreamIn;

use crate::{error::GolgiError, messages::SsbMessageValue, sbot::Sbot, utils};

impl Sbot {
    /// Call the `createHistoryStream` RPC method.
    ///
    /// # Example
    ///
    /// ```rust
    /// use async_std::stream::StreamExt;
    /// use golgi::{Sbot, GolgiError};
    ///
    /// async fn history() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let ssb_id = "@zqshk7o2Rpd/OaZ/MxH6xXONgonP1jH+edK9+GZb/NY=.ed25519".to_string();
    ///
    ///     let history_stream = sbot_client.create_history_stream(ssb_id).await?;
    ///
    ///     history_stream.for_each(|msg| {
    ///         match msg {
    ///             Ok(val) => println!("msg value: {:?}", val),
    ///             Err(e) => eprintln!("error: {}", e),
    ///         }
    ///     }).await;
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn create_history_stream(
        &mut self,
        id: String,
    ) -> Result<impl Stream<Item = Result<SsbMessageValue, GolgiError>>, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let args = CreateHistoryStreamIn::new(id);
        let req_id = sbot_connection
            .client
            .create_history_stream_req_send(&args)
            .await?;
        let history_stream = utils::get_source_stream(
            sbot_connection.rpc_reader,
            req_id,
            utils::ssb_message_res_parse,
        )
        .await;
        Ok(history_stream)
    }
}
