//! Perform subset queries.
//!
//! Implements the following methods:
//!
//! - [`Sbot::get_subset_stream`]

use async_std::stream::Stream;

use crate::{error::GolgiError, messages::SsbMessageValue, sbot::Sbot, utils};

// re-export subset-related kuska types
pub use kuska_ssb::api::dto::content::{SubsetQuery, SubsetQueryOptions};

impl Sbot {
    /// Make a subset query, as defined by the [Subset replication for SSB specification](https://github.com/ssb-ngi-pointer/ssb-subset-replication-spec).
    ///
    /// Calls the `partialReplication. getSubset` RPC method.
    ///
    /// # Arguments
    ///
    /// * `query` - A `SubsetQuery` which specifies the desired query filters.
    /// * `options` - An Option<`SubsetQueryOptions`> which, if provided, adds
    /// additional specifications to the query, such as page limit and/or
    /// descending results.
    ///
    /// # Example
    ///
    /// ```rust
    /// use async_std::stream::StreamExt;
    /// use golgi::{
    ///     Sbot,
    ///     GolgiError,
    ///     api::get_subset::{
    ///         SubsetQuery,
    ///         SubsetQueryOptions
    ///     }
    /// };
    ///
    /// async fn query() -> Result<(), GolgiError> {
    ///     let mut sbot_client = Sbot::init(None, None).await?;
    ///
    ///     let post_query = SubsetQuery::Type {
    ///         op: "type".to_string(),
    ///         string: "post".to_string()
    ///     };
    ///
    ///     let post_query_opts = SubsetQueryOptions {
    ///         descending: Some(true),
    ///         keys: None,
    ///         page_limit: Some(5),
    ///     };
    ///
    ///     // Return 5 `post` type messages from any author in descending order.
    ///     let query_stream = sbot_client
    ///         .get_subset_stream(post_query, Some(post_query_opts))
    ///         .await?;
    ///
    ///     // Iterate over the stream and pretty-print each returned message
    ///     // value while ignoring any errors.
    ///     query_stream.for_each(|msg| match msg {
    ///         Ok(val) => println!("{:#?}", val),
    ///         Err(_) => (),
    ///     });
    ///
    ///     Ok(())
    /// }
    /// ```
    pub async fn get_subset_stream(
        &mut self,
        query: SubsetQuery,
        options: Option<SubsetQueryOptions>,
    ) -> Result<impl Stream<Item = Result<SsbMessageValue, GolgiError>>, GolgiError> {
        let mut sbot_connection = self.get_sbot_connection().await?;
        let req_id = sbot_connection
            .client
            .getsubset_req_send(query, options)
            .await?;
        let get_subset_stream = utils::get_source_stream(
            sbot_connection.rpc_reader,
            req_id,
            utils::ssb_message_res_parse,
        )
        .await;
        Ok(get_subset_stream)
    }
}
