use std::time::Duration;

use anyhow::*;
use clap::Parser;
use colored::*;
use image::{GenericImageView, ImageFormat, Pixel};
use regex::Regex;
use reqwest::{header, redirect, Client};
use serde::{Deserialize, Serialize};
use tracing::{error, info, Level};

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
struct Jigsaw {
    pub small_image: String,
    pub big_image: String,
    pub y_height: u32,
}

#[derive(Parser, Debug)]
#[clap(about = "A command line Check-on Attendance tool", version = "1.0.1")]
struct Args {
    #[clap(short, long)]
    username: String,
    #[clap(short, long)]
    password: String,
    #[clap(
        short,
        long,
        default_value_t = 0,
        help = "delay time before all logic (in seconds)"
    )]
    delay: u64,
}

struct Attandance {
    currentempoid: String,
    records: Vec<String>,
}

#[tokio::main]
async fn main() -> Result<()> {
    tracing_subscriber::fmt().with_max_level(Level::INFO).init();

    print_header();

    let args = Args::parse();

    if args.delay > 0 {
        info!("starting sleep");
        tokio::time::sleep(Duration::from_secs(args.delay)).await;
        info!("sleep end");
    }

    let client = build_http_client()?;

    while letsgo(&client, &args).await.is_err() {
        info!("starting retry...");
    }

    Ok(())
}

async fn letsgo(client: &Client, args: &Args) -> Result<()> {
    // get the session
    let resp = client.get("http://kq.neusoft.com").send().await?;
    let set_cookie = resp
        .headers()
        .get("Set-Cookie")
        .unwrap()
        .to_str()
        .unwrap()
        .to_string(); // avoid borrow

    //Set-Cookie: JSESSIONID=1xkZhkbFzhj3gs57xJGM2Q23SlDw6SVWtQ74RkYHYFLJSS53njwL!1224734250; path=/
    let session_id = extract_once(&set_cookie, r"JSESSIONID=(.+);", 1);
    info!("session_id: {}", session_id);

    let html = resp.text().await?;
    // <input type="hidden" name="neusoft_key" value="ID1638177637830PWD1638177637830" />
    let ts = extract_once(&html, r"ID(\d+)PWD", 1);
    info!("timestamp given by system: {}", ts);

    // get the jigsaw
    let jiasaw = client
        .post("http://kq.neusoft.com/jigsaw")
        .header("Cookie", format!("JSESSIONID={}", session_id))
        .header("Origin", "http://kq.neusoft.com")
        .header("Refer", "http://kq.neusoft.com/")
        .send()
        .await?
        .json::<Jigsaw>()
        .await?;

    info!("jiasaw img: {}.png", jiasaw.big_image);

    let img_bytes = client
        .post(&format!(
            "http://kq.neusoft.com/upload/jigsawImg/{}.png",
            jiasaw.big_image
        ))
        .header("Cookie", format!("JSESSIONID={}", session_id))
        .send()
        .await?
        .bytes()
        .await?;
    let img = image::load_from_memory_with_format(&img_bytes, ImageFormat::Png)?;
    info!(
        "jigsaw img loaded, dimensions: {:?}, size: {} bytes",
        img.dimensions(),
        img_bytes.len()
    );

    let x = find_x(&img, jiasaw.y_height).unwrap();
    info!("jiasaw x_width: {}", x);

    // login
    let resp = client
        .post("http://kq.neusoft.com/login.jsp")
        .header("Cookie", format!("JSESSIONID={}", session_id))
        .header("Origin", "http://kq.neusoft.com")
        .header("Referer", "http://kq.neusoft.com/")
        .form(&[
            ("login", "true"),
            ("neusoft_attendance_online", ""),
            (&format!("KEY{}", ts), ""),
            ("neusoft_key", &format!("ID{}PWD{}", ts, ts)),
            (&format!("YZM{}!{}", session_id, ts), &x.to_string()),
            (&format!("ID{}!{}", session_id, ts), &args.username),
            (&format!("KEY{}!{}", session_id, ts), &args.password),
        ])
        .send()
        .await?;
    if !resp
        .headers()
        .get("location")
        .unwrap()
        .to_str()
        .unwrap()
        .contains("attendance")
    {
        error!("login failed");
        return Err(anyhow!("login failed"));
    }
    info!("login passed");

    let attendance = load_attendance(&client, session_id).await?;
    info!("currentempoid: {}", attendance.currentempoid);
    info!("before record: {:?}", attendance.records);

    //record
    let resp = client
        .post("http://kq.neusoft.com/record.jsp")
        .header("Cookie", format!("JSESSIONID={}", session_id))
        .header("Origin", "http://kq.neusoft.com")
        .header("Referer", "http://kq.neusoft.com/attendance.jsp")
        .form(&[
            ("currentempoid", &attendance.currentempoid[..]),
            ("browser", "Chrome"),
        ])
        .send()
        .await?;

    if !resp
        .headers()
        .get("location")
        .unwrap()
        .to_str()
        .unwrap()
        .contains("attendance")
    {
        error!("record failed");
        return Err(anyhow!("record failed"));
    }
    info!("record passed");

    let attendance = load_attendance(&client, session_id).await?;
    info!("after record: {:?}", attendance.records);

    Ok(())
}

fn print_header() {
    println!(
        "{}",
        "严禁一切不正当的在线打卡考勤行为，否则将严肃处理。".red()
    );
    println!(
        "{}",
        "All the false attendance action is forbidden otherwise will be punished in serious.".red()
    );
    println!();
}

fn build_http_client() -> Result<Client> {
    let mut headers = header::HeaderMap::new();
    headers.insert("Accept", header::HeaderValue::from_static("text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9"));
    headers.insert(
        "Accept-Encoding",
        header::HeaderValue::from_static("zh-CN,zh;q=0.9,en;q=0.8,en-GB;q=0.7,en-US;q=0.6"),
    );
    headers.insert(
        "Cache-Control",
        header::HeaderValue::from_static("no-cache"),
    );
    headers.insert("Connection", header::HeaderValue::from_static("keep-alive"));
    headers.insert("Host", header::HeaderValue::from_static("kq.neusoft.com"));
    headers.insert("Pragma", header::HeaderValue::from_static("no-cache"));
    headers.insert(
        "Upgrade-Insecure-Requests",
        header::HeaderValue::from_static("1"),
    );
    headers.insert("User-Agent", header::HeaderValue::from_static("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.131 Safari/537.36 Edg/92.0.902.73"));
    Ok(reqwest::Client::builder()
        .redirect(redirect::Policy::none())
        .default_headers(headers)
        .build()?)
}

fn extract_once<'a>(text: &'a str, re: &'static str, group: usize) -> &'a str {
    let regex = Regex::new(re).unwrap();
    let caps = regex.captures(text).unwrap();
    caps.get(group).unwrap().as_str()
}

fn extract_all<'a>(text: &'a str, re: &'static str, group: usize) -> Vec<&'a str> {
    let regex = Regex::new(re).unwrap();
    regex
        .captures_iter(text)
        .map(|caps| caps.get(group).unwrap().as_str())
        .collect()
}

async fn load_attendance(client: &Client, session_id: &str) -> Result<Attandance> {
    // attandance list
    let html = client
        .get("http://kq.neusoft.com/attendance.jsp")
        .header("Cookie", format!("JSESSIONID={}", session_id))
        .header("Origin", "http://kq.neusoft.com")
        .header("Referer", "http://kq.neusoft.com/")
        .send()
        .await?
        .text()
        .await?;

    // <input type="hidden" name="currentempoid" value="10972125123">
    let currentempoid = extract_once(&html, r#"name="currentempoid" value="(\d+)""#, 1);
    let records = extract_all(&html, r"<td>(\d{2}:\d{2}:\d{2})</td>", 1);

    Ok(Attandance {
        currentempoid: currentempoid.to_string(),
        records: records.into_iter().map(|rec| rec.to_string()).collect(),
    })
}

fn find_x(img: &image::DynamicImage, y: u32) -> Option<u32> {
    let mut start = 0;
    let mut combos = 0;

    for i in 0..img.dimensions().0 {
        let rgb = img.get_pixel(i, y + 13).to_rgb();
        if rgb.0 == [255, 255, 255] {
            if start == 0 {
                start = i;
                combos = 1;
            } else {
                combos += 1;
                if combos == 5 {
                    return Some(start);
                }
            }
        } else {
            start = 0;
            combos = 0;
        }
    }
    None
}
