#![cfg_attr(not(test), no_std)]

mod protocol;
mod utils;

use crate::control_packet::PropertyType;
use core::{
    convert::TryFrom,
    fmt::{Debug, Display, Formatter},
    num::NonZeroU16,
};
use utils::write_u16;

pub mod control_packet;

pub use protocol::Protocol;

pub use crate::control_packet::{
    clone_packet, read_packet, subscribe::SubscribeFilter, write_packet,
};

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum MqttError {
    /// Passed buffer has insufficient size
    /// It is the caller's responsibility to pass a large enough buffer
    InsufficientBufferSize,

    /// Trying to read/write an invalid length.
    /// Refers to an invalid/corrupted length, rather than a buffer size issue
    InvalidLength,

    /// PacketId cannot be zero
    PacketIdZero,

    /// SubscriptionIdentifier cannot be zero
    SubscriptionIdentifierZero,
    /// Attempted to decode a non-UTF8 string
    InvalidString(core::str::Utf8Error),

    /// UserProperties heapless vector has a limit size of 32 units
    TooManyUserProperties,

    TooManyFilters,
    TooManySubscriptionIdentifiers,
    TooManySubscribeReasonCodes,
    IncorrectPacketFormat,
    PayloadTooLong,
    PayloadRequired,
    MalformedRemainingLength,
    MalformedPacket,
    InsufficientBytes,
    InvalidHeader,
    InvalidPid,
    InvalidProtocol(u8),
    InvalidPropertyByte(u8),
    InvalidPropertyType(PropertyType),
    InvalidConnectReasonCode(u8),
    InvalidReasonCode(u8),
    InvalidRetainForwardRule(u8),
    InvalidQoS(u8),
    InvalidDisconnectReason(u8),
}

impl Display for MqttError {
    fn fmt(&self, f: &mut Formatter<'_>) -> core::fmt::Result {
        write!(f, "{:?}", self)
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Pid(NonZeroU16);
impl Pid {
    /// Returns a new `Pid`
    pub fn new(n: u16) -> Self {
        Pid(NonZeroU16::new(n).unwrap())
    }

    /// Get the `Pid` as a raw `u16`.
    pub fn get(self) -> u16 {
        self.0.get()
    }

    pub fn read(buf: &[u8], offset: &mut usize) -> Result<Self, MqttError> {
        let pid = ((buf[*offset] as u16) << 8) | buf[*offset + 1] as u16;
        *offset += 2;
        Self::try_from(pid)
    }

    pub fn write(self, buf: &mut [u8], offset: &mut usize) {
        write_u16(buf, offset, self.get())
    }
}

impl Default for Pid {
    fn default() -> Pid {
        Pid::new(1)
    }
}

impl core::ops::Add<u16> for Pid {
    type Output = Pid;

    /// Adding a `u16` to a `Pid` will wrap around and avoid 0.
    fn add(self, u: u16) -> Pid {
        let n = match self.get().overflowing_add(u) {
            (n, false) => n,
            (n, true) => n + 1,
        };
        Pid(NonZeroU16::new(n).unwrap())
    }
}

impl core::ops::Sub<u16> for Pid {
    type Output = Pid;

    /// Adding a `u16` to a `Pid` will wrap around and avoid 0.
    fn sub(self, u: u16) -> Pid {
        let n = match self.get().overflowing_sub(u) {
            (0, _) => core::u16::MAX,
            (n, false) => n,
            (n, true) => n - 1,
        };
        Pid(NonZeroU16::new(n).unwrap())
    }
}

impl From<Pid> for u16 {
    /// Convert `Pid` to `u16`.
    fn from(p: Pid) -> Self {
        p.0.get()
    }
}

impl TryFrom<u16> for Pid {
    type Error = MqttError;

    /// Convert `u16` to `Pid`. Will fail for value 0.
    fn try_from(u: u16) -> Result<Self, MqttError> {
        match NonZeroU16::new(u) {
            Some(nz) => Ok(Pid(nz)),
            None => Err(MqttError::InvalidPid),
        }
    }
}

/// Quality of service
#[repr(u8)]
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd)]
pub enum QoS {
    AtMostOnce = 0,
    AtLeastOnce = 1,
    ExactlyOnce = 2,
}

/// Maps a number to QoS
impl TryFrom<u8> for QoS {
    type Error = MqttError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            0 => Ok(QoS::AtMostOnce),
            1 => Ok(QoS::AtLeastOnce),
            2 => Ok(QoS::ExactlyOnce),
            qos => Err(MqttError::InvalidQoS(qos)),
        }
    }
}

/// Combined `QoS`/`Pid`
///
/// Used only in `Publish` packets
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum QosPid {
    AtMostOnce,
    AtLeastOnce(Pid),
    ExactlyOnce(Pid),
}

impl QosPid {
    /// Extract the `Pid` from a `QosPid`, if any
    pub fn pid(self) -> Option<Pid> {
        match self {
            QosPid::AtMostOnce => None,
            QosPid::AtLeastOnce(p) => Some(p),
            QosPid::ExactlyOnce(p) => Some(p),
        }
    }

    /// Extract the `QoS` from a `QosPid`
    pub fn qos(self) -> QoS {
        match self {
            QosPid::AtMostOnce => QoS::AtMostOnce,
            QosPid::AtLeastOnce(_) => QoS::AtLeastOnce,
            QosPid::ExactlyOnce(_) => QoS::ExactlyOnce,
        }
    }
}
