//! A library for making requests to gemini servers and parsing
//! reponses.
//!

#[derive(Debug, PartialEq, Eq, Clone, Copy)]
/// The status code of the gemini response header.
///
/// A gemini response header contains a status code part, and this part is listed as two decimal
/// digits, where the first digit contains the main status code, and the second digit is a
/// specification on top of that code.
pub enum StatusCode {
    /// # 1x INPUT.
    /// This code is returned when a user input is required. It is expected that you will
    /// try to load the same page with a query part added to the request which is user input.
    /// ## META
    /// The META will contain a prompt for the user
    ///
    /// ## Subcodes
    /// - 11: SENSITIVE INPUT. The client should treat it the same as INPUT, but should obfuscate
    /// the input to the user. Used for things like passwords.
    Input(u8),
    /// # 2x SUCCESS.
    /// This code is returned when the page was successfully loaded.
    /// ## META
    /// The META will contain a MIME type for the data sent
    Success(u8),
    /// # 3x REDIRECT.
    /// This code is returned when the server is redirecting the client to a new page.
    /// ## META
    /// The META of the header will contain the page to redirect to
    ///
    /// ## Subcodes
    /// - 30: TEMPORARY REDIRECT.
    /// - 31: PERMANENT REDIRECT. This page will never exist again and is permanently relocated to
    /// the link sent
    Redirect(u8),
    /// # 4x TEMPORARY FAILURE.
    /// This code is returned when there is a failure handling the request that may work later on.
    /// ## META
    /// The META of this header contains additional info about the failure. This should be
    /// displayed to human users.
    ///
    /// ## Additional info:
    /// Aggregators or crawlers should NOT repeat this request
    ///
    /// ## Subcodes
    /// - 41: SERVER UNAVAILABLE. The server is unavailable due to maintainence or slow down.
    /// - 42: CGI ERROR. A CGI process, or similar dynamic content system, has died or timed out
    /// unexpectedly.
    /// - 43: PROXY ERROR. A proxy request failed because the server was unable to successfully
    /// complete a transaction with the remote host.
    /// - 44: SLOW DOWN. Rate limiting is in effect. The META is an integer showing how long the
    /// client should wait before another request is made.
    TemporaryFailure(u8),
    /// # 5x PERMANENT FAILURE
    /// This code is returned when there is a failure handling the request. This request will NEVER
    /// work in the future and will fail in the same way again with an identical request.
    /// ## META
    /// The META of this header contains additional info about the failure and should be shown to
    /// human users.
    ///
    /// ## Additional info:
    /// Aggregators or crawlers should NOT repeat this request
    ///
    /// ## Subcodes:
    /// - 51: NOT FOUND. Akin to HTTP's 404, this request is accesing a resource that is not
    /// available. This resource may be available later on but not in the near future.
    /// - 52: GONE. This resource is gone and will never be in this location again. Search engines
    /// and aggregators should remove this entry and convey to users that this resource is gone.
    /// - 53: PROXY REQUEST REFUSED. This request was made for a different domain and this server
    /// does not except proxy requests.
    /// - 59: BAD REQUEST. The request header was malformed in some way or form.
    PermanentFailure(u8),
    /// # 6x CLIENT CERTIFICATE REQUIRED
    /// This code is returned when the requested resource requires a client certificate. If the
    /// request was made without a client certificate, it should provide one. If it was made with
    /// one, the server did not accept it and should be made with a different certificate.
    /// ## META
    /// The META of the header will contain more information as to why the certificate is required
    /// or as to why the certificate was denied and should be shown to the user
    ///
    /// ## Subcodes:
    /// - 61: CERTIFICATE NOT AUTHORISED. The certificate is not authorised to access the given
    /// resource. The certificate is not necessairly the problem, it is just simply not authorized
    /// to access this specific resource.
    /// - 62: CERTIFICATE NOT VALID. The certificate is not a valid certificate and was not
    /// accepted. Unlike code 61, the certificate itself is the problem. It could be that the
    /// certificate is expired or the start date is in the future, or it could be because it is a
    /// violation of the X509 standard.
    ClientCertRequired(u8),
    /// # Unknown status code
    /// This is returned when the status code returned from the server is unknown.
    /// Unlike the rest of the status codes, contained in this enum variant is the return code in
    /// its entirety. Eg. if the status code is 84, 84 will be contained in the variant and not
    /// just 4.
    Unknown(u8),
}

impl From<u8> for StatusCode {
    fn from(i: u8) -> Self {
        if i > 99 {
            return Self::Unknown(i);
        }
        let first_digit = i % 10;
        let second_digit = i / 10;
        match second_digit {
            1 => Self::Input(first_digit),
            2 => Self::Success(first_digit),
            3 => Self::Redirect(first_digit),
            4 => Self::TemporaryFailure(first_digit),
            5 => Self::PermanentFailure(first_digit),
            6 => Self::ClientCertRequired(first_digit),
            _ => Self::Unknown(i),
        }
    }
}

#[derive(Debug, PartialEq, Eq)]
/// An error in parsing a response header from a server
pub enum ResponseParseError {
    /// The entire response was empty.
    EmptyResponse,
    /// The response header was invalid and could not be parsed
    InvalidResponseHeader,
}

#[derive(Debug, PartialEq, Eq)]
/// A Gemini response.
///
/// A Gemini response consists of two parts: The header and the content. The header is separated by
/// a new line (CRLF or just LF) and contains two parts in itself, the status code, and a META
/// string with more info about the status code.
pub struct Response {
    /// The status code of the response header.
    pub status: StatusCode,
    /// The META string of the response header.
    pub meta: String,
    /// The data returned from the header.
    pub data: Vec<u8>,
}

impl Response {
    /// Parses a response from a u8 slice.
    ///
    /// # Arguments:
    ///
    /// * `raw_response` - The raw response bytes to parse
    ///
    /// # Returns:
    ///
    /// * A Result with either a fully parsed response or an [error describing what went wrong when
    /// parsing](ResponseParseError)
    ///
    /// # Example:
    /// ```
    /// # use gmi::req_resp::Response;
    /// # use gmi::req_resp::StatusCode;
    /// # fn main() -> Result<(), gmi::req_resp::ResponseParseError> {
    /// let res = Response::parse_slice(br#"20 text/gemini
    /// ## Test response
    /// Hello!"#)?;
    ///     assert_eq!(res.status, StatusCode::Success(0));
    ///     assert_eq!(res.meta, "text/gemini");
    ///     assert_eq!(String::from_utf8_lossy(&res.data).into_owned(), "# Test response\nHello!");
    /// # Ok(())
    /// }
    /// ```
    pub fn parse_slice(raw_response: &[u8]) -> Result<Self, ResponseParseError> {
        if raw_response.len() == 0 {
            return Err(ResponseParseError::EmptyResponse);
        }
        // Let's find the first LF in the response.
        // Since CR is before the LF we can just clip that off if the response contains it
        let mut first_lf = 0;
        for (i, b) in raw_response.iter().enumerate() {
            if *b == b'\n' {
                first_lf = i;
                break;
            }
        }
        // If the first_lf was not found then we can assume that the response header is invalid,
        // since it needs to end in a CRLF
        if first_lf == 0 {
            return Err(ResponseParseError::InvalidResponseHeader);
        }

        // Now we'll convert the slice into a string with the last of the lf
        let response_header: &str = match std::str::from_utf8(&raw_response[..first_lf]) {
            Ok(s) => s,
            Err(_) => return Err(ResponseParseError::InvalidResponseHeader),
        };

        // We'll split on whitespace
        let (status_code, meta) = match response_header.split_once(' ') {
            None => return Err(ResponseParseError::InvalidResponseHeader),
            Some(r) => r,
        };
        // Then we'll trim the meta
        let meta = meta.trim();
        // And then we'll check how long the meta is
        if meta.len() > 1024 {
            return Err(ResponseParseError::InvalidResponseHeader);
        }
        let status_code = match status_code.parse::<u8>() {
            Ok(s) => s,
            Err(_) => return Err(ResponseParseError::InvalidResponseHeader),
        };

        let status = StatusCode::from(status_code);

        let data = Vec::from(&raw_response[first_lf + 1..]);

        Ok(Self {
            status,
            meta: String::from(meta),
            data,
        })
    }
}

#[derive(Debug)]
/// A catch-all enum for any errors that may happen
/// while making and parsing the request
pub enum RequestError {
    /// Occurs when an [IO Error](std::io::Error) occurs.
    IoError(std::io::Error),
    /// Occurs when a DNS error occurs.
    DnsError,
    /// Occurs when some sort of [TLS error](rustls::Error) occurs
    TlsError(rustls::Error),
    /// Occurs when the scheme given is unknown. Returns the scheme name.
    UnknownScheme(String),
    /// Occurs when the response from the server cannot be parsed.
    ResponseParseError(ResponseParseError),
}

// This is for the TLS for Gemini. This will just simply trust any TLS
// certs we get for now. We can implement TOFU later on.
struct DummyVerifier {}

impl DummyVerifier {
    pub fn new() -> Self {
        Self {}
    }
}

impl rustls::client::ServerCertVerifier for DummyVerifier {
    fn verify_server_cert(
        &self,
        _end_entity: &rustls::Certificate,
        _intermediates: &[rustls::Certificate],
        _server_name: &rustls::ServerName,
        _scts: &mut dyn Iterator<Item = &[u8]>,
        _ocsp_response: &[u8],
        _now: std::time::SystemTime,
    ) -> Result<rustls::client::ServerCertVerified, rustls::Error> {
        Ok(rustls::client::ServerCertVerified::assertion())
    }
}

/*
struct TofuVerifier {
    pub certs: std::collections::HashMap<rustls::ServerName, rustls::Certificate>,
}

impl TofuVerifier {
    pub fn new() -> Self {
        Self {
            certs: std::collections::HashMap::new(),
        }
    }
}

impl rustls::client::ServerCertVerifier for TofuVerifier {
    fn verify_server_cert(
        &self,
        end_entity: &rustls::Certificate,
        _intermediates: &[rustls::Certificate],
        server_name: &rustls::ServerName,
        _scts: &mut dyn Iterator<Item = &[u8]>,
        _ocsp_response: &[u8],
        _now: std::time::SystemTime
    ) -> Result<rustls::client::ServerCertVerified, rustls::Error> {
        Ok(rustls::client::ServerCertVerified::assertion())
    }
}
*/

/// Open a TCP stream to a [`Url`](gmi::url::Url) given with a default port listed.
fn open_tcp_stream(url: &crate::url::Url, default_port: u16) -> Result<std::net::TcpStream, RequestError> {
    let tcp_stream = match std::net::TcpStream::connect(
        url.authority.to_string() + ":" + &url.authority.port.unwrap_or(default_port).to_string(),
    ) {
        Err(e) => return Err(RequestError::IoError(e)),
        Ok(s) => s,
    };
    tcp_stream.set_read_timeout(Some(std::time::Duration::from_secs(15))).unwrap();
    Ok(tcp_stream)
}

/// Create the request to send into the stream
fn make_gemini_merc_request(url: &crate::url::Url, input: Option<&str>) -> String {
    if input.is_some() {
        unimplemented!("Can't send input requests to servers yet");
    }
    url.to_string() + "\r\n"
}

/// Use a stream given (std::io::Write) to write a request
fn use_stream_do_request(req: &str, stream: &mut dyn std::io::Write) -> Result<(), RequestError> {
    match stream.write(req.as_bytes()) {
        Err(e) => Err(RequestError::IoError(e)),
        Ok(_) => Ok(()),
    }
}

/// Use a stream (std::io::Read) to read a response and parse that response
fn use_stream_get_resp(stream: &mut dyn std::io::Read) -> Result<Response, RequestError> {
    let mut buffer: Vec<u8> = Vec::new();
    match stream.read_to_end(&mut buffer) {
        Err(e) => return Err(RequestError::IoError(e)),
        Ok(_) => (),
    }
    parse_merc_gemini_resp(&buffer)
}

/// Parse a response taken from a server
fn parse_merc_gemini_resp(resp: &[u8]) -> Result<Response, RequestError> {
    match Response::parse_slice(resp) {
        Ok(r) => Ok(r),
        Err(e) => Err(RequestError::ResponseParseError(e)),
    }
}

/// Make a request to a gemini server
fn make_gemini_request(
    url: &crate::url::Url,
    input: Option<&str>,
) -> Result<Response, RequestError> {
    // These are only needed in this funcion, so we'll put a use here.
    use rustls::client::{ClientConfig, ClientConnection};
    use std::convert::TryInto;
    use std::sync::Arc;

    // Get our request string
    let request = make_gemini_merc_request(url, input);

    let authority = url.authority.to_string();

    // Get our DNS name
    let dnsname = match authority.as_str().try_into() {
        Ok(s) => s,
        Err(_) => return Err(RequestError::DnsError),
    };

    // Set up rustls
    let cfg = ClientConfig::builder()
        .with_safe_defaults()
        .with_custom_certificate_verifier(Arc::new(DummyVerifier::new()))
        .with_no_client_auth();

    // Set up our TLS client
    let client = ClientConnection::new(Arc::new(cfg), dnsname).unwrap();

    // Open up a socket
    let tcp_stream = open_tcp_stream(url, 1965)?;
    let mut tls_stream = rustls::StreamOwned::new(client, tcp_stream);


    use_stream_do_request(request.as_str(), &mut tls_stream)?;
    use_stream_get_resp(&mut tls_stream)
}

/// Make a request to a mercury server
fn make_mercury_request(
    url: &crate::url::Url,
    input: Option<&str>,
) -> Result<Response, RequestError> {
    let request = make_gemini_merc_request(url, input);
    let mut stream = open_tcp_stream(url, 1963)?;
    use_stream_do_request(request.as_str(), &mut stream)?;
    use_stream_get_resp(&mut stream)
}

/// Make a request to a [URL](crate::url::Url).
///
/// # Errors:
/// Will return a [`RequestError`] on any sort of error
/// # Example:
/// ```
/// # use gmi::req_resp;
/// # use gmi::req_resp::StatusCode;
/// # use gmi::url::Url;
/// # fn main() -> Result<(), req_resp::RequestError> {
/// let url = Url::from_str("gemini://gemini.circumlunar.space/").unwrap();
/// let response = req_resp::make_request(&url, None)?;
/// assert_eq!(response.status, StatusCode::Success(0));
/// # Ok(())
/// # }
pub fn make_request(url: &crate::url::Url, input: Option<&str>) -> Result<Response, RequestError> {
    // Get the scheme, and see what type of request we're making
    match url
        .scheme
        .as_ref()
        .unwrap_or(&"mercury".to_string())
        .as_str()
    {
        "gemini" => make_gemini_request(url, input),
        "mercury" => make_mercury_request(url, input),
        s => Err(RequestError::UnknownScheme(String::from(s))),
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn status_code_from_u8_input() {
        assert_eq!(StatusCode::from(18), StatusCode::Input(8));
    }
    #[test]
    fn response_parse_slice() {
        let raw_response = "20 text/gemini\r\n# Hello!";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap();
        assert_eq!(parsed_response.status, StatusCode::Success(0));
        assert_eq!(parsed_response.meta, "text/gemini");
        assert_eq!(parsed_response.data, "# Hello!".as_bytes());
    }
    #[test]
    fn response_parse_slice_error_empty() {
        let raw_response = "";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap_err();
        assert_eq!(parsed_response, ResponseParseError::EmptyResponse);
    }
    #[test]
    fn response_parse_slice_error_invalid_header_missing_space() {
        let raw_response = "20text/gemini\r\n#Hello!";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap_err();
        assert_eq!(parsed_response, ResponseParseError::InvalidResponseHeader);
    }
    #[test]
    fn response_parse_slice_error_invalid_header_missing_space_and_meta() {
        let raw_response = "20\r\n# Hello!";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap_err();
        assert_eq!(parsed_response, ResponseParseError::InvalidResponseHeader);
    }
    #[test]
    fn response_parse_slice_error_invalid_header_meta_long() {
        let mut raw_response: String = String::from("20 ");
        for _ in 0..2048 {
            raw_response.push('a');
        }
        raw_response.push_str("\r\n# Hello!");
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap_err();
        assert_eq!(parsed_response, ResponseParseError::InvalidResponseHeader);
    }
    #[test]
    fn response_parse_slice_empty_body() {
        let raw_response = "20 text/gemini\r\n";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap();
        assert_eq!(parsed_response.status, StatusCode::Success(0));
        assert_eq!(parsed_response.meta, "text/gemini");
        assert_eq!(parsed_response.data, []);
    }
    #[test]
    fn response_parse_slice_empty_meta() {
        let raw_response = "20 \r\n";
        let parsed_response = Response::parse_slice(raw_response.as_bytes()).unwrap();
        assert_eq!(parsed_response.status, StatusCode::Success(0));
        assert_eq!(parsed_response.meta, "");
        assert_eq!(parsed_response.data, []);
    }
    #[test]
    fn make_request_invalid_scheme_error() {
        use crate::url::Url;
        let raw_url = Url::from_str("https://example.com").unwrap();
        let response = make_request(&raw_url, None).unwrap_err();
        match response {
            RequestError::UnknownScheme(s) => {
                assert_eq!(s, "https");
            }
            e => {
                panic!("Error returned was not an UnknownScheme but instead {:?}", e);
            }
        }
    }
}
