//! A gemtext parser
//!
//! This library will parse gemtext into various [Nodes](enum.GemtextNode.html)

#[derive(Debug)]
/// A singular gemtext node.
pub enum GemtextNode {
    /// A pure text block. The string contained within is the entire line of text
    Text(String),
    /// A link.
    ///
    /// A link is found by a line starting with the characters "=>" followed by a space
    ///
    /// The first string contained is the link itself and the second string is an optional
    /// descriptor
    Link(String, Option<String>),
    /// A heading
    ///
    /// A heading starts with a singular # with a space following.
    /// The string contained is the text that follows the heading marker
    Heading(String),
    /// A subheading
    ///
    /// A subheading starts with the characters "##" with a space following.
    /// The string contained is the text that folllows the subheading marker.
    SubHeading(String),
    /// A subsubheading
    ///
    /// A subsubheading starts with the characters "###" with a space following.
    /// The string contained is the text that follows the subheading marker
    SubSubHeading(String),
    /// A list item
    ///
    /// A list item starts with the character "*".
    /// Unlike markdown, '-' is not allowed to start a list item
    ///
    /// The string contained is the text that follows the list item marker.
    ListItem(String),
    /// A block quote
    ///
    /// A blockquote starts with the character ">".
    ///
    /// The string contained is the text that follows the blockquote marker
    Blockquote(String),
    /// A block of preformatted text.
    ///
    /// A preformatted text block starts with the characters "\`\`\`"
    ///
    /// The first string contained is the text within the preformatted text (newlines and all). The second string is an optional formatting tag for the preformatted text a la Markdown. It's worth noting that this is more clearly listed as an "alt text" as opposed to a formatting tag.
    Preformatted(String, Option<String>),
    /// A singular empty line
    EmptyLine,
}

impl std::fmt::Display for GemtextNode {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            GemtextNode::EmptyLine => write!(f, ""),
            GemtextNode::Text(s) => write!(f, "{}", s),
            GemtextNode::Link(s, d) => {
                write!(f, "=> {} {}", s, d.as_ref().unwrap_or(&"".to_string()))
            }
            GemtextNode::Heading(s) => write!(f, "# {}", s),
            GemtextNode::SubHeading(s) => write!(f, "## {}", s),
            GemtextNode::SubSubHeading(s) => write!(f, "### {}", s),
            GemtextNode::ListItem(s) => write!(f, "* {}", s),
            GemtextNode::Blockquote(s) => write!(f, "> {}", s),
            GemtextNode::Preformatted(s, d) => {
                write!(f, "```{}\n{}```", d.as_ref().unwrap_or(&"".to_string()), s)
            }
        }
    }
}

#[derive(Debug)]
enum ParseState {
    Searching,
    Text,
    FirstLinkChar,
    SecondLinkChar,
    LinkLink,
    LinkDesc,
    ListWaitForSpace,
    ListItem,
    FirstTick,
    SecondTick,
    PreformattedTextType,
    HeadingStart,
    Heading,
    SubHeadingStart,
    SubHeading,
    SubSubHeadingStart,
    SubSubHeading,
    BlockquoteStart,
    Blockquote,
}

/// Parse gemtext into a vector of GemtextNodes
///
/// This will take a &str and return a vector of GemtextNodes. Because of the nature of the way
/// gemtext works, this parsing step cannot fail. It can only return garbage.
pub fn parse_gemtext(text: &str) -> Vec<GemtextNode> {
    // Let's define our parsing flags
    let mut is_in_preformatted = false;
    let mut preformatted_text_has_type = false;

    let mut nodes: Vec<GemtextNode> = Vec::new();
    let mut preformatted_text: String = String::new();
    let mut preformatted_text_type: String = String::new();

    for line in text.lines() {
        if is_in_preformatted {
            if line.starts_with("```") {
                if preformatted_text_has_type {
                    nodes.push(GemtextNode::Preformatted(
                        preformatted_text.clone(),
                        Some(preformatted_text_type.clone()),
                    ));
                } else {
                    nodes.push(GemtextNode::Preformatted(preformatted_text.clone(), None));
                }
                is_in_preformatted = false;
                preformatted_text.clear();
            } else {
                preformatted_text.push_str(line);
                preformatted_text.push('\n');
            }
            continue;
        }
        let trimmed_line = line.trim();
        if trimmed_line.is_empty() {
            nodes.push(GemtextNode::EmptyLine);
            continue;
        }
        // A simple enum to keep our parsing state
        let mut current_parse_state: ParseState = ParseState::Searching;
        let mut temp1 = String::new();
        let mut temp2 = String::new();
        // Go character by character and set our state accordingly
        for c in line.chars() {
            match current_parse_state {
                ParseState::Searching => match c {
                    '=' => current_parse_state = ParseState::FirstLinkChar,
                    '*' => current_parse_state = ParseState::ListWaitForSpace,
                    '`' => current_parse_state = ParseState::FirstTick,
                    '#' => current_parse_state = ParseState::HeadingStart,
                    '>' => current_parse_state = ParseState::BlockquoteStart,
                    _ => {
                        current_parse_state = ParseState::Text;
                    }
                },
                //=====
                //Text parsing
                //=====
                ParseState::Text => break,
                //=====
                //Link parsing
                //=====
                ParseState::FirstLinkChar => match c {
                    '>' => current_parse_state = ParseState::SecondLinkChar,
                    _ => {
                        current_parse_state = ParseState::Text;
                    }
                },
                ParseState::SecondLinkChar => {
                    if !c.is_whitespace() {
                        current_parse_state = ParseState::LinkLink;
                        temp1.push(c);
                    }
                }
                ParseState::LinkLink => {
                    if c.is_whitespace() {
                        current_parse_state = ParseState::LinkDesc;
                    } else {
                        temp1.push(c);
                    }
                }
                ParseState::LinkDesc => temp2.push(c),
                //=====
                //List parsing
                //=====
                ParseState::ListWaitForSpace => {
                    if !c.is_whitespace() {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::ListItem;
                    }
                }
                ParseState::ListItem => temp1.push(c),
                //======
                //Preformatted text
                //======
                ParseState::FirstTick => {
                    if c != '`' {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::SecondTick;
                    }
                }
                ParseState::SecondTick => {
                    if c != '`' {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::PreformattedTextType;
                        preformatted_text_type.clear();
                    }
                }
                ParseState::PreformattedTextType => preformatted_text_type.push(c),
                //=====
                //Headings
                //=====
                ParseState::HeadingStart => {
                    if c == '#' {
                        current_parse_state = ParseState::SubHeadingStart;
                    } else if !c.is_whitespace() {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::Heading;
                    }
                }
                ParseState::Heading => temp1.push(c),
                ParseState::SubHeadingStart => {
                    if c == '#' {
                        current_parse_state = ParseState::SubSubHeadingStart;
                    } else if !c.is_whitespace() {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::SubHeading;
                    }
                }
                ParseState::SubHeading => temp1.push(c),
                ParseState::SubSubHeadingStart => {
                    if c == '#' {
                        current_parse_state = ParseState::SubSubHeading;
                    } else if !c.is_whitespace() {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::SubSubHeading;
                    }
                }
                ParseState::SubSubHeading => temp1.push(c),
                ParseState::BlockquoteStart => {
                    if !c.is_whitespace() {
                        current_parse_state = ParseState::Text;
                    } else {
                        current_parse_state = ParseState::Blockquote;
                    }
                }
                ParseState::Blockquote => temp1.push(c),
            }
        }
        // Clean up any parse state we are in
        match current_parse_state {
            ParseState::Text => nodes.push(GemtextNode::Text(line.to_string())),

            ParseState::SecondLinkChar => nodes.push(GemtextNode::Text("=".to_string())),
            ParseState::LinkLink => nodes.push(GemtextNode::Link(temp1, None)),
            ParseState::LinkDesc => {
                if temp2.is_empty() {
                    nodes.push(GemtextNode::Link(temp1, None));
                } else {
                    nodes.push(GemtextNode::Link(temp1, Some(temp2)));
                }
            }

            ParseState::ListItem => nodes.push(GemtextNode::ListItem(temp1)),

            ParseState::FirstTick => nodes.push(GemtextNode::Text("`".to_string())),
            ParseState::SecondTick => nodes.push(GemtextNode::Text("``".to_string())),
            ParseState::PreformattedTextType => {
                is_in_preformatted = true;
                if preformatted_text_type.is_empty() {
                    preformatted_text_has_type = false;
                } else {
                    preformatted_text_has_type = true;
                }
            }
            ParseState::Heading => nodes.push(GemtextNode::Heading(temp1)),
            ParseState::HeadingStart => nodes.push(GemtextNode::Text("#".to_string())),
            ParseState::SubHeading => nodes.push(GemtextNode::SubHeading(temp1)),
            ParseState::SubHeadingStart => nodes.push(GemtextNode::Text("##".to_string())),
            ParseState::SubSubHeading => nodes.push(GemtextNode::SubSubHeading(temp1)),
            ParseState::SubSubHeadingStart => nodes.push(GemtextNode::Text("###".to_string())),
            ParseState::Blockquote => nodes.push(GemtextNode::Blockquote(temp1)),
            s => panic!("Invalid state: {:?}", s),
        }
    }
    nodes
}
