#![allow(non_camel_case_types)]

use crate::gl_types::*;

// Note: this file is not to be used as actual rust source code. It happens to
// look like rust source code, but that's because then rust-analyzer is helpful
// in editing the file. In truth, this is a "GL Function Spec" file. The parser
// for this file is quite simplistic. The format is that each entry has 1 or
// more docs lines ("///"), then the function signature ("pub type") on a single
// line. Blank lines are ignored, and also any line starting with "#!", "use ",
// or "// ". Any unexpected input will hopefully panic the parser.

/// [glClear](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClear.xhtml) (GL 1.0, GLES 2.0)
///
/// This sets the bitplane area of the window to values previously selected by
/// [ClearColor](Self::ClearColor), [ClearDepth](Self::ClearDepth), and
/// [ClearStencil](Self::ClearStencil).
///
/// * `mask` can contain any of the following bits:
///   * `GL_COLOR_BUFFER_BIT`
///   * `GL_DEPTH_BUFFER_BIT`
///   * `GL_STENCIL_BUFFER_BIT`
///
/// Multiple color buffers can be cleared simultaneously by selecting more than
/// one buffer at a time using [DrawBuffer](Self::DrawBuffer).
///
/// The pixel ownership test, the scissor test, dithering, and the buffer
/// writemasks affect the operation of `Clear`. The scissor box bounds the
/// cleared region. Alpha function, blend function, logical operation,
/// stenciling, texture mapping, and depth-buffering are ignored by `Clear`.
///
/// ## Errors
/// * `GL_INVALID_VALUE` is generated if any bit other than the three defined
///   bits is set in `mask`.
pub type glClear_t = extern "system" fn(mask: GLbitfield);

/// [glClearColor](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearColor.xhtml) (GL 1.0, GLES 2.0)
///
/// Specifies the `red`, `green`, `blue`, and `alpha` values used by
/// [Clear](Self::Clear) when clearing the color buffers.
///
/// All inputs are silently clamped to `0.0 ..= 1.0`
///
/// ## Associated Gets
/// * `GL_COLOR_CLEAR_VALUE`
pub type glClearColor_t = extern "system" fn(red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat);

/// [glClearDepth](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearDepth.xhtml) (GL 1.0, GLES -)
///
/// Specifies the depth value used when the depth buffer is cleared. The initial
/// value is 1.0
///
/// ## Associated Gets
/// * `GL_DEPTH_CLEAR_VALUE`
pub type glClearDepth_t = extern "system" fn(depth: GLdouble);

/// [glClearStencil](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearStencil.xhtml) (GL 1.0, GLES 2.0)
///
/// Specifies the index used when the stencil buffer is cleared. The initial
/// value is 0.
///
/// The input is masked with `2**m - 1`, where `m` is the number of bits in the
/// stencil buffer.
///
/// ## Associated Gets
/// * `GL_STENCIL_CLEAR_VALUE`
/// * `GL_STENCIL_BITS`
pub type glClearStencil_t = extern "system" fn(s: GLint);

/// [glViewport](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glViewport.xhtml) (GL 1.0, GLES 2.0)
///
/// Specifies the affine transformation of `x` and `y` from normalized device
/// coordinates to window coordinates. The window coordinates are computed as
/// follows:
///
/// ```txt
/// x_win = (x_ndc + 1)*(width  / 2) + x
/// y_win = (y_ndc + 1)*(height / 2) + y
/// ```
///
/// * `x` and `y` specify the lower left corner of the viewport rectangle, in
///   pixels. The initial value is (0,0).
/// * `width` and `height` specify the width and height of the viewport. When a
///   GL context is first attached to a window, width and height are set to the
///   dimensions of that window.
///
/// ## Errors
/// * `GL_INVALID_VALUE` is generated if either `width` or `height` are
///   negative.
///
/// ## Associated Gets
/// * `GL_VIEWPORT`
/// * `GL_MAX_VIEWPORT_DIMS`
pub type glViewport_t = extern "system" fn(x: GLint, y: GLint, width: GLsizei, height: GLsizei);

// after here is the functions we don't have full docs written for yet

/// [glActiveTexture](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glActiveTexture.xhtml)
pub type glActiveTexture_t = extern "system" fn(texture: GLenum);

/// [glAttachShader](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glAttachShader.xhtml)
pub type glAttachShader_t = extern "system" fn(program: GLuint, shader: GLuint);

/// [glBindBuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindBuffer.xhtml)
pub type glBindBuffer_t = extern "system" fn(target: GLenum, buffer: GLuint);

/// [glBindBufferBase](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindBufferBase.xhtml)
pub type glBindBufferBase_t = unsafe extern "system" fn(target: GLenum, index: GLuint, buffer: GLuint);

/// [glBindBufferRange](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindBufferRange.xhtml)
pub type glBindBufferRange_t = unsafe extern "system" fn(target: GLenum, index: GLuint, buffer: GLuint, offset: GLintptr, size: GLsizeiptr);

/// [glBindFramebuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindFramebuffer.xhtml)
pub type glBindFramebuffer_t = unsafe extern "system" fn(target: GLenum, framebuffer: GLuint);

/// [glBindRenderbuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindRenderbuffer.xhtml)
pub type glBindRenderbuffer_t = extern "system" fn(target: GLenum, renderbuffer: GLuint);

/// [glBindTexture](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindTexture.xhtml)
pub type glBindTexture_t = extern "system" fn(target: GLenum, texture: GLuint);

/// [glBindVertexArray](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindVertexArray.xhtml)
pub type glBindVertexArray_t = extern "system" fn(array: GLuint);

/// [glBlendEquation](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendEquation.xhtml)
pub type glBlendEquation_t = extern "system" fn(mode: GLenum);

/// [glBlendFunc](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFunc.xhtml)
pub type glBlendFunc_t = extern "system" fn(sfactor: GLenum, dfactor: GLenum);

/// [glBlendFunci](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFunc.xhtml)
pub type glBlendFunci_t = extern "system" fn(buf: GLuint, sfactor: GLenum, dfactor: GLenum);

/// [glBlitFramebuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlitFramebuffer.xhtml)
pub type glBlitFramebuffer_t = unsafe extern "system" fn(srcX0: GLint, srcY0: GLint, srcX1: GLint, srcY1: GLint, dstX0: GLint, dstY0: GLint, dstX1: GLint, dstY1: GLint, mask: GLbitfield, filter: GLenum);

/// [glBufferData](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBufferData.xhtml)
pub type glBufferData_t = unsafe extern "system" fn(target: GLenum, size: GLsizeiptr, data: *const void, usage: GLenum);

/// [glBufferSubData](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBufferSubData.xhtml)
pub type glBufferSubData_t = unsafe extern "system" fn(target: GLenum, offset: GLintptr, size: GLsizeiptr, data: *const void);

/// [glCheckFramebufferStatus](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCheckFramebufferStatus.xhtml)
pub type glCheckFramebufferStatus_t = extern "system" fn(target: GLenum) -> GLenum;

/// [glClearBufferfv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearBuffer.xhtml)
pub type glClearBufferfv_t = unsafe extern "system" fn(buffer: GLenum, drawbuffer: GLint, value: *const GLfloat);

/// [glCompileShader](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompileShader.xhtml)
pub type glCompileShader_t = extern "system" fn(shader: GLuint);

/// [glCreateProgram](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCreateProgram.xhtml)
pub type glCreateProgram_t = extern "system" fn() -> GLuint;

/// [glCreateShader](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCreateShader.xhtml)
pub type glCreateShader_t = extern "system" fn(shaderType: GLenum) -> GLuint;

/// [glCullFace](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCullFace.xhtml)
pub type glCullFace_t = extern "system" fn(mode: GLenum);

/// [glDebugMessageCallback](https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt) (GL_KHR_debug, GL 4.3, GLES 3.2)
///
/// Called by the GL driver to notify the program of various events.
pub type glDebugMessageCallback_t = Option<unsafe extern "system" fn(callback: Option<GLDEBUGPROC>, userParam: *const void)>;

/// [glDebugMessageControl](https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt) (GL_KHR_debug, GL 4.3, GLES 3.2)
///
/// Allows the program to set filters on what events will trigger the debug
/// callback.
pub type glDebugMessageControl_t = Option<unsafe extern "system" fn(source: GLenum, type_: GLenum, severity: GLenum, count: GLsizei, ids: *const GLuint, enabled: GLboolean)>;

/// [glDebugMessageInsert](https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt) (GL_KHR_debug, GL 4.3, GLES 3.2)
///
/// Allows the program to send a debug event to the driver, which will then call
/// the callback (if the event isn't filtered).
pub type glDebugMessageInsert_t = Option<unsafe extern "system" fn(source: GLenum, type_: GLenum, id: GLuint, severity: GLenum, length: GLint, buf: *const u8)>;

/// [glDeleteBuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteBuffers.xhtml)
pub type glDeleteBuffers_t = unsafe extern "system" fn(n: GLsizei, buffers: *const GLuint);

/// [glDeleteFramebuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteFramebuffers.xhtml)
pub type glDeleteFramebuffers_t = unsafe extern "system" fn(n: GLsizei, framebuffers: *mut GLuint);

/// [glDeleteProgram](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteProgram.xhtml)
pub type glDeleteProgram_t = extern "system" fn(program: GLuint);

/// [glDeleteShader](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteShader.xhtml)
pub type glDeleteShader_t = extern "system" fn(shader: GLuint);

/// [glDeleteTextures](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteTextures.xhtml)
pub type glDeleteTextures_t = unsafe extern "system" fn(n: GLsizei, textures: *const GLuint);

/// [glDeleteVertexArrays](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteVertexArrays.xhtml)
pub type glDeleteVertexArrays_t = unsafe extern "system" fn(n: GLsizei, arrays: *const GLuint);

/// [glDepthFunc](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDepthFunc.xhtml)
pub type glDepthFunc_t = extern "system" fn(func: GLenum);

/// [glDepthMask](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDepthMask.xhtml)
pub type glDepthMask_t = extern "system" fn(flag: GLboolean);

/// [glDisable](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDisable.xhtml)
pub type glDisable_t = extern "system" fn(cap: GLenum);

/// [glDrawArrays](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawArrays.xhtml)
pub type glDrawArrays_t = unsafe extern "system" fn(mode: GLenum, first: GLint, count: GLsizei);

/// [glDrawArraysInstanced](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawArraysInstanced.xhtml)
pub type glDrawArraysInstanced_t = unsafe extern "system" fn(mode: GLenum, first: GLint, count: GLsizei, instancecount: GLsizei);

/// [glDrawBuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawBuffer.xhtml)
pub type glDrawBuffer_t = extern "system" fn(buf: GLenum);

/// [glDrawBuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawBuffers.xhtml)
pub type glDrawBuffers_t = unsafe extern "system" fn(n: GLsizei, bufs: *const GLenum);

/// [glDrawElements](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawElements.xhtml)
pub type glDrawElements_t = unsafe extern "system" fn(mode: GLenum, count: GLsizei, type_: GLenum, indices: *const void);

/// [glDrawElementsInstanced](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawElementsInstanced.xhtml)
pub type glDrawElementsInstanced_t = unsafe extern "system" fn(mode: GLenum, count: GLsizei, type_: GLenum, indices: *const void, instancecount: GLsizei);

/// [glEnable](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnable.xhtml)
pub type glEnable_t = extern "system" fn(cap: GLenum);

/// [glEnableVertexAttribArray](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnableVertexAttribArray.xhtml)
pub type glEnableVertexAttribArray_t = extern "system" fn(index: GLuint);

/// [glDisableVertexAttribArray](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnableVertexAttribArray.xhtml)
pub type glDisableVertexAttribArray_t = extern "system" fn(index: GLuint);

/// [glFramebufferRenderbuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFramebufferRenderbuffer.xhtml)
pub type glFramebufferRenderbuffer_t = extern "system" fn(target: GLenum, attachment: GLenum, renderbuffertarget: GLenum, renderbuffer: GLuint);

/// [glFramebufferTexture](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFramebufferTexture.xhtml)
pub type glFramebufferTexture_t = extern "system" fn(target: GLenum, attachment: GLenum, texture: GLuint, level: GLint);

/// [glFramebufferTexture2D](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFramebufferTexture.xhtml)
pub type glFramebufferTexture2D_t = extern "system" fn(target: GLenum, attachment: GLenum, textarget: GLenum, texture: GLuint, level: GLint);

/// [glFrontFace](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFrontFace.xhtml)
pub type glFrontFace_t = extern "system" fn(mode: GLenum);

/// [glGenBuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenBuffers.xhtml)
pub type glGenBuffers_t = unsafe extern "system" fn(n: GLsizei, buffers: *mut GLuint);

/// [glGenerateMipmap](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenerateMipmap.xhtml)
pub type glGenerateMipmap_t = extern "system" fn(target: GLenum);

/// [glGenFramebuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenFramebuffers.xhtml)
pub type glGenFramebuffers_t = unsafe extern "system" fn(n: GLsizei, ids: *mut GLuint);

/// [glGenRenderbuffers](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenRenderbuffers.xhtml)
pub type glGenRenderbuffers_t = unsafe extern "system" fn(n: GLsizei, renderbuffers: *mut GLuint);

/// [glGenTextures](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenTextures.xhtml)
pub type glGenTextures_t = unsafe extern "system" fn(n: GLsizei, textures: *mut GLuint);

/// [glGenVertexArrays](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenVertexArrays.xhtml)
pub type glGenVertexArrays_t = unsafe extern "system" fn(n: GLsizei, arrays: *mut GLuint);

/// [glGetError](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetError.xhtml)
pub type glGetError_t = extern "system" fn() -> GLenum;

/// [glGetIntegerv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml)
pub type glGetIntegerv_t = unsafe extern "system" fn(pname: GLenum, data: *mut GLint);

/// [glGetProgramInfoLog](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetProgramInfoLog.xhtml)
pub type glGetProgramInfoLog_t = unsafe extern "system" fn(program: GLuint, maxLength: GLsizei, length: *mut GLsizei, infoLog: *mut GLchar);

/// [glGetProgramiv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetProgram.xhtml)
pub type glGetProgramiv_t = unsafe extern "system" fn(program: GLuint, pname: GLenum, params: *mut GLint);

/// [glGetShaderInfoLog](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetShaderInfoLog.xhtml)
pub type glGetShaderInfoLog_t = unsafe extern "system" fn(shader: GLuint, maxLength: GLsizei, length: *mut GLsizei, infoLog: *mut GLchar);

/// [glGetShaderiv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetShader.xhtml)
pub type glGetShaderiv_t = unsafe extern "system" fn(shader: GLuint, pname: GLenum, params: *mut GLint);

/// [glGetString](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetString.xhtml)
pub type glGetString_t = extern "system" fn(name: GLenum) -> *const GLubyte;

/// [glGetStringi](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetString.xhtml)
pub type glGetStringi_t = extern "system" fn(name: GLenum, index: GLuint) -> *const GLubyte;

/// [glGetUniformBlockIndex](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniformBlockIndex.xhtml)
pub type glGetUniformBlockIndex_t = unsafe extern "system" fn(program: GLuint, uniformBlockName: *const GLchar) -> GLuint;

/// [glGetUniformLocation](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniformLocation.xhtml)
pub type glGetUniformLocation_t = unsafe extern "system" fn(program: GLuint, name: *const GLchar) -> GLint;

/// [glLinkProgram](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glLinkProgram.xhtml)
pub type glLinkProgram_t = extern "system" fn(program: GLuint);

/// [glPixelStorei](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPixelStore.xhtml)
pub type glPixelStorei_t = extern "system" fn(pname: GLenum, param: GLint);

/// [glPolygonMode](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPolygonMode.xhtml)
pub type glPolygonMode_t = extern "system" fn(face: GLenum, mode: GLenum);

/// [glReadBuffer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glReadBuffer.xhtml)
pub type glReadBuffer_t = extern "system" fn(mode: GLenum);

/// [glRenderbufferStorage](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glRenderbufferStorage.xhtml)
pub type glRenderbufferStorage_t = extern "system" fn(target: GLenum, internalformat: GLenum, width: GLsizei, height: GLsizei);

/// [glRenderbufferStorageMultisample](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glRenderbufferStorageMultisample.xhtml)
pub type glRenderbufferStorageMultisample_t = extern "system" fn(target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei);

/// [glShaderSource](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glShaderSource.xhtml)
pub type glShaderSource_t = unsafe extern "system" fn(shader: GLuint, count: GLsizei, string: *const *mut GLchar, length: *const GLint);

/// [glStencilFunc](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilFunc.xhtml)
pub type glStencilFunc_t = extern "system" fn(func: GLenum, ref_: GLint, mask: GLuint);

/// [glStencilMask](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilMask.xhtml)
pub type glStencilMask_t = extern "system" fn(mask: GLuint);

/// [glStencilOp](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilOp.xhtml)
pub type glStencilOp_t = extern "system" fn(sfail: GLenum, dpfail: GLenum, dppass: GLenum);

/// [glTexImage2D](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage2D.xhtml)
pub type glTexImage2D_t = unsafe extern "system" fn(target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, border: GLint, format: GLenum, type_: GLenum, data: *const void);

/// [glTexImage2DMultisample](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage2DMultisample.xhtml)
pub type glTexImage2DMultisample_t = extern "system" fn(target: GLenum, samples: GLsizei, internalformat: GLenum, width: GLsizei, height: GLsizei, fixedsamplelocations: GLboolean);

/// [glTexImage3D](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage3D.xhtml)
pub type glTexImage3D_t = unsafe extern "system" fn(target: GLenum, level: GLint, internalformat: GLint, width: GLsizei, height: GLsizei, depth: GLsizei, border: GLint, format: GLenum, type_: GLenum, data: *const void);

/// [glTexParameterfv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml)
pub type glTexParameterfv_t = unsafe extern "system" fn(target: GLenum, pname: GLenum, params: *const GLfloat);

/// [glTexParameteri](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml)
pub type glTexParameteri_t = extern "system" fn(target: GLenum, pname: GLenum, param: GLint);

/// [glUniform1f](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform1f_t = extern "system" fn(location: GLint, v0: GLfloat);

/// [glUniform1fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform1fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, value: *const [GLfloat; 1]);

/// [glUniform1i](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform1i_t = extern "system" fn(location: GLint, v0: GLint);

/// [glUniform1iv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform1iv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, value: *const GLint);

/// [glUniform2f](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform2f_t = extern "system" fn(location: GLint, v0: GLfloat, v1: GLfloat);

/// [glUniform2fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform2fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, value: *const [GLfloat; 2]);

/// [glUniform3f](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform3f_t = extern "system" fn(location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat);

/// [glUniform3fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform3fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, value: *const [GLfloat; 3]);

/// [glUniform4f](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform4f_t = extern "system" fn(location: GLint, v0: GLfloat, v1: GLfloat, v2: GLfloat, v3: GLfloat);

/// [glUniform4fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniform4fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, value: *const [GLfloat; 4]);

/// [glUniformBlockBinding](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniformBlockBinding.xhtml)
pub type glUniformBlockBinding_t = unsafe extern "system" fn(program: GLuint, uniformBlockIndex: GLuint, uniformBlockBinding: GLuint);

/// [glUniformMatrix2fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniformMatrix2fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, transpose: GLboolean, value: *const [GLfloat; 4]);

/// [glUniformMatrix3fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniformMatrix3fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, transpose: GLboolean, value: *const [GLfloat; 9]);

/// [glUniformMatrix4fv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml)
pub type glUniformMatrix4fv_t = unsafe extern "system" fn(location: GLint, count: GLsizei, transpose: GLboolean, value: *const [GLfloat; 16]);

/// [glUseProgram](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUseProgram.xhtml)
pub type glUseProgram_t = extern "system" fn(program: GLuint);

/// [glVertexAttribDivisor](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttribDivisor.xhtml) (GL 3.3)
pub type glVertexAttribDivisor_t = extern "system" fn(index: GLuint, divisor: GLuint);

/// [glVertexAttribIPointer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttribPointer.xhtml)
pub type glVertexAttribIPointer_t = unsafe extern "system" fn(index: GLuint, size: GLint, type_: GLenum, stride: GLsizei, pointer: *const void);

/// [glVertexAttribPointer](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttribPointer.xhtml)
pub type glVertexAttribPointer_t = unsafe extern "system" fn(index: GLuint, size: GLint, type_: GLenum, normalized: GLboolean, stride: GLsizei, pointer: *const void);

/// [glGetFloatv](https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml)
pub type glGetFloatv_t = unsafe extern "system" fn(pname: GLenum, data: *mut GLfloat);
