#![allow(unused_labels)]

use std::{fs::File, io::Write};

#[allow(unused)]
macro_rules! f_write {
  ($( $tokens:tt )*) => {
    write!( $( $tokens )* ).unwrap();
  }
}
#[allow(unused)]
macro_rules! f_writeln {
  ($( $tokens:tt )*) => {
    writeln!( $( $tokens )* ).unwrap();
  }
}

#[derive(Debug)]
struct FnSpec {
  docs: String,
  type_declaration: String,
}
impl FnSpec {
  pub fn name(&self) -> &str {
    assert!(self.type_declaration.starts_with("pub type "));
    let eq = self.type_declaration.find('=').unwrap();
    &self.type_declaration[9..(eq - 3)]
  }
  pub fn ret(&self) -> &str {
    match self.type_declaration.find("->") {
      None => "",
      Some(i) => &self.type_declaration[i..self.type_declaration.len() - 1],
    }
  }
  pub fn arg_name_ty_list(&self) -> &str {
    let open = self.type_declaration.find('(').unwrap();
    let close = self.type_declaration.find(')').unwrap();
    &self.type_declaration[(open + 1)..close]
  }
  pub fn arg_name_list(&self) -> String {
    let mut out = String::new();
    for name_ty in self.arg_name_ty_list().split(',') {
      out.push_str(name_ty.split(':').next().unwrap());
      out.push(',');
    }
    out.pop();
    out
  }
}

fn main() {
  let gl_fn_types = std::fs::read_to_string("src/gl_fn_types.rs").unwrap();
  let mut gl_struct: File = std::fs::OpenOptions::new().read(true).write(true).create(true).truncate(true).open("../src/gl_struct.rs").unwrap();

  // read in all the function spec entries
  let mut entries = Vec::new();
  {
    // parse the function types file.
    let mut line_it = gl_fn_types.lines();
    'entry_find: while let Some(line) = line_it.next() {
      if line.is_empty() || line.starts_with("#!") || line.starts_with("use ") || line.starts_with("// ") {
        continue;
      } else if line.starts_with("///") {
        let mut docs = String::from(line);
        'entry_read: loop {
          let line = line_it.next().unwrap();
          if line.starts_with("///") {
            docs.push('\n');
            docs.push_str(line);
          } else if line.starts_with("pub type ") {
            entries.push(FnSpec { docs, type_declaration: String::from(line) });
            continue 'entry_find;
          } else {
            panic!("UNHANDLED CASE WHILE READING AN ENTRY: {:?}", line);
          }
        }
      } else {
        panic!("UNHANDLED CASE WHEN FINDING AN ENTRY: {:?}", line);
      }
    }
  }

  // ensure that all entries are sorted
  entries.sort_unstable_by_key(|e| e.name().to_string());

  // print our use statements.
  f_writeln!(gl_struct, "use crate::{{gl_fn_types::*, gl_types::*}};");
  f_writeln!(gl_struct, "use core::ffi::c_void;");
  f_writeln!(gl_struct, "use zstring::{{zstr, ZStr}};");
  f_writeln!(gl_struct);

  // print out the struct itself
  f_writeln!(gl_struct, "#[allow(non_snake_case)]");
  f_writeln!(gl_struct, "pub struct GlFns {{");
  f_writeln!(gl_struct, "  not_send_sync: core::marker::PhantomData<*const ()>,");
  for entry in entries.iter() {
    f_writeln!(gl_struct, "  {name}_p: {name}_t,", name = entry.name());
  }
  f_writeln!(gl_struct, "}}");

  // print out the loader constructor
  f_writeln!(gl_struct, "impl GlFns {{");
  f_writeln!(gl_struct, "  #[must_use]");
  f_writeln!(gl_struct, "  #[inline(never)]");
  f_writeln!(gl_struct, "  #[rustfmt::skip]");
  f_writeln!(gl_struct, "  pub unsafe fn from_loader(load: &dyn Fn(ZStr<'_>) -> *mut c_void) -> Option<Self> {{");
  f_writeln!(gl_struct, "    /// filters away known-bad pointer return values while converting to the fn type");
  f_writeln!(gl_struct, "    unsafe fn filter<T>(p: *mut c_void) -> Option<T> {{");
  f_writeln!(gl_struct, "      match p as usize {{");
  f_writeln!(gl_struct, "        0 | 1 | 2 | 3 | usize::MAX => None,");
  f_writeln!(gl_struct, "        _ => Some(core::mem::transmute_copy::<*mut c_void, T>(&p)),");
  f_writeln!(gl_struct, "      }}");
  f_writeln!(gl_struct, "    }}");
  f_writeln!(gl_struct, "    Some(Self {{");
  f_writeln!(gl_struct, "      not_send_sync: core::marker::PhantomData,");
  for entry in entries.iter() {
    if entry.type_declaration.contains("= Option<") {
      // if the type of this function entry is already optional then we can't
      // send it through filter (we'd get an Option<Option<fn>>). Instead we
      // transmute it directly.
      f_writeln!(gl_struct, r#"      {name}_p: core::mem::transmute(load(zstr!("{name}"))), // ferris plz"#, name = entry.name());
    } else {
      f_writeln!(gl_struct, r#"      {name}_p: filter( load(zstr!("{name}")) )?,"#, name = entry.name());
    }
  }
  f_writeln!(gl_struct, "    }})");
  f_writeln!(gl_struct, "  }}");
  f_writeln!(gl_struct, "}}");

  // print out the functions
  f_writeln!(gl_struct, "#[allow(non_snake_case)]");
  f_writeln!(gl_struct, "impl GlFns {{");
  for entry in entries.iter() {
    for doc_line in entry.docs.lines() {
      f_writeln!(gl_struct, "  {}", doc_line);
    }
    f_writeln!(gl_struct, "  #[inline]");
    let name = entry.name();
    let k_unsafe = if entry.type_declaration.contains("unsafe") { "unsafe " } else { "" };
    let short_name = &name[2..];
    let arg_name_ty_list = entry.arg_name_ty_list();
    let ret = entry.ret();
    let arg_name_list = entry.arg_name_list();
    f_writeln!(gl_struct, "  pub {k_unsafe}fn {short_name}(&self, {arg_name_ty_list}) {ret} {{", k_unsafe = k_unsafe, short_name = short_name, arg_name_ty_list = arg_name_ty_list, ret = ret);
    if entry.type_declaration.contains("= Option<") {
      f_writeln!(gl_struct, "    if let Some(f) = self.{name}_p {{ f({arg_name_list}) }}", name = name, arg_name_list = arg_name_list);
    } else {
      f_writeln!(gl_struct, "    (self.{name}_p)({arg_name_list})", name = name, arg_name_list = arg_name_list);
    }
    f_writeln!(gl_struct, "  }}");
  }
  f_writeln!(gl_struct, "}}");
}
