pub mod lib;

use anyhow::Result;
use clap::{app_from_crate, arg};
use std::fs::File;
use std::io::{BufReader, BufWriter, Read, Write};
use std::path::Path;

fn main() -> Result<()> {
    let matches = app_from_crate!()
        .name("gitlab-clippy")
        .after_help(
            "The expected input is generated by running 'cargo clippy --message-format=json'.",
        )
        .arg(
            arg!(<input> "input file; reads from stdin if none is given")
                .required(false)
                .takes_value(true)
                .allow_invalid_utf8(true),
        )
        .arg(
            arg!(-o --output <output> "output file; writes to stdout if none is given")
                .required(false)
                .takes_value(true)
                .allow_invalid_utf8(true),
        )
        .get_matches();

    let read = match matches.value_of_os("input").map(Path::new) {
        Some(path) => Box::new(File::open(path)?) as Box<dyn Read>,
        None => Box::new(std::io::stdin()) as Box<dyn Read>,
    };
    let reader = BufReader::new(read);

    let write = match matches.value_of_os("output").map(Path::new) {
        Some(path) => Box::new(File::create(path)?) as Box<dyn Write>,
        None => Box::new(std::io::stdout()) as Box<dyn Write>,
    };
    let writer = BufWriter::new(write);

    lib::parse_to_writer(reader, writer)
}

#[cfg(test)]
mod tests {
    use super::*;
    use std::fs::read_to_string;

    fn fixture(path: &str) -> BufReader<File> {
        BufReader::new(File::open(path).expect("Unable to open file"))
    }

    fn valid_fixture() -> BufReader<File> {
        fixture("tests/fixtures/valid.txt")
    }

    fn invalid_fixture() -> BufReader<File> {
        fixture("tests/fixtures/invalid.txt")
    }

    fn valid_result() -> String {
        read_to_string("tests/fixtures/valid.json").expect("Unable to open file")
    }

    #[test]
    fn parse_valid_fixture_to_string() {
        let result = lib::parse_to_string(valid_fixture());
        assert!(result.is_ok());
        assert_eq!(result.unwrap(), valid_result());
    }

    #[test]
    fn parse_valid_fixture_to_writer() {
        let result = lib::parse_to_writer(valid_fixture(), std::io::sink());
        assert!(result.is_ok());
    }

    #[test]
    fn parse_invalid_fixture_to_string() {
        let result = lib::parse_to_string(invalid_fixture());
        assert!(result.is_err());
    }

    #[test]
    fn parse_invalid_fixture_to_writer() {
        let result = lib::parse_to_writer(invalid_fixture(), std::io::sink());
        assert!(result.is_err());
    }
}
