//! This module contains an abstraction for gitignore files
use crate::errors::*;
use log::trace;
use std::fs::OpenOptions;
use std::path::PathBuf;

/// An abstraction of the gitignore file
#[derive(Debug, Default)]
pub struct Gitignore {
    /// The path to the gitignore
    path: PathBuf,
}
impl Gitignore {
    /// Create a new Gitignore struct from a git root folder path
    ///
    /// The given `path` must be a valid path to an existing git root folder
    pub fn from_path(path: &PathBuf) -> Self {
        trace!("Creating gitignore file object for path {}", path.to_string_lossy());
        Gitignore { path: path.clone() }
    }

    /// Append a line to the file
    pub fn add_line(&self, line: &str) -> Result<()> {
        use std::io::prelude::*;
        let mut file = OpenOptions::new()
            .write(true)
            .append(true)
            .create(true)
            .open(&self.path)
            .chain_err(|| "Error while opening gitignore file")?;
        writeln!(file, "{}", line).chain_err(|| "Error while writing line to gitignore")?;
        Ok(())
    }
}

impl std::fmt::Display for Gitignore {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::result::Result<(), std::fmt::Error> {
        write!(f, "gitignore file {}", self.path.to_string_lossy())
    }
}
