use std::io;
use std::path::{Path, PathBuf};
use std::process::Command;

pub struct Repository {
	path: PathBuf,
}

impl Repository {
	pub fn new(path: &Path) -> Repository {
		Repository {
			path: path.to_path_buf(),
		}
	}

	pub fn path(&self) -> &Path {
		Path::new(&self.path)
	}

	pub fn command(&self, args: &[&str]) -> io::Result<()> {
		let command = &format!("`git {}` in {}", args.join(" "), self.path.display());
		let help_message = &format!("Failed to execute {}", command);
		let status = Command::new("git")
			.current_dir(self.path())
			.args(args)
			.status()
			.expect(help_message);

		log::info!("{} exited with {}", command, status.code().unwrap(),);

		if status.success() {
			Ok(())
		} else {
			Err(io::Error::new(
				io::ErrorKind::Other,
				help_message.to_string(),
			))
		}
	}

	fn query(&self, args: &[&str]) -> io::Result<String> {
		let query = &format!("`git {}` in {}", args.join(" "), self.path.display());
		let help_message = &format!("Failed to execute {}", query);
		let cmd = Command::new("git")
			.current_dir(self.path())
			.args(args)
			.output()
			.expect(help_message);

		log::info!("query {} exited with {}", query, cmd.status.code().unwrap(),);

		if cmd.status.success() {
			Ok(String::from_utf8(cmd.stdout).unwrap())
		} else {
			Err(io::Error::new(
				io::ErrorKind::Other,
				help_message.to_string(),
			))
		}
	}

	fn get_commit_hash_for(&self, revision: &str) -> io::Result<String> {
		self.query(&["rev-parse", revision])
	}

	pub fn head_is_failing_ref(&self) -> bool {
		let head_commit_hash = self.get_commit_hash_for("@");
		let ref_is_failing_commit_hash = self.get_commit_hash_for("gamble-is-failing");

		match (head_commit_hash, ref_is_failing_commit_hash) {
			(Ok(head_commit_hash), Ok(ref_is_failing_commit_hash)) => {
				head_commit_hash == ref_is_failing_commit_hash
			}
			_ => false,
		}
	}
}
