use log::{error, info};
use std::io;
use std::process::{exit, Command};
use structopt::StructOpt;
use yansi::Paint;

mod app;
use app::App;

mod repository;

fn main() -> io::Result<()> {
	use repository::Repository;

	pretty_env_logger::init();

	let opt = App::from_args();
	info!("{:#?}", opt);

	let gamble_strategy = if opt.pass { tcr } else { trc };

	let test_command = reformat_command(opt.test_command);

	if test_command.is_none() {
		error!("Can't parse test command");
		exit(1)
	}

	let command_choice = gamble_strategy(test_command.unwrap());

	let repository = Repository::new(opt.repository_path.as_path());

	match command_choice {
		RepositoryCommandChoice::ShouldCommit => {
			info!("commit");
			if !opt.dry_run {
				repository.command(&["add", "--all"])?;

				let base_options = vec!["commit", "--allow-empty-message", "--no-edit"];

				let message_options = if opt.message.is_empty() && repository.head_is_failing_ref()
				{
					vec!["--reuse-message", "HEAD"]
				} else {
					vec!["--message", &opt.message]
				};

				let amend_options = if repository.head_is_failing_ref() {
					vec!["--amend"]
				} else {
					vec![]
				};

				let no_verify_options = if opt.no_verify {
					vec!["--no-verify"]
				} else {
					vec![]
				};

				repository.command(
					&[
						base_options,
						message_options,
						amend_options,
						no_verify_options,
					]
					.concat(),
				)?;

				if opt.fail {
					repository.command(&["update-ref", "refs/gamble-is-failing", "@"])?;
				}
			}
			info!("committed");
			println!("{}", Paint::blue("Committed!").bold());
		}
		RepositoryCommandChoice::ShouldRevert => {
			info!("revert");
			if !opt.dry_run {
				repository.command(&["reset", "--hard"])?;
			}
			info!("revertted");
			println!("{}", Paint::red("Reverted!").bold());
		}
	};

	Ok(())
}

fn reformat_command(command: Vec<String>) -> Option<Vec<String>> {
	if command.len() == 1 {
		shlex::split(command[0].as_str())
	} else {
		Some(command)
	}
}

#[derive(Debug, PartialEq)]
enum RepositoryCommandChoice {
	ShouldCommit,
	ShouldRevert,
}

fn tcr(command: Vec<String>) -> RepositoryCommandChoice {
	let status = Command::new(&command[0])
		.args(&command[1..])
		.status()
		.expect("failed to run tests");
	if status.success() {
		RepositoryCommandChoice::ShouldCommit
	} else {
		RepositoryCommandChoice::ShouldRevert
	}
}

fn trc(command: Vec<String>) -> RepositoryCommandChoice {
	let status = Command::new(&command[0])
		.args(&command[1..])
		.status()
		.expect("failed to run tests");
	if status.success() {
		RepositoryCommandChoice::ShouldRevert
	} else {
		RepositoryCommandChoice::ShouldCommit
	}
}

#[cfg(test)]
extern crate speculate;

#[cfg(test)]
use speculate::speculate;

#[cfg(test)]
speculate! {
	describe "tcr" {
		it "should commit when command exited with a zero code" {
			assert_eq!(tcr(vec!(String::from("true"))), RepositoryCommandChoice::ShouldCommit)
		}

		it "should revert when command exited with a non zero code" {
			assert_eq!(tcr(vec!(String::from("false"))), RepositoryCommandChoice::ShouldRevert)
		}
	}

	describe "trc" {
		it "should revert when command exited with a zero code" {
			assert_eq!(trc(vec!(String::from("true"))), RepositoryCommandChoice::ShouldRevert)
		}

		it "should commit when command exited with a non zero code" {
			assert_eq!(trc(vec!(String::from("false"))), RepositoryCommandChoice::ShouldCommit)
		}
	}
}
