#![deny(missing_docs)]
//! Hello World GhostActor Example

use ghost_actor::*;
use observability::span_context;
use tracing_futures::Instrument;

// Most of the GhostActor magic happens in this macro.
// Sender and Handler traits will be generated here.
ghost_chan! {
    /// HelloWorld Api Types
    pub chan HelloWorldApi<GhostError> {
        /// Returns the string "hello world!"
        fn hello_world() -> String;
    }
}

/// Main Entrypoint
#[tokio::main]
async fn main() -> Result<(), GhostError> {
    observability::test_run_open().ok();
    let span = tracing::debug_span!("main");
    let _g = span.enter();
    span_context!(span);
    // spawn our actor, getting the actor sender.
    let sender = spawn_hello_world().await?;
    span_context!(span);

    // we can make async calls on the sender
    assert_eq!("hello world!", &sender.hello_world().await?);
    span_context!(span);
    println!("{}", sender.hello_world().await?);
    span_context!(span);

    Ok(())
}

/// Use the GhostActorBuilder to construct the actor task.
pub async fn spawn_hello_world(
) -> Result<GhostSender<HelloWorldApi>, GhostError> {
    // first we need a builder
    let builder = actor_builder::GhostActorBuilder::new();

    // now let's register an event channel with this actor.
    let sender = builder
        .channel_factory()
        .create_channel::<HelloWorldApi>()
        .await?;

    // actually spawn the actor driver task
    // providing our implementation
    tokio::task::spawn(builder.spawn(HelloWorldImpl));

    // return the sender that controls the actor
    Ok(sender)
}

// -- private -- //

/// We need a struct to implement our handler upon.
struct HelloWorldImpl;

/// All handlers must implement GhostControlHandler.
/// This provides a default no-op handle_ghost_actor_shutdown impl.
impl GhostControlHandler for HelloWorldImpl {}

/// Implement GhostHandler for your specific GhostEvent type.
/// Don't worry, the compiler will let you know if you forget this : )
impl GhostHandler<HelloWorldApi> for HelloWorldImpl {}

/// Now implement your actual handler -
/// auto generated by the `ghost_chan!` macro.
impl HelloWorldApiHandler for HelloWorldImpl {
    fn handle_hello_world(&mut self) -> HelloWorldApiHandlerResult<String> {
        Ok(must_future::MustBoxFuture::new(
            async move {
                let span = tracing::Span::current();
                span_context!(span);
                Ok("hello world!".to_string())
            }
            .instrument(tracing::debug_span!("handle_hello_world")),
        ))
    }
}
