#[cfg(feature = "with-mvt")]
pub mod mvt {
    //! MVT conversions.
    mod mvt_commands {
        //! Encode geometries according to MVT spec
        //! https://github.com/mapbox/vector-tile-spec/tree/master/2.1
        use std::vec::Vec;
        /// Command to be executed and the number of times that the command will be executed
        /// https://github.com/mapbox/vector-tile-spec/tree/master/2.1#431-command-integers
        struct CommandInteger(u32);
        enum Command {
            MoveTo = 1,
            LineTo = 2,
            ClosePath = 7,
        }
        impl CommandInteger {
            fn new(id: Command, count: u32) -> CommandInteger {
                CommandInteger(((id as u32) & 0x7) | (count << 3))
            }
        }
        /// Commands requiring parameters are followed by a ParameterInteger for each parameter required by that command
        /// https://github.com/mapbox/vector-tile-spec/tree/master/2.1#432-parameter-integers
        struct ParameterInteger(u32);
        impl ParameterInteger {
            fn new(value: i32) -> ParameterInteger {
                ParameterInteger(((value << 1) ^ (value >> 31)) as u32)
            }
        }
        pub struct CommandSequence(pub Vec<u32>);
        impl CommandSequence {
            pub fn new() -> CommandSequence {
                CommandSequence(Vec::new())
            }
            pub fn vec(&self) -> Vec<u32> {
                self.0.clone()
            }
            fn push(&mut self, value: u32) {
                self.0.push(value);
            }
        }
    }
    pub(crate) mod mvt_reader {
        use crate::error::Result;
        use crate::mvt::vector_tile::{tile, tile::GeomType};
        use crate::{GeomProcessor, GeozeroGeometry};
        impl GeozeroGeometry for tile::Feature {
            fn process_geom<P: GeomProcessor>(&self, processor: &mut P) -> Result<()> {
                process_geom(self, processor)
            }
        }
        /// Process MVT geometry.
        pub fn process_geom<P: GeomProcessor>(
            geom: &tile::Feature,
            processor: &mut P,
        ) -> Result<()> {
            process_geom_n(geom, 0, processor)
        }
        fn process_geom_n<P: GeomProcessor>(
            geom: &tile::Feature,
            idx: usize,
            processor: &mut P,
        ) -> Result<()> {
            match geom.r#type {
                _ => {}
            }
            Ok(())
        }
        fn process_coord<P: GeomProcessor>(
            coord: &tile::Feature,
            idx: usize,
            processor: &mut P,
        ) -> Result<()> {
            Ok(())
        }
        fn process_linestring<P: GeomProcessor>(
            geom: &tile::Feature,
            tagged: bool,
            idx: usize,
            processor: &mut P,
        ) -> Result<()> {
            let multi = processor.multi_dim();
            processor.linestring_end(tagged, idx)
        }
        fn process_polygon<P: GeomProcessor>(
            geom: &tile::Feature,
            tagged: bool,
            idx: usize,
            processor: &mut P,
        ) -> Result<()> {
            processor.polygon_end(tagged, idx)
        }
    }
    mod vector_tile {
        pub struct Tile {
            #[prost(message, repeated, tag = "3")]
            pub layers: ::prost::alloc::vec::Vec<tile::Layer>,
        }
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::clone::Clone for Tile {
            #[inline]
            fn clone(&self) -> Tile {
                match *self {
                    Tile {
                        layers: ref __self_0_0,
                    } => Tile {
                        layers: ::core::clone::Clone::clone(&(*__self_0_0)),
                    },
                }
            }
        }
        impl ::core::marker::StructuralPartialEq for Tile {}
        #[automatically_derived]
        #[allow(unused_qualifications)]
        impl ::core::cmp::PartialEq for Tile {
            #[inline]
            fn eq(&self, other: &Tile) -> bool {
                match *other {
                    Tile {
                        layers: ref __self_1_0,
                    } => match *self {
                        Tile {
                            layers: ref __self_0_0,
                        } => (*__self_0_0) == (*__self_1_0),
                    },
                }
            }
            #[inline]
            fn ne(&self, other: &Tile) -> bool {
                match *other {
                    Tile {
                        layers: ref __self_1_0,
                    } => match *self {
                        Tile {
                            layers: ref __self_0_0,
                        } => (*__self_0_0) != (*__self_1_0),
                    },
                }
            }
        }
        impl ::prost::Message for Tile {
            #[allow(unused_variables)]
            fn encode_raw<B>(&self, buf: &mut B)
            where
                B: ::prost::bytes::BufMut,
            {
                for msg in &self.layers {
                    ::prost::encoding::message::encode(3u32, msg, buf);
                }
            }
            #[allow(unused_variables)]
            fn merge_field<B>(
                &mut self,
                tag: u32,
                wire_type: ::prost::encoding::WireType,
                buf: &mut B,
                ctx: ::prost::encoding::DecodeContext,
            ) -> ::core::result::Result<(), ::prost::DecodeError>
            where
                B: ::prost::bytes::Buf,
            {
                const STRUCT_NAME: &'static str = "Tile";
                match tag {
                    3u32 => {
                        let mut value = &mut self.layers;
                        ::prost::encoding::message::merge_repeated(wire_type, value, buf, ctx)
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "layers");
                                error
                            })
                    }
                    _ => ::prost::encoding::skip_field(wire_type, tag, buf, ctx),
                }
            }
            #[inline]
            fn encoded_len(&self) -> usize {
                0 + ::prost::encoding::message::encoded_len_repeated(3u32, &self.layers)
            }
            fn clear(&mut self) {
                self.layers.clear();
            }
        }
        impl ::core::default::Default for Tile {
            fn default() -> Self {
                Tile {
                    layers: ::core::default::Default::default(),
                }
            }
        }
        impl ::core::fmt::Debug for Tile {
            fn fmt(&self, f: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                let mut builder = f.debug_struct("Tile");
                let builder = {
                    let wrapper = &self.layers;
                    builder.field("layers", &wrapper)
                };
                builder.finish()
            }
        }
        /// Nested message and enum types in `Tile`.
        pub mod tile {
            /// Variant type encoding
            /// The use of values is described in section 4.1 of the specification
            pub struct Value {
                /// Exactly one of these values must be present in a valid message
                #[prost(string, optional, tag = "1")]
                pub string_value: ::core::option::Option<::prost::alloc::string::String>,
                #[prost(float, optional, tag = "2")]
                pub float_value: ::core::option::Option<f32>,
                #[prost(double, optional, tag = "3")]
                pub double_value: ::core::option::Option<f64>,
                #[prost(int64, optional, tag = "4")]
                pub int_value: ::core::option::Option<i64>,
                #[prost(uint64, optional, tag = "5")]
                pub uint_value: ::core::option::Option<u64>,
                #[prost(sint64, optional, tag = "6")]
                pub sint_value: ::core::option::Option<i64>,
                #[prost(bool, optional, tag = "7")]
                pub bool_value: ::core::option::Option<bool>,
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::clone::Clone for Value {
                #[inline]
                fn clone(&self) -> Value {
                    match *self {
                        Value {
                            string_value: ref __self_0_0,
                            float_value: ref __self_0_1,
                            double_value: ref __self_0_2,
                            int_value: ref __self_0_3,
                            uint_value: ref __self_0_4,
                            sint_value: ref __self_0_5,
                            bool_value: ref __self_0_6,
                        } => Value {
                            string_value: ::core::clone::Clone::clone(&(*__self_0_0)),
                            float_value: ::core::clone::Clone::clone(&(*__self_0_1)),
                            double_value: ::core::clone::Clone::clone(&(*__self_0_2)),
                            int_value: ::core::clone::Clone::clone(&(*__self_0_3)),
                            uint_value: ::core::clone::Clone::clone(&(*__self_0_4)),
                            sint_value: ::core::clone::Clone::clone(&(*__self_0_5)),
                            bool_value: ::core::clone::Clone::clone(&(*__self_0_6)),
                        },
                    }
                }
            }
            impl ::core::marker::StructuralPartialEq for Value {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::PartialEq for Value {
                #[inline]
                fn eq(&self, other: &Value) -> bool {
                    match *other {
                        Value {
                            string_value: ref __self_1_0,
                            float_value: ref __self_1_1,
                            double_value: ref __self_1_2,
                            int_value: ref __self_1_3,
                            uint_value: ref __self_1_4,
                            sint_value: ref __self_1_5,
                            bool_value: ref __self_1_6,
                        } => match *self {
                            Value {
                                string_value: ref __self_0_0,
                                float_value: ref __self_0_1,
                                double_value: ref __self_0_2,
                                int_value: ref __self_0_3,
                                uint_value: ref __self_0_4,
                                sint_value: ref __self_0_5,
                                bool_value: ref __self_0_6,
                            } => {
                                (*__self_0_0) == (*__self_1_0)
                                    && (*__self_0_1) == (*__self_1_1)
                                    && (*__self_0_2) == (*__self_1_2)
                                    && (*__self_0_3) == (*__self_1_3)
                                    && (*__self_0_4) == (*__self_1_4)
                                    && (*__self_0_5) == (*__self_1_5)
                                    && (*__self_0_6) == (*__self_1_6)
                            }
                        },
                    }
                }
                #[inline]
                fn ne(&self, other: &Value) -> bool {
                    match *other {
                        Value {
                            string_value: ref __self_1_0,
                            float_value: ref __self_1_1,
                            double_value: ref __self_1_2,
                            int_value: ref __self_1_3,
                            uint_value: ref __self_1_4,
                            sint_value: ref __self_1_5,
                            bool_value: ref __self_1_6,
                        } => match *self {
                            Value {
                                string_value: ref __self_0_0,
                                float_value: ref __self_0_1,
                                double_value: ref __self_0_2,
                                int_value: ref __self_0_3,
                                uint_value: ref __self_0_4,
                                sint_value: ref __self_0_5,
                                bool_value: ref __self_0_6,
                            } => {
                                (*__self_0_0) != (*__self_1_0)
                                    || (*__self_0_1) != (*__self_1_1)
                                    || (*__self_0_2) != (*__self_1_2)
                                    || (*__self_0_3) != (*__self_1_3)
                                    || (*__self_0_4) != (*__self_1_4)
                                    || (*__self_0_5) != (*__self_1_5)
                                    || (*__self_0_6) != (*__self_1_6)
                            }
                        },
                    }
                }
            }
            impl ::prost::Message for Value {
                #[allow(unused_variables)]
                fn encode_raw<B>(&self, buf: &mut B)
                where
                    B: ::prost::bytes::BufMut,
                {
                    if let ::core::option::Option::Some(ref value) = self.string_value {
                        ::prost::encoding::string::encode(1u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.float_value {
                        ::prost::encoding::float::encode(2u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.double_value {
                        ::prost::encoding::double::encode(3u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.int_value {
                        ::prost::encoding::int64::encode(4u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.uint_value {
                        ::prost::encoding::uint64::encode(5u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.sint_value {
                        ::prost::encoding::sint64::encode(6u32, value, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.bool_value {
                        ::prost::encoding::bool::encode(7u32, value, buf);
                    }
                }
                #[allow(unused_variables)]
                fn merge_field<B>(
                    &mut self,
                    tag: u32,
                    wire_type: ::prost::encoding::WireType,
                    buf: &mut B,
                    ctx: ::prost::encoding::DecodeContext,
                ) -> ::core::result::Result<(), ::prost::DecodeError>
                where
                    B: ::prost::bytes::Buf,
                {
                    const STRUCT_NAME: &'static str = "Value";
                    match tag {
                        1u32 => {
                            let mut value = &mut self.string_value;
                            ::prost::encoding::string::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "string_value");
                                error
                            })
                        }
                        2u32 => {
                            let mut value = &mut self.float_value;
                            ::prost::encoding::float::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "float_value");
                                error
                            })
                        }
                        3u32 => {
                            let mut value = &mut self.double_value;
                            ::prost::encoding::double::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "double_value");
                                error
                            })
                        }
                        4u32 => {
                            let mut value = &mut self.int_value;
                            ::prost::encoding::int64::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "int_value");
                                error
                            })
                        }
                        5u32 => {
                            let mut value = &mut self.uint_value;
                            ::prost::encoding::uint64::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "uint_value");
                                error
                            })
                        }
                        6u32 => {
                            let mut value = &mut self.sint_value;
                            ::prost::encoding::sint64::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "sint_value");
                                error
                            })
                        }
                        7u32 => {
                            let mut value = &mut self.bool_value;
                            ::prost::encoding::bool::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "bool_value");
                                error
                            })
                        }
                        _ => ::prost::encoding::skip_field(wire_type, tag, buf, ctx),
                    }
                }
                #[inline]
                fn encoded_len(&self) -> usize {
                    0 + self.string_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::string::encoded_len(1u32, value)
                    }) + self.float_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::float::encoded_len(2u32, value)
                    }) + self.double_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::double::encoded_len(3u32, value)
                    }) + self.int_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::int64::encoded_len(4u32, value)
                    }) + self.uint_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::uint64::encoded_len(5u32, value)
                    }) + self.sint_value.as_ref().map_or(0, |value| {
                        ::prost::encoding::sint64::encoded_len(6u32, value)
                    }) + self
                        .bool_value
                        .as_ref()
                        .map_or(0, |value| ::prost::encoding::bool::encoded_len(7u32, value))
                }
                fn clear(&mut self) {
                    self.string_value = ::core::option::Option::None;
                    self.float_value = ::core::option::Option::None;
                    self.double_value = ::core::option::Option::None;
                    self.int_value = ::core::option::Option::None;
                    self.uint_value = ::core::option::Option::None;
                    self.sint_value = ::core::option::Option::None;
                    self.bool_value = ::core::option::Option::None;
                }
            }
            impl ::core::default::Default for Value {
                fn default() -> Self {
                    Value {
                        string_value: ::core::option::Option::None,
                        float_value: ::core::option::Option::None,
                        double_value: ::core::option::Option::None,
                        int_value: ::core::option::Option::None,
                        uint_value: ::core::option::Option::None,
                        sint_value: ::core::option::Option::None,
                        bool_value: ::core::option::Option::None,
                    }
                }
            }
            impl ::core::fmt::Debug for Value {
                fn fmt(&self, f: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                    let mut builder = f.debug_struct("Value");
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(
                                &'a ::core::option::Option<::prost::alloc::string::String>,
                            );
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.string_value)
                        };
                        builder.field("string_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<f32>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.float_value)
                        };
                        builder.field("float_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<f64>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.double_value)
                        };
                        builder.field("double_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<i64>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.int_value)
                        };
                        builder.field("int_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<u64>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.uint_value)
                        };
                        builder.field("uint_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<i64>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.sint_value)
                        };
                        builder.field("sint_value", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<bool>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.bool_value)
                        };
                        builder.field("bool_value", &wrapper)
                    };
                    builder.finish()
                }
            }
            #[allow(dead_code)]
            impl Value {
                ///Returns the value of `string_value`, or the default value if `string_value` is unset.
                pub fn string_value(&self) -> &str {
                    match self.string_value {
                        ::core::option::Option::Some(ref val) => &val[..],
                        ::core::option::Option::None => "",
                    }
                }
                ///Returns the value of `float_value`, or the default value if `float_value` is unset.
                pub fn float_value(&self) -> f32 {
                    match self.float_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0f32,
                    }
                }
                ///Returns the value of `double_value`, or the default value if `double_value` is unset.
                pub fn double_value(&self) -> f64 {
                    match self.double_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0f64,
                    }
                }
                ///Returns the value of `int_value`, or the default value if `int_value` is unset.
                pub fn int_value(&self) -> i64 {
                    match self.int_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0i64,
                    }
                }
                ///Returns the value of `uint_value`, or the default value if `uint_value` is unset.
                pub fn uint_value(&self) -> u64 {
                    match self.uint_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0u64,
                    }
                }
                ///Returns the value of `sint_value`, or the default value if `sint_value` is unset.
                pub fn sint_value(&self) -> i64 {
                    match self.sint_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0i64,
                    }
                }
                ///Returns the value of `bool_value`, or the default value if `bool_value` is unset.
                pub fn bool_value(&self) -> bool {
                    match self.bool_value {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => false,
                    }
                }
            }
            /// Features are described in section 4.2 of the specification
            pub struct Feature {
                #[prost(uint64, optional, tag = "1", default = "0")]
                pub id: ::core::option::Option<u64>,
                /// Tags of this feature are encoded as repeated pairs of
                /// integers.
                /// A detailed description of tags is located in sections
                /// 4.2 and 4.4 of the specification
                #[prost(uint32, repeated, tag = "2")]
                pub tags: ::prost::alloc::vec::Vec<u32>,
                /// The type of geometry stored in this feature.
                #[prost(enumeration = "GeomType", optional, tag = "3", default = "Unknown")]
                pub r#type: ::core::option::Option<i32>,
                /// Contains a stream of commands and parameters (vertices).
                /// A detailed description on geometry encoding is located in
                /// section 4.3 of the specification.
                #[prost(uint32, repeated, tag = "4")]
                pub geometry: ::prost::alloc::vec::Vec<u32>,
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::clone::Clone for Feature {
                #[inline]
                fn clone(&self) -> Feature {
                    match *self {
                        Feature {
                            id: ref __self_0_0,
                            tags: ref __self_0_1,
                            r#type: ref __self_0_2,
                            geometry: ref __self_0_3,
                        } => Feature {
                            id: ::core::clone::Clone::clone(&(*__self_0_0)),
                            tags: ::core::clone::Clone::clone(&(*__self_0_1)),
                            r#type: ::core::clone::Clone::clone(&(*__self_0_2)),
                            geometry: ::core::clone::Clone::clone(&(*__self_0_3)),
                        },
                    }
                }
            }
            impl ::core::marker::StructuralPartialEq for Feature {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::PartialEq for Feature {
                #[inline]
                fn eq(&self, other: &Feature) -> bool {
                    match *other {
                        Feature {
                            id: ref __self_1_0,
                            tags: ref __self_1_1,
                            r#type: ref __self_1_2,
                            geometry: ref __self_1_3,
                        } => match *self {
                            Feature {
                                id: ref __self_0_0,
                                tags: ref __self_0_1,
                                r#type: ref __self_0_2,
                                geometry: ref __self_0_3,
                            } => {
                                (*__self_0_0) == (*__self_1_0)
                                    && (*__self_0_1) == (*__self_1_1)
                                    && (*__self_0_2) == (*__self_1_2)
                                    && (*__self_0_3) == (*__self_1_3)
                            }
                        },
                    }
                }
                #[inline]
                fn ne(&self, other: &Feature) -> bool {
                    match *other {
                        Feature {
                            id: ref __self_1_0,
                            tags: ref __self_1_1,
                            r#type: ref __self_1_2,
                            geometry: ref __self_1_3,
                        } => match *self {
                            Feature {
                                id: ref __self_0_0,
                                tags: ref __self_0_1,
                                r#type: ref __self_0_2,
                                geometry: ref __self_0_3,
                            } => {
                                (*__self_0_0) != (*__self_1_0)
                                    || (*__self_0_1) != (*__self_1_1)
                                    || (*__self_0_2) != (*__self_1_2)
                                    || (*__self_0_3) != (*__self_1_3)
                            }
                        },
                    }
                }
            }
            impl ::prost::Message for Feature {
                #[allow(unused_variables)]
                fn encode_raw<B>(&self, buf: &mut B)
                where
                    B: ::prost::bytes::BufMut,
                {
                    if let ::core::option::Option::Some(ref value) = self.id {
                        ::prost::encoding::uint64::encode(1u32, value, buf);
                    }
                    ::prost::encoding::uint32::encode_packed(2u32, &self.tags, buf);
                    if let ::core::option::Option::Some(ref value) = self.r#type {
                        ::prost::encoding::int32::encode(3u32, value, buf);
                    }
                    ::prost::encoding::uint32::encode_packed(4u32, &self.geometry, buf);
                }
                #[allow(unused_variables)]
                fn merge_field<B>(
                    &mut self,
                    tag: u32,
                    wire_type: ::prost::encoding::WireType,
                    buf: &mut B,
                    ctx: ::prost::encoding::DecodeContext,
                ) -> ::core::result::Result<(), ::prost::DecodeError>
                where
                    B: ::prost::bytes::Buf,
                {
                    const STRUCT_NAME: &'static str = "Feature";
                    match tag {
                        1u32 => {
                            let mut value = &mut self.id;
                            ::prost::encoding::uint64::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "id");
                                error
                            })
                        }
                        2u32 => {
                            let mut value = &mut self.tags;
                            ::prost::encoding::uint32::merge_repeated(wire_type, value, buf, ctx)
                                .map_err(|mut error| {
                                    error.push(STRUCT_NAME, "tags");
                                    error
                                })
                        }
                        3u32 => {
                            let mut value = &mut self.r#type;
                            ::prost::encoding::int32::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "r#type");
                                error
                            })
                        }
                        4u32 => {
                            let mut value = &mut self.geometry;
                            ::prost::encoding::uint32::merge_repeated(wire_type, value, buf, ctx)
                                .map_err(|mut error| {
                                    error.push(STRUCT_NAME, "geometry");
                                    error
                                })
                        }
                        _ => ::prost::encoding::skip_field(wire_type, tag, buf, ctx),
                    }
                }
                #[inline]
                fn encoded_len(&self) -> usize {
                    0 + self.id.as_ref().map_or(0, |value| {
                        ::prost::encoding::uint64::encoded_len(1u32, value)
                    }) + ::prost::encoding::uint32::encoded_len_packed(2u32, &self.tags)
                        + self.r#type.as_ref().map_or(0, |value| {
                            ::prost::encoding::int32::encoded_len(3u32, value)
                        })
                        + ::prost::encoding::uint32::encoded_len_packed(4u32, &self.geometry)
                }
                fn clear(&mut self) {
                    self.id = ::core::option::Option::None;
                    self.tags.clear();
                    self.r#type = ::core::option::Option::None;
                    self.geometry.clear();
                }
            }
            impl ::core::default::Default for Feature {
                fn default() -> Self {
                    Feature {
                        id: ::core::option::Option::None,
                        tags: ::prost::alloc::vec::Vec::new(),
                        r#type: ::core::option::Option::None,
                        geometry: ::prost::alloc::vec::Vec::new(),
                    }
                }
            }
            impl ::core::fmt::Debug for Feature {
                fn fmt(&self, f: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                    let mut builder = f.debug_struct("Feature");
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<u64>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.id)
                        };
                        builder.field("id", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::prost::alloc::vec::Vec<u32>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    let mut vec_builder = f.debug_list();
                                    for v in self.0 {
                                        fn Inner<T>(v: T) -> T {
                                            v
                                        }
                                        vec_builder.entry(&Inner(v));
                                    }
                                    vec_builder.finish()
                                }
                            }
                            ScalarWrapper(&self.tags)
                        };
                        builder.field("tags", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<i32>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    struct Inner<'a>(&'a i32);
                                    impl<'a> ::core::fmt::Debug for Inner<'a> {
                                        fn fmt(
                                            &self,
                                            f: &mut ::core::fmt::Formatter,
                                        ) -> ::core::fmt::Result
                                        {
                                            match GeomType::from_i32(*self.0) {
                                                None => ::core::fmt::Debug::fmt(&self.0, f),
                                                Some(en) => ::core::fmt::Debug::fmt(&en, f),
                                            }
                                        }
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.r#type)
                        };
                        builder.field("r#type", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::prost::alloc::vec::Vec<u32>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    let mut vec_builder = f.debug_list();
                                    for v in self.0 {
                                        fn Inner<T>(v: T) -> T {
                                            v
                                        }
                                        vec_builder.entry(&Inner(v));
                                    }
                                    vec_builder.finish()
                                }
                            }
                            ScalarWrapper(&self.geometry)
                        };
                        builder.field("geometry", &wrapper)
                    };
                    builder.finish()
                }
            }
            #[allow(dead_code)]
            impl Feature {
                ///Returns the value of `id`, or the default value if `id` is unset.
                pub fn id(&self) -> u64 {
                    match self.id {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 0u64,
                    }
                }
                ///Returns the enum value of `type`, or the default if the field is unset or set to an invalid enum value.
                pub fn r#type(&self) -> GeomType {
                    self.r#type
                        .and_then(GeomType::from_i32)
                        .unwrap_or(GeomType::Unknown)
                }
                ///Sets `type` to the provided enum value.
                pub fn set_type(&mut self, value: GeomType) {
                    self.r#type = ::core::option::Option::Some(value as i32);
                }
            }
            /// Layers are described in section 4.1 of the specification
            pub struct Layer {
                /// Any compliant implementation must first read the version
                /// number encoded in this message and choose the correct
                /// implementation for this version number before proceeding to
                /// decode other parts of this message.
                #[prost(uint32, required, tag = "15", default = "1")]
                pub version: u32,
                #[prost(string, required, tag = "1")]
                pub name: ::prost::alloc::string::String,
                /// The actual features in this tile.
                #[prost(message, repeated, tag = "2")]
                pub features: ::prost::alloc::vec::Vec<Feature>,
                /// Dictionary encoding for keys
                #[prost(string, repeated, tag = "3")]
                pub keys: ::prost::alloc::vec::Vec<::prost::alloc::string::String>,
                /// Dictionary encoding for values
                #[prost(message, repeated, tag = "4")]
                pub values: ::prost::alloc::vec::Vec<Value>,
                /// Although this is an "optional" field it is required by the specification.
                /// See <https://github.com/mapbox/vector-tile-spec/issues/47>
                #[prost(uint32, optional, tag = "5", default = "4096")]
                pub extent: ::core::option::Option<u32>,
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::clone::Clone for Layer {
                #[inline]
                fn clone(&self) -> Layer {
                    match *self {
                        Layer {
                            version: ref __self_0_0,
                            name: ref __self_0_1,
                            features: ref __self_0_2,
                            keys: ref __self_0_3,
                            values: ref __self_0_4,
                            extent: ref __self_0_5,
                        } => Layer {
                            version: ::core::clone::Clone::clone(&(*__self_0_0)),
                            name: ::core::clone::Clone::clone(&(*__self_0_1)),
                            features: ::core::clone::Clone::clone(&(*__self_0_2)),
                            keys: ::core::clone::Clone::clone(&(*__self_0_3)),
                            values: ::core::clone::Clone::clone(&(*__self_0_4)),
                            extent: ::core::clone::Clone::clone(&(*__self_0_5)),
                        },
                    }
                }
            }
            impl ::core::marker::StructuralPartialEq for Layer {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::PartialEq for Layer {
                #[inline]
                fn eq(&self, other: &Layer) -> bool {
                    match *other {
                        Layer {
                            version: ref __self_1_0,
                            name: ref __self_1_1,
                            features: ref __self_1_2,
                            keys: ref __self_1_3,
                            values: ref __self_1_4,
                            extent: ref __self_1_5,
                        } => match *self {
                            Layer {
                                version: ref __self_0_0,
                                name: ref __self_0_1,
                                features: ref __self_0_2,
                                keys: ref __self_0_3,
                                values: ref __self_0_4,
                                extent: ref __self_0_5,
                            } => {
                                (*__self_0_0) == (*__self_1_0)
                                    && (*__self_0_1) == (*__self_1_1)
                                    && (*__self_0_2) == (*__self_1_2)
                                    && (*__self_0_3) == (*__self_1_3)
                                    && (*__self_0_4) == (*__self_1_4)
                                    && (*__self_0_5) == (*__self_1_5)
                            }
                        },
                    }
                }
                #[inline]
                fn ne(&self, other: &Layer) -> bool {
                    match *other {
                        Layer {
                            version: ref __self_1_0,
                            name: ref __self_1_1,
                            features: ref __self_1_2,
                            keys: ref __self_1_3,
                            values: ref __self_1_4,
                            extent: ref __self_1_5,
                        } => match *self {
                            Layer {
                                version: ref __self_0_0,
                                name: ref __self_0_1,
                                features: ref __self_0_2,
                                keys: ref __self_0_3,
                                values: ref __self_0_4,
                                extent: ref __self_0_5,
                            } => {
                                (*__self_0_0) != (*__self_1_0)
                                    || (*__self_0_1) != (*__self_1_1)
                                    || (*__self_0_2) != (*__self_1_2)
                                    || (*__self_0_3) != (*__self_1_3)
                                    || (*__self_0_4) != (*__self_1_4)
                                    || (*__self_0_5) != (*__self_1_5)
                            }
                        },
                    }
                }
            }
            impl ::prost::Message for Layer {
                #[allow(unused_variables)]
                fn encode_raw<B>(&self, buf: &mut B)
                where
                    B: ::prost::bytes::BufMut,
                {
                    ::prost::encoding::string::encode(1u32, &self.name, buf);
                    for msg in &self.features {
                        ::prost::encoding::message::encode(2u32, msg, buf);
                    }
                    ::prost::encoding::string::encode_repeated(3u32, &self.keys, buf);
                    for msg in &self.values {
                        ::prost::encoding::message::encode(4u32, msg, buf);
                    }
                    if let ::core::option::Option::Some(ref value) = self.extent {
                        ::prost::encoding::uint32::encode(5u32, value, buf);
                    }
                    ::prost::encoding::uint32::encode(15u32, &self.version, buf);
                }
                #[allow(unused_variables)]
                fn merge_field<B>(
                    &mut self,
                    tag: u32,
                    wire_type: ::prost::encoding::WireType,
                    buf: &mut B,
                    ctx: ::prost::encoding::DecodeContext,
                ) -> ::core::result::Result<(), ::prost::DecodeError>
                where
                    B: ::prost::bytes::Buf,
                {
                    const STRUCT_NAME: &'static str = "Layer";
                    match tag {
                        1u32 => {
                            let mut value = &mut self.name;
                            ::prost::encoding::string::merge(wire_type, value, buf, ctx).map_err(
                                |mut error| {
                                    error.push(STRUCT_NAME, "name");
                                    error
                                },
                            )
                        }
                        2u32 => {
                            let mut value = &mut self.features;
                            ::prost::encoding::message::merge_repeated(wire_type, value, buf, ctx)
                                .map_err(|mut error| {
                                    error.push(STRUCT_NAME, "features");
                                    error
                                })
                        }
                        3u32 => {
                            let mut value = &mut self.keys;
                            ::prost::encoding::string::merge_repeated(wire_type, value, buf, ctx)
                                .map_err(|mut error| {
                                    error.push(STRUCT_NAME, "keys");
                                    error
                                })
                        }
                        4u32 => {
                            let mut value = &mut self.values;
                            ::prost::encoding::message::merge_repeated(wire_type, value, buf, ctx)
                                .map_err(|mut error| {
                                    error.push(STRUCT_NAME, "values");
                                    error
                                })
                        }
                        5u32 => {
                            let mut value = &mut self.extent;
                            ::prost::encoding::uint32::merge(
                                wire_type,
                                value.get_or_insert_with(::core::default::Default::default),
                                buf,
                                ctx,
                            )
                            .map_err(|mut error| {
                                error.push(STRUCT_NAME, "extent");
                                error
                            })
                        }
                        15u32 => {
                            let mut value = &mut self.version;
                            ::prost::encoding::uint32::merge(wire_type, value, buf, ctx).map_err(
                                |mut error| {
                                    error.push(STRUCT_NAME, "version");
                                    error
                                },
                            )
                        }
                        _ => ::prost::encoding::skip_field(wire_type, tag, buf, ctx),
                    }
                }
                #[inline]
                fn encoded_len(&self) -> usize {
                    0 + ::prost::encoding::string::encoded_len(1u32, &self.name)
                        + ::prost::encoding::message::encoded_len_repeated(2u32, &self.features)
                        + ::prost::encoding::string::encoded_len_repeated(3u32, &self.keys)
                        + ::prost::encoding::message::encoded_len_repeated(4u32, &self.values)
                        + self.extent.as_ref().map_or(0, |value| {
                            ::prost::encoding::uint32::encoded_len(5u32, value)
                        })
                        + ::prost::encoding::uint32::encoded_len(15u32, &self.version)
                }
                fn clear(&mut self) {
                    self.name.clear();
                    self.features.clear();
                    self.keys.clear();
                    self.values.clear();
                    self.extent = ::core::option::Option::None;
                    self.version = 1u32;
                }
            }
            impl ::core::default::Default for Layer {
                fn default() -> Self {
                    Layer {
                        name: ::prost::alloc::string::String::new(),
                        features: ::core::default::Default::default(),
                        keys: ::prost::alloc::vec::Vec::new(),
                        values: ::core::default::Default::default(),
                        extent: ::core::option::Option::None,
                        version: 1u32,
                    }
                }
            }
            impl ::core::fmt::Debug for Layer {
                fn fmt(&self, f: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                    let mut builder = f.debug_struct("Layer");
                    let builder = {
                        let wrapper = {
                            fn ScalarWrapper<T>(v: T) -> T {
                                v
                            }
                            ScalarWrapper(&self.version)
                        };
                        builder.field("version", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            fn ScalarWrapper<T>(v: T) -> T {
                                v
                            }
                            ScalarWrapper(&self.name)
                        };
                        builder.field("name", &wrapper)
                    };
                    let builder = {
                        let wrapper = &self.features;
                        builder.field("features", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(
                                &'a ::prost::alloc::vec::Vec<::prost::alloc::string::String>,
                            );
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    let mut vec_builder = f.debug_list();
                                    for v in self.0 {
                                        fn Inner<T>(v: T) -> T {
                                            v
                                        }
                                        vec_builder.entry(&Inner(v));
                                    }
                                    vec_builder.finish()
                                }
                            }
                            ScalarWrapper(&self.keys)
                        };
                        builder.field("keys", &wrapper)
                    };
                    let builder = {
                        let wrapper = &self.values;
                        builder.field("values", &wrapper)
                    };
                    let builder = {
                        let wrapper = {
                            struct ScalarWrapper<'a>(&'a ::core::option::Option<u32>);
                            impl<'a> ::core::fmt::Debug for ScalarWrapper<'a> {
                                fn fmt(
                                    &self,
                                    f: &mut ::core::fmt::Formatter,
                                ) -> ::core::fmt::Result {
                                    fn Inner<T>(v: T) -> T {
                                        v
                                    }
                                    ::core::fmt::Debug::fmt(&self.0.as_ref().map(Inner), f)
                                }
                            }
                            ScalarWrapper(&self.extent)
                        };
                        builder.field("extent", &wrapper)
                    };
                    builder.finish()
                }
            }
            #[allow(dead_code)]
            impl Layer {
                ///Returns the value of `extent`, or the default value if `extent` is unset.
                pub fn extent(&self) -> u32 {
                    match self.extent {
                        ::core::option::Option::Some(val) => val,
                        ::core::option::Option::None => 4096u32,
                    }
                }
            }
            /// GeomType is described in section 4.3.4 of the specification
            #[repr(i32)]
            pub enum GeomType {
                Unknown = 0,
                Point = 1,
                Linestring = 2,
                Polygon = 3,
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::clone::Clone for GeomType {
                #[inline]
                fn clone(&self) -> GeomType {
                    {
                        *self
                    }
                }
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::marker::Copy for GeomType {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::fmt::Debug for GeomType {
                fn fmt(&self, f: &mut ::core::fmt::Formatter) -> ::core::fmt::Result {
                    match (&*self,) {
                        (&GeomType::Unknown,) => ::core::fmt::Formatter::write_str(f, "Unknown"),
                        (&GeomType::Point,) => ::core::fmt::Formatter::write_str(f, "Point"),
                        (&GeomType::Linestring,) => {
                            ::core::fmt::Formatter::write_str(f, "Linestring")
                        }
                        (&GeomType::Polygon,) => ::core::fmt::Formatter::write_str(f, "Polygon"),
                    }
                }
            }
            impl ::core::marker::StructuralPartialEq for GeomType {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::PartialEq for GeomType {
                #[inline]
                fn eq(&self, other: &GeomType) -> bool {
                    {
                        let __self_vi = ::core::intrinsics::discriminant_value(&*self);
                        let __arg_1_vi = ::core::intrinsics::discriminant_value(&*other);
                        if true && __self_vi == __arg_1_vi {
                            match (&*self, &*other) {
                                _ => true,
                            }
                        } else {
                            false
                        }
                    }
                }
            }
            impl ::core::marker::StructuralEq for GeomType {}
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::Eq for GeomType {
                #[inline]
                #[doc(hidden)]
                #[no_coverage]
                fn assert_receiver_is_total_eq(&self) -> () {
                    {}
                }
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::hash::Hash for GeomType {
                fn hash<__H: ::core::hash::Hasher>(&self, state: &mut __H) -> () {
                    match (&*self,) {
                        _ => ::core::hash::Hash::hash(
                            &::core::intrinsics::discriminant_value(self),
                            state,
                        ),
                    }
                }
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::PartialOrd for GeomType {
                #[inline]
                fn partial_cmp(
                    &self,
                    other: &GeomType,
                ) -> ::core::option::Option<::core::cmp::Ordering> {
                    {
                        let __self_vi = ::core::intrinsics::discriminant_value(&*self);
                        let __arg_1_vi = ::core::intrinsics::discriminant_value(&*other);
                        if true && __self_vi == __arg_1_vi {
                            match (&*self, &*other) {
                                _ => ::core::option::Option::Some(::core::cmp::Ordering::Equal),
                            }
                        } else {
                            ::core::cmp::PartialOrd::partial_cmp(&__self_vi, &__arg_1_vi)
                        }
                    }
                }
            }
            #[automatically_derived]
            #[allow(unused_qualifications)]
            impl ::core::cmp::Ord for GeomType {
                #[inline]
                fn cmp(&self, other: &GeomType) -> ::core::cmp::Ordering {
                    {
                        let __self_vi = ::core::intrinsics::discriminant_value(&*self);
                        let __arg_1_vi = ::core::intrinsics::discriminant_value(&*other);
                        if true && __self_vi == __arg_1_vi {
                            match (&*self, &*other) {
                                _ => ::core::cmp::Ordering::Equal,
                            }
                        } else {
                            ::core::cmp::Ord::cmp(&__self_vi, &__arg_1_vi)
                        }
                    }
                }
            }
            impl GeomType {
                ///Returns `true` if `value` is a variant of `GeomType`.
                pub fn is_valid(value: i32) -> bool {
                    match value {
                        0 => true,
                        1 => true,
                        2 => true,
                        3 => true,
                        _ => false,
                    }
                }
                ///Converts an `i32` to a `GeomType`, or `None` if `value` is not a valid variant.
                pub fn from_i32(value: i32) -> ::core::option::Option<GeomType> {
                    match value {
                        0 => ::core::option::Option::Some(GeomType::Unknown),
                        1 => ::core::option::Option::Some(GeomType::Point),
                        2 => ::core::option::Option::Some(GeomType::Linestring),
                        3 => ::core::option::Option::Some(GeomType::Polygon),
                        _ => ::core::option::Option::None,
                    }
                }
            }
            impl ::core::default::Default for GeomType {
                fn default() -> GeomType {
                    GeomType::Unknown
                }
            }
            impl ::core::convert::From<GeomType> for i32 {
                fn from(value: GeomType) -> i32 {
                    value as i32
                }
            }
        }
    }
    pub use mvt_reader::*;
}
