#![allow(clippy::assign_op_pattern)]
use crate::{simd::*, *};
use std::ops::{Add, AddAssign, Div, DivAssign, Mul, MulAssign, Neg, Sub, SubAssign};

#[derive(Clone, Copy, Debug)]
pub struct Scalar {
    /// 1
    pub g0: f32,
}

#[derive(Clone, Copy, Debug)]
pub struct DualNumber {
    /// 1, e01
    pub g0: Simd32x2,
}

impl Zero for Scalar {
    fn zero() -> Self {
        Scalar { g0: 0.0 }
    }
}

impl One for Scalar {
    fn one() -> Self {
        Scalar { g0: 1.0 }
    }
}

impl Neg for Scalar {
    type Output = Scalar;

    fn neg(self) -> Scalar {
        Scalar { g0: self.g0 * -1.0 }
    }
}

impl Automorphism for Scalar {
    type Output = Scalar;

    fn automorphism(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Reversal for Scalar {
    type Output = Scalar;

    fn reversal(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Conjugation for Scalar {
    type Output = Scalar;

    fn conjugation(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Add<Scalar> for Scalar {
    type Output = Scalar;

    fn add(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<Scalar> for Scalar {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for Scalar {
    type Output = Scalar;

    fn sub(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<Scalar> for Scalar {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn geometric_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl OuterProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn outer_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl InnerProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn inner_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl LeftContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl RightContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl ScalarProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl Add<DualNumber> for Scalar {
    type Output = DualNumber;

    fn add(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) + other.g0 }
    }
}

impl Sub<DualNumber> for Scalar {
    type Output = DualNumber;

    fn sub(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) - other.g0 }
    }
}

impl GeometricProduct<DualNumber> for Scalar {
    type Output = DualNumber;

    fn geometric_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RegressiveProduct<DualNumber> for Scalar {
    type Output = Scalar;

    fn regressive_product(self, other: DualNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[1] }
    }
}

impl OuterProduct<DualNumber> for Scalar {
    type Output = DualNumber;

    fn outer_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl InnerProduct<DualNumber> for Scalar {
    type Output = DualNumber;

    fn inner_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl LeftContraction<DualNumber> for Scalar {
    type Output = DualNumber;

    fn left_contraction(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RightContraction<DualNumber> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: DualNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl ScalarProduct<DualNumber> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: DualNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl SquaredMagnitude for Scalar {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for Scalar {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for Scalar {
    type Output = Scalar;

    fn signum(self) -> Scalar {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for Scalar {
    type Output = Scalar;

    fn inverse(self) -> Scalar {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Zero for DualNumber {
    fn zero() -> Self {
        DualNumber { g0: Simd32x2::from(0.0) }
    }
}

impl One for DualNumber {
    fn one() -> Self {
        DualNumber { g0: Simd32x2::from([1.0, 0.0]) }
    }
}

impl Neg for DualNumber {
    type Output = DualNumber;

    fn neg(self) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(-1.0) }
    }
}

impl Automorphism for DualNumber {
    type Output = DualNumber;

    fn automorphism(self) -> DualNumber {
        DualNumber { g0: self.g0 }
    }
}

impl Reversal for DualNumber {
    type Output = DualNumber;

    fn reversal(self) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Conjugation for DualNumber {
    type Output = DualNumber;

    fn conjugation(self) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Dual for DualNumber {
    type Output = DualNumber;

    fn dual(self) -> DualNumber {
        DualNumber { g0: swizzle!(self.g0, 1, 0) }
    }
}

impl Into<Scalar> for DualNumber {
    fn into(self) -> Scalar {
        Scalar { g0: self.g0[0] }
    }
}

impl Add<Scalar> for DualNumber {
    type Output = DualNumber;

    fn add(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 + Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl AddAssign<Scalar> for DualNumber {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for DualNumber {
    type Output = DualNumber;

    fn sub(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 - Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl SubAssign<Scalar> for DualNumber {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for DualNumber {
    type Output = DualNumber;

    fn geometric_product(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl RegressiveProduct<Scalar> for DualNumber {
    type Output = Scalar;

    fn regressive_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[1] * other.g0 }
    }
}

impl OuterProduct<Scalar> for DualNumber {
    type Output = DualNumber;

    fn outer_product(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl InnerProduct<Scalar> for DualNumber {
    type Output = DualNumber;

    fn inner_product(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl LeftContraction<Scalar> for DualNumber {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl RightContraction<Scalar> for DualNumber {
    type Output = DualNumber;

    fn right_contraction(self, other: Scalar) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl ScalarProduct<Scalar> for DualNumber {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl Add<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn add(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<DualNumber> for DualNumber {
    fn add_assign(&mut self, other: DualNumber) {
        *self = (*self).add(other);
    }
}

impl Sub<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn sub(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<DualNumber> for DualNumber {
    fn sub_assign(&mut self, other: DualNumber) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn geometric_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + self.g0 * Simd32x2::from(other.g0[0]) * Simd32x2::from([0.0, 1.0]) }
    }
}

impl RegressiveProduct<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn regressive_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0[1]) * other.g0 + Simd32x2::from(self.g0[0]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl OuterProduct<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn outer_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + self.g0 * Simd32x2::from(other.g0[0]) * Simd32x2::from([0.0, 1.0]) }
    }
}

impl InnerProduct<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn inner_product(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + self.g0 * Simd32x2::from(other.g0[0]) * Simd32x2::from([0.0, 1.0]) }
    }
}

impl LeftContraction<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn left_contraction(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 }
    }
}

impl RightContraction<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn right_contraction(self, other: DualNumber) -> DualNumber {
        DualNumber { g0: self.g0 * Simd32x2::from(other.g0[0]) }
    }
}

impl ScalarProduct<DualNumber> for DualNumber {
    type Output = Scalar;

    fn scalar_product(self, other: DualNumber) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0[0] }
    }
}

impl SquaredMagnitude for DualNumber {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for DualNumber {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for DualNumber {
    type Output = DualNumber;

    fn signum(self) -> DualNumber {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for DualNumber {
    type Output = DualNumber;

    fn inverse(self) -> DualNumber {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Mul<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn mul(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<DualNumber> for DualNumber {
    fn mul_assign(&mut self, other: DualNumber) {
        *self = (*self).mul(other);
    }
}

impl Powi for DualNumber {
    type Output = DualNumber;

    fn powi(self, exponent: isize) -> DualNumber {
        if exponent == 0 {
            return DualNumber::one();
        }
        let mut x: DualNumber = if exponent < 0 { self.inverse() } else { self };
        let mut y: DualNumber = DualNumber::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn div(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<DualNumber> for DualNumber {
    fn div_assign(&mut self, other: DualNumber) {
        *self = (*self).div(other);
    }
}

impl Transformation<DualNumber> for DualNumber {
    type Output = DualNumber;

    fn transformation(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<Scalar> for DualNumber {
    type Output = DualNumber;

    fn mul(self, other: Scalar) -> DualNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for DualNumber {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Div<Scalar> for DualNumber {
    type Output = DualNumber;

    fn div(self, other: Scalar) -> DualNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for DualNumber {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for DualNumber {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal()).into()
    }
}

impl Mul<DualNumber> for Scalar {
    type Output = DualNumber;

    fn mul(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other)
    }
}

impl Div<DualNumber> for Scalar {
    type Output = DualNumber;

    fn div(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other.inverse())
    }
}

impl Transformation<DualNumber> for Scalar {
    type Output = DualNumber;

    fn transformation(self, other: DualNumber) -> DualNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<Scalar> for Scalar {
    type Output = Scalar;

    fn mul(self, other: Scalar) -> Scalar {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for Scalar {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Powi for Scalar {
    type Output = Scalar;

    fn powi(self, exponent: isize) -> Scalar {
        if exponent == 0 {
            return Scalar::one();
        }
        let mut x: Scalar = if exponent < 0 { self.inverse() } else { self };
        let mut y: Scalar = Scalar::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<Scalar> for Scalar {
    type Output = Scalar;

    fn div(self, other: Scalar) -> Scalar {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for Scalar {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for Scalar {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

