#![allow(clippy::assign_op_pattern)]
use crate::{simd::*, *};
use std::ops::{Add, AddAssign, Div, DivAssign, Mul, MulAssign, Neg, Sub, SubAssign};

#[derive(Clone, Copy, Debug)]
pub struct Scalar {
    /// 1
    pub g0: f32,
}

#[derive(Clone, Copy, Debug)]
pub struct SplitComplexNumber {
    /// 1, e01
    pub g0: Simd32x2,
}

impl Zero for Scalar {
    fn zero() -> Self {
        Scalar { g0: 0.0 }
    }
}

impl One for Scalar {
    fn one() -> Self {
        Scalar { g0: 1.0 }
    }
}

impl Neg for Scalar {
    type Output = Scalar;

    fn neg(self) -> Scalar {
        Scalar { g0: self.g0 * -1.0 }
    }
}

impl Automorphism for Scalar {
    type Output = Scalar;

    fn automorphism(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Reversal for Scalar {
    type Output = Scalar;

    fn reversal(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Conjugation for Scalar {
    type Output = Scalar;

    fn conjugation(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Add<Scalar> for Scalar {
    type Output = Scalar;

    fn add(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<Scalar> for Scalar {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for Scalar {
    type Output = Scalar;

    fn sub(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<Scalar> for Scalar {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn geometric_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl OuterProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn outer_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl InnerProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn inner_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl LeftContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl RightContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl ScalarProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl Add<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn add(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) + other.g0 }
    }
}

impl Sub<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn sub(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) - other.g0 }
    }
}

impl GeometricProduct<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn geometric_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RegressiveProduct<SplitComplexNumber> for Scalar {
    type Output = Scalar;

    fn regressive_product(self, other: SplitComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[1] }
    }
}

impl OuterProduct<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn outer_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl InnerProduct<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn inner_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl LeftContraction<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn left_contraction(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RightContraction<SplitComplexNumber> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: SplitComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl ScalarProduct<SplitComplexNumber> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: SplitComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl SquaredMagnitude for Scalar {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for Scalar {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for Scalar {
    type Output = Scalar;

    fn signum(self) -> Scalar {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for Scalar {
    type Output = Scalar;

    fn inverse(self) -> Scalar {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Zero for SplitComplexNumber {
    fn zero() -> Self {
        SplitComplexNumber { g0: Simd32x2::from(0.0) }
    }
}

impl One for SplitComplexNumber {
    fn one() -> Self {
        SplitComplexNumber { g0: Simd32x2::from([1.0, 0.0]) }
    }
}

impl Neg for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn neg(self) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from(-1.0) }
    }
}

impl Automorphism for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn automorphism(self) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 }
    }
}

impl Reversal for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn reversal(self) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Conjugation for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn conjugation(self) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Dual for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn dual(self) -> SplitComplexNumber {
        SplitComplexNumber { g0: swizzle!(self.g0, 1, 0) }
    }
}

impl Into<Scalar> for SplitComplexNumber {
    fn into(self) -> Scalar {
        Scalar { g0: self.g0[0] }
    }
}

impl Add<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn add(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 + Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl AddAssign<Scalar> for SplitComplexNumber {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn sub(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 - Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl SubAssign<Scalar> for SplitComplexNumber {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn geometric_product(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl RegressiveProduct<Scalar> for SplitComplexNumber {
    type Output = Scalar;

    fn regressive_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[1] * other.g0 }
    }
}

impl OuterProduct<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn outer_product(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl InnerProduct<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn inner_product(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl LeftContraction<Scalar> for SplitComplexNumber {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl RightContraction<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn right_contraction(self, other: Scalar) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl ScalarProduct<Scalar> for SplitComplexNumber {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl Add<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn add(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<SplitComplexNumber> for SplitComplexNumber {
    fn add_assign(&mut self, other: SplitComplexNumber) {
        *self = (*self).add(other);
    }
}

impl Sub<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn sub(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<SplitComplexNumber> for SplitComplexNumber {
    fn sub_assign(&mut self, other: SplitComplexNumber) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn geometric_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) }
    }
}

impl RegressiveProduct<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn regressive_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[1]) * other.g0 + Simd32x2::from(self.g0[0]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl OuterProduct<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn outer_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + self.g0 * Simd32x2::from(other.g0[0]) * Simd32x2::from([0.0, 1.0]) }
    }
}

impl InnerProduct<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn inner_product(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) }
    }
}

impl LeftContraction<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn left_contraction(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + swizzle!(self.g0, 1, 0) * swizzle!(other.g0, 1, 0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl RightContraction<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn right_contraction(self, other: SplitComplexNumber) -> SplitComplexNumber {
        SplitComplexNumber { g0: Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) + Simd32x2::from(self.g0[0]) * Simd32x2::from(other.g0[0]) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl ScalarProduct<SplitComplexNumber> for SplitComplexNumber {
    type Output = Scalar;

    fn scalar_product(self, other: SplitComplexNumber) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0[0] + self.g0[1] * other.g0[1] }
    }
}

impl SquaredMagnitude for SplitComplexNumber {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for SplitComplexNumber {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn signum(self) -> SplitComplexNumber {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn inverse(self) -> SplitComplexNumber {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Mul<Scalar> for Scalar {
    type Output = Scalar;

    fn mul(self, other: Scalar) -> Scalar {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for Scalar {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Powi for Scalar {
    type Output = Scalar;

    fn powi(self, exponent: isize) -> Scalar {
        if exponent == 0 {
            return Scalar::one();
        }
        let mut x: Scalar = if exponent < 0 { self.inverse() } else { self };
        let mut y: Scalar = Scalar::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<Scalar> for Scalar {
    type Output = Scalar;

    fn div(self, other: Scalar) -> Scalar {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for Scalar {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for Scalar {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn mul(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other)
    }
}

impl Div<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn div(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl Transformation<SplitComplexNumber> for Scalar {
    type Output = SplitComplexNumber;

    fn transformation(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn mul(self, other: Scalar) -> SplitComplexNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for SplitComplexNumber {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Div<Scalar> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn div(self, other: Scalar) -> SplitComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for SplitComplexNumber {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for SplitComplexNumber {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal()).into()
    }
}

impl Mul<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn mul(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<SplitComplexNumber> for SplitComplexNumber {
    fn mul_assign(&mut self, other: SplitComplexNumber) {
        *self = (*self).mul(other);
    }
}

impl Powi for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn powi(self, exponent: isize) -> SplitComplexNumber {
        if exponent == 0 {
            return SplitComplexNumber::one();
        }
        let mut x: SplitComplexNumber = if exponent < 0 { self.inverse() } else { self };
        let mut y: SplitComplexNumber = SplitComplexNumber::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn div(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<SplitComplexNumber> for SplitComplexNumber {
    fn div_assign(&mut self, other: SplitComplexNumber) {
        *self = (*self).div(other);
    }
}

impl Transformation<SplitComplexNumber> for SplitComplexNumber {
    type Output = SplitComplexNumber;

    fn transformation(self, other: SplitComplexNumber) -> SplitComplexNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

