#![allow(clippy::assign_op_pattern)]
use crate::{simd::*, *};
use std::ops::{Add, AddAssign, Div, DivAssign, Mul, MulAssign, Neg, Sub, SubAssign};

#[derive(Clone, Copy, Debug)]
pub struct Scalar {
    /// 1
    pub g0: f32,
}

#[derive(Clone, Copy, Debug)]
pub struct ComplexNumber {
    /// 1, e01
    pub g0: Simd32x2,
}

impl Zero for Scalar {
    fn zero() -> Self {
        Scalar { g0: 0.0 }
    }
}

impl One for Scalar {
    fn one() -> Self {
        Scalar { g0: 1.0 }
    }
}

impl Neg for Scalar {
    type Output = Scalar;

    fn neg(self) -> Scalar {
        Scalar { g0: self.g0 * -1.0 }
    }
}

impl Automorphism for Scalar {
    type Output = Scalar;

    fn automorphism(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Reversal for Scalar {
    type Output = Scalar;

    fn reversal(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Conjugation for Scalar {
    type Output = Scalar;

    fn conjugation(self) -> Scalar {
        Scalar { g0: self.g0 }
    }
}

impl Add<Scalar> for Scalar {
    type Output = Scalar;

    fn add(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<Scalar> for Scalar {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for Scalar {
    type Output = Scalar;

    fn sub(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<Scalar> for Scalar {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn geometric_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl OuterProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn outer_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl InnerProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn inner_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl LeftContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl RightContraction<Scalar> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl ScalarProduct<Scalar> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0 * other.g0 }
    }
}

impl Add<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn add(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) + other.g0 }
    }
}

impl Sub<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn sub(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * Simd32x2::from([1.0, 0.0]) - other.g0 }
    }
}

impl GeometricProduct<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn geometric_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RegressiveProduct<ComplexNumber> for Scalar {
    type Output = Scalar;

    fn regressive_product(self, other: ComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[1] }
    }
}

impl OuterProduct<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn outer_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl InnerProduct<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn inner_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl LeftContraction<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn left_contraction(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0) * other.g0 }
    }
}

impl RightContraction<ComplexNumber> for Scalar {
    type Output = Scalar;

    fn right_contraction(self, other: ComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl ScalarProduct<ComplexNumber> for Scalar {
    type Output = Scalar;

    fn scalar_product(self, other: ComplexNumber) -> Scalar {
        Scalar { g0: self.g0 * other.g0[0] }
    }
}

impl SquaredMagnitude for Scalar {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for Scalar {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for Scalar {
    type Output = Scalar;

    fn signum(self) -> Scalar {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for Scalar {
    type Output = Scalar;

    fn inverse(self) -> Scalar {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Zero for ComplexNumber {
    fn zero() -> Self {
        ComplexNumber { g0: Simd32x2::from(0.0) }
    }
}

impl One for ComplexNumber {
    fn one() -> Self {
        ComplexNumber { g0: Simd32x2::from([1.0, 0.0]) }
    }
}

impl Neg for ComplexNumber {
    type Output = ComplexNumber;

    fn neg(self) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from(-1.0) }
    }
}

impl Automorphism for ComplexNumber {
    type Output = ComplexNumber;

    fn automorphism(self) -> ComplexNumber {
        ComplexNumber { g0: self.g0 }
    }
}

impl Reversal for ComplexNumber {
    type Output = ComplexNumber;

    fn reversal(self) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Conjugation for ComplexNumber {
    type Output = ComplexNumber;

    fn conjugation(self) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from([1.0, -1.0]) }
    }
}

impl Dual for ComplexNumber {
    type Output = ComplexNumber;

    fn dual(self) -> ComplexNumber {
        ComplexNumber { g0: swizzle!(self.g0, 1, 0) }
    }
}

impl Into<Scalar> for ComplexNumber {
    fn into(self) -> Scalar {
        Scalar { g0: self.g0[0] }
    }
}

impl Add<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn add(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 + Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl AddAssign<Scalar> for ComplexNumber {
    fn add_assign(&mut self, other: Scalar) {
        *self = (*self).add(other);
    }
}

impl Sub<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn sub(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 - Simd32x2::from(other.g0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl SubAssign<Scalar> for ComplexNumber {
    fn sub_assign(&mut self, other: Scalar) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn geometric_product(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl RegressiveProduct<Scalar> for ComplexNumber {
    type Output = Scalar;

    fn regressive_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[1] * other.g0 }
    }
}

impl OuterProduct<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn outer_product(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl InnerProduct<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn inner_product(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl LeftContraction<Scalar> for ComplexNumber {
    type Output = Scalar;

    fn left_contraction(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl RightContraction<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn right_contraction(self, other: Scalar) -> ComplexNumber {
        ComplexNumber { g0: self.g0 * Simd32x2::from(other.g0) }
    }
}

impl ScalarProduct<Scalar> for ComplexNumber {
    type Output = Scalar;

    fn scalar_product(self, other: Scalar) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0 }
    }
}

impl Add<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn add(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: self.g0 + other.g0 }
    }
}

impl AddAssign<ComplexNumber> for ComplexNumber {
    fn add_assign(&mut self, other: ComplexNumber) {
        *self = (*self).add(other);
    }
}

impl Sub<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn sub(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: self.g0 - other.g0 }
    }
}

impl SubAssign<ComplexNumber> for ComplexNumber {
    fn sub_assign(&mut self, other: ComplexNumber) {
        *self = (*self).sub(other);
    }
}

impl GeometricProduct<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn geometric_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([-1.0, 1.0]) }
    }
}

impl RegressiveProduct<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn regressive_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[1]) * other.g0 + Simd32x2::from(self.g0[0]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl OuterProduct<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn outer_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + self.g0 * Simd32x2::from(other.g0[0]) * Simd32x2::from([0.0, 1.0]) }
    }
}

impl InnerProduct<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn inner_product(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([-1.0, 1.0]) }
    }
}

impl LeftContraction<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn left_contraction(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[0]) * other.g0 + swizzle!(self.g0, 1, 0) * swizzle!(other.g0, 1, 0) * Simd32x2::from([-1.0, 0.0]) }
    }
}

impl RightContraction<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn right_contraction(self, other: ComplexNumber) -> ComplexNumber {
        ComplexNumber { g0: Simd32x2::from(self.g0[1]) * swizzle!(other.g0, 1, 0) * Simd32x2::from([-1.0, 1.0]) + Simd32x2::from(self.g0[0]) * Simd32x2::from(other.g0[0]) * Simd32x2::from([1.0, 0.0]) }
    }
}

impl ScalarProduct<ComplexNumber> for ComplexNumber {
    type Output = Scalar;

    fn scalar_product(self, other: ComplexNumber) -> Scalar {
        Scalar { g0: self.g0[0] * other.g0[0] - self.g0[1] * other.g0[1] }
    }
}

impl SquaredMagnitude for ComplexNumber {
    type Output = Scalar;

    fn squared_magnitude(self) -> Scalar {
        self.scalar_product(self.reversal())
    }
}

impl Magnitude for ComplexNumber {
    type Output = Scalar;

    fn magnitude(self) -> Scalar {
        Scalar { g0: self.squared_magnitude().g0.sqrt() }
    }
}

impl Signum for ComplexNumber {
    type Output = ComplexNumber;

    fn signum(self) -> ComplexNumber {
        self.geometric_product(Scalar { g0: 1.0 / self.magnitude().g0 })
    }
}

impl Inverse for ComplexNumber {
    type Output = ComplexNumber;

    fn inverse(self) -> ComplexNumber {
        self.reversal().geometric_product(Scalar { g0: 1.0 / self.squared_magnitude().g0 })
    }
}

impl Mul<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn mul(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<ComplexNumber> for ComplexNumber {
    fn mul_assign(&mut self, other: ComplexNumber) {
        *self = (*self).mul(other);
    }
}

impl Powi for ComplexNumber {
    type Output = ComplexNumber;

    fn powi(self, exponent: isize) -> ComplexNumber {
        if exponent == 0 {
            return ComplexNumber::one();
        }
        let mut x: ComplexNumber = if exponent < 0 { self.inverse() } else { self };
        let mut y: ComplexNumber = ComplexNumber::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn div(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<ComplexNumber> for ComplexNumber {
    fn div_assign(&mut self, other: ComplexNumber) {
        *self = (*self).div(other);
    }
}

impl Transformation<ComplexNumber> for ComplexNumber {
    type Output = ComplexNumber;

    fn transformation(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn mul(self, other: Scalar) -> ComplexNumber {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for ComplexNumber {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Div<Scalar> for ComplexNumber {
    type Output = ComplexNumber;

    fn div(self, other: Scalar) -> ComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for ComplexNumber {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for ComplexNumber {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal()).into()
    }
}

impl Mul<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn mul(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other)
    }
}

impl Div<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn div(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other.inverse())
    }
}

impl Transformation<ComplexNumber> for Scalar {
    type Output = ComplexNumber;

    fn transformation(self, other: ComplexNumber) -> ComplexNumber {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

impl Mul<Scalar> for Scalar {
    type Output = Scalar;

    fn mul(self, other: Scalar) -> Scalar {
        self.geometric_product(other)
    }
}

impl MulAssign<Scalar> for Scalar {
    fn mul_assign(&mut self, other: Scalar) {
        *self = (*self).mul(other);
    }
}

impl Powi for Scalar {
    type Output = Scalar;

    fn powi(self, exponent: isize) -> Scalar {
        if exponent == 0 {
            return Scalar::one();
        }
        let mut x: Scalar = if exponent < 0 { self.inverse() } else { self };
        let mut y: Scalar = Scalar::one();
        let mut n: isize = exponent.abs();
        while 1 < n {
            if n & 1 == 1 {
                y = x.geometric_product(y);
            }
            x = x.geometric_product(x);
            n = n >> 1;
        }
        x.geometric_product(y)
    }
}

impl Div<Scalar> for Scalar {
    type Output = Scalar;

    fn div(self, other: Scalar) -> Scalar {
        self.geometric_product(other.inverse())
    }
}

impl DivAssign<Scalar> for Scalar {
    fn div_assign(&mut self, other: Scalar) {
        *self = (*self).div(other);
    }
}

impl Transformation<Scalar> for Scalar {
    type Output = Scalar;

    fn transformation(self, other: Scalar) -> Scalar {
        self.geometric_product(other).geometric_product(self.reversal())
    }
}

