use pest::Parser;
use pest_derive::Parser;

use crate::Error;
use crate::Grammar;
use crate::Node;
use crate::Result;

#[derive(Parser)]
#[grammar = "genx.pest"]
struct GenxParser;

// type PestError = pest::error::Error<Rule>;

fn parse_node(pair: pest::iterators::Pair<Rule>) -> Result<Node> {
    match pair.as_rule() {
        Rule::non_terminal => Ok(Node::NonTerminal(
            pair.into_inner().next().unwrap().as_str().to_string(),
        )),
        Rule::var_ref => Ok(Node::VarRef(
            pair.into_inner().next().unwrap().as_str().to_string(),
        )),
        Rule::text => Ok(Node::Text(pair.as_str().to_string())),
        Rule::optional => {
            // Optional nodes contain an implicit sequence.
            Ok(Node::Optional(Box::new(parse_implicit_sequence(pair)?)))
        }
        Rule::choice => {
            // Each choice_arm child is an implicit sequence.
            let arms = pair
                .into_inner()
                .map(parse_implicit_sequence)
                .collect::<Result<Vec<_>>>()?;
            Ok(Node::Choice(arms))
        }
        _ => Err(Error::InternalParserError(format!(
            "Unrecognized rule while parsing node: {:?}",
            pair.as_rule()
        ))),
    }
}

fn parse_implicit_sequence(pair: pest::iterators::Pair<Rule>) -> Result<Node> {
    Ok(Node::Sequence(
        pair.into_inner()
            .map(parse_node)
            .collect::<Result<Vec<_>>>()?,
    ))
}

pub(crate) fn parse_grammar(s: &str) -> Result<Grammar> {
    let mut grammar = Grammar::new();
    match GenxParser::parse(Rule::grammar, s) {
        Ok(pairs) => {
            for pair in pairs {
                if pair.as_rule() == Rule::EOI {
                    break;
                }
                let mut inner = pair.into_inner();
                let lhs = inner.next().unwrap().as_str();
                let rhs_nodes: Result<Vec<Node>> =
                    inner.next().unwrap().into_inner().map(parse_node).collect();
                let rhs = Node::Sequence(rhs_nodes?);
                grammar.add_rule(lhs, rhs);
            }
        }
        Err(e) => return Err(Error::ParseError(e.to_string())),
    };
    Ok(grammar)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parse() {
        let s = r#"
          top = hi <name>?:[, my dear #gender#,] in <location>
          name = #name#
          location = [city of #city#|#county# county]
        "#;
        let grammar = parse_grammar(s).unwrap();
        assert_eq!(
            grammar.get_rule("top").unwrap(),
            &Node::Sequence(vec![
                Node::Text("hi ".to_string()),
                Node::NonTerminal("name".to_string()),
                Node::Optional(Box::new(Node::Sequence(vec![
                    Node::Text(", my dear ".to_string()),
                    Node::VarRef("gender".to_string()),
                    Node::Text(",".to_string())
                ]))),
                Node::Text(" in ".to_string()),
                Node::NonTerminal("location".to_string()),
            ])
        );
        assert_eq!(
            grammar.get_rule("name").unwrap(),
            &Node::Sequence(vec![Node::VarRef("name".to_string())])
        );
        assert_eq!(
            grammar.get_rule("location").unwrap(),
            &Node::Sequence(vec![Node::Choice(vec![
                Node::Sequence(vec![
                    Node::Text("city of ".to_string()),
                    Node::VarRef("city".to_string())
                ]),
                Node::Sequence(vec![
                    Node::VarRef("county".to_string()),
                    Node::Text(" county".to_string()),
                ]),
            ])])
        );
    }
}
