use core::ops::{Deref, DerefMut};
use std::ops::{Index, IndexMut};

pub struct WeakPointer<T>(*mut T);

impl<T> WeakPointer<T> {
    /// # Safety
    ///
    /// Up to the user to enusure that the data pointing to exists and is safe
    pub unsafe fn new(ptr: *const T) -> Self {
        Self(ptr as _)
    }

    /// # Safety
    ///
    /// Up to the user to set this to a valid location
    pub unsafe fn null() -> Self {
        Self(0 as _)
    }

    pub fn is_null(&self) -> bool {
        self.0.is_null()
    }
}

impl<T> Index<usize> for WeakPointer<T> {
    type Output = T;

    fn index(&self, index: usize) -> &Self::Output {
        unsafe { &*self.0.add(index) }
    }
}

impl<T> IndexMut<usize> for WeakPointer<T> {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        unsafe { &mut *self.0.add(index) }
    }
}

impl<T> Deref for WeakPointer<T> {
    type Target = T;

    fn deref(&self) -> &T {
        unsafe { &(*self.0) }
    }
}

impl<T> DerefMut for WeakPointer<T> {
    fn deref_mut(&mut self) -> &mut T {
        unsafe { &mut (*self.0) }
    }
}

unsafe impl<T> Send for WeakPointer<T> {}
unsafe impl<T> Sync for WeakPointer<T> {}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_index() {
        let vals = [1, 2, 3, 4, 5];
        let ptr = unsafe { WeakPointer::new(&vals[0]) };

        assert_eq!(ptr[0], 1);
        assert_eq!(ptr[1], 2);
        assert_eq!(ptr[2], 3);
        assert_eq!(ptr[3], 4);
        assert_eq!(ptr[4], 5);
    }
}
