use super::*;
use core::mem;
use core::ops::{Deref, DerefMut};
use std::alloc::{alloc_zeroed, dealloc, Layout};

#[derive(Clone)]
pub struct BoxedByteSlice(mem::ManuallyDrop<Box<[u8]>>);

impl BoxedByteSlice {
    #[inline]
    pub fn new_zeroed(len: usize) -> Self {
        let layout = Layout::from_size_align((len + 7) & !7, 8).unwrap();
        unsafe {
            BoxedByteSlice(mem::ManuallyDrop::new(Box::from_raw(
                core::slice::from_raw_parts_mut(alloc_zeroed(layout), len),
            )))
        }
    }

    #[inline]
    pub fn as_byte_slice(&self) -> ByteSlice {
        ByteSlice::new(&self[..])
    }

    #[inline]
    pub fn as_byte_mut_slice(&mut self) -> ByteMutSlice {
        ByteMutSlice::new(&mut self[..])
    }
}

impl Deref for BoxedByteSlice {
    type Target = [u8];
    #[inline]
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl DerefMut for BoxedByteSlice {
    #[inline]
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl Drop for BoxedByteSlice {
    #[inline]
    fn drop(&mut self) {
        let layout = Layout::from_size_align((self.0.len() + 7) & !7, 8).unwrap();
        unsafe {
            dealloc(
                Box::into_raw(mem::ManuallyDrop::take(&mut self.0)) as *mut u8,
                layout,
            )
        }
    }
}

impl_byteorder!(BoxedByteSlice);
