use std::{
    cell::UnsafeCell,
    ops::{Deref, DerefMut},
    rc::Rc,
};

/// Opt-out of runtime borrow checking of RefCell by using UnsafeCell
///
/// # Safety
///
/// Up to the user to make sure the usage of the shared object is safe
#[derive(Debug, Clone, Default)]
pub struct Shared<T>(Rc<UnsafeCell<T>>);

impl<T> Shared<T> {
    pub fn new(t: T) -> Shared<T> {
        Shared(Rc::new(UnsafeCell::new(t)))
    }

    /// # Safety
    ///
    /// Up to the user to make sure the usage of the shared object is safe
    #[allow(clippy::mut_from_ref)]
    pub unsafe fn inner_unsafe(&self) -> &mut T {
        &mut (*self.0.get())
    }

    pub fn clone_inner(&self) -> T
    where
        T: Clone,
    {
        self.deref().clone()
    }
}

impl<T> Deref for Shared<T> {
    type Target = T;

    fn deref(&self) -> &Self::Target {
        unsafe { &(*self.0.get()) }
    }
}

impl<T> DerefMut for Shared<T> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        unsafe { &mut (*self.0.get()) }
    }
}
