use core::ops::{Deref, DerefMut};

pub struct WeakPointer<T>(*mut T);

impl<T> WeakPointer<T> {
    /// # Safety
    ///
    /// Up to the user to enusure that the data pointing to exists and is safe
    pub unsafe fn new(ptr: *const T) -> Self {
        Self(ptr as _)
    }

    /// # Safety
    ///
    /// Up to the user to set this to a valid location
    pub unsafe fn null() -> Self {
        Self(0 as _)
    }

    pub fn is_null(&self) -> bool {
        self.0.is_null()
    }
}

impl<T> Deref for WeakPointer<T> {
    type Target = T;

    fn deref(&self) -> &T {
        unsafe { &(*self.0) }
    }
}

impl<T> DerefMut for WeakPointer<T> {
    fn deref_mut(&mut self) -> &mut T {
        unsafe { &mut (*self.0) }
    }
}

unsafe impl<T> Send for WeakPointer<T> {}
unsafe impl<T> Sync for WeakPointer<T> {}
