#pragma once
/**
 * @brief Gemachain logging utilities
 */

#include <gema/types.h>
#include <gema/string.h>
#include <gema/entrypoint.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Prints a string to stdout
 */
void gema_log_(const char *, uint64_t);
#define gema_log(message) gema_log_(message, gema_strlen(message))

/**
 * Prints a 64 bit values represented in hexadecimal to stdout
 */
void gema_log_64_(uint64_t, uint64_t, uint64_t, uint64_t, uint64_t);
#define gema_log_64 gema_log_64_

/**
 * Prints the current compute unit consumption to stdout
 */
void gema_log_compute_units_();
#define gema_log_compute_units() gema_log_compute_units_()

/**
 * Prints the hexadecimal representation of an array
 *
 * @param array The array to print
 */
static void gema_log_array(const uint8_t *array, int len) {
  for (int j = 0; j < len; j++) {
    gema_log_64(0, 0, 0, j, array[j]);
  }
}

/**
 * Print the base64 representation of some arrays.
 */
void gema_log_data(GemaBytes *fields, uint64_t fields_len);

/**
 * Prints the program's input parameters
 *
 * @param params Pointer to a GemaParameters structure
 */
static void gema_log_params(const GemaParameters *params) {
  gema_log("- Program identifier:");
  gema_log_pubkey(params->program_id);

  gema_log("- Number of KeyedAccounts");
  gema_log_64(0, 0, 0, 0, params->ka_num);
  for (int i = 0; i < params->ka_num; i++) {
    gema_log("  - Is signer");
    gema_log_64(0, 0, 0, 0, params->ka[i].is_signer);
    gema_log("  - Is writable");
    gema_log_64(0, 0, 0, 0, params->ka[i].is_writable);
    gema_log("  - Key");
    gema_log_pubkey(params->ka[i].key);
    gema_log("  - Carats");
    gema_log_64(0, 0, 0, 0, *params->ka[i].carats);
    gema_log("  - data");
    gema_log_array(params->ka[i].data, params->ka[i].data_len);
    gema_log("  - Owner");
    gema_log_pubkey(params->ka[i].owner);
    gema_log("  - Executable");
    gema_log_64(0, 0, 0, 0, params->ka[i].executable);
    gema_log("  - Rent Epoch");
    gema_log_64(0, 0, 0, 0, params->ka[i].rent_epoch);
  }
  gema_log("- Instruction data\0");
  gema_log_array(params->data, params->data_len);
}

#ifdef GEMA_TEST
/**
 * Stub functions when building tests
 */
#include <stdio.h>

void gema_log_(const char *s, uint64_t len) {
  printf("Program log: %s\n", s);
}
void gema_log_64(uint64_t arg1, uint64_t arg2, uint64_t arg3, uint64_t arg4, uint64_t arg5) {
  printf("Program log: %llu, %llu, %llu, %llu, %llu\n", arg1, arg2, arg3, arg4, arg5);
}

void gema_log_compute_units_() {
  printf("Program consumption: __ units remaining\n");
}
#endif

#ifdef __cplusplus
}
#endif

/**@}*/
