#pragma once
/**
 * @brief Gemachain Cross-Program Invocation
 */

#include <gema/types.h>
#include <gema/pubkey.h>
#include <gema/entrypoint.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Account Meta
 */
typedef struct {
  GemaPubkey *pubkey; /** An account's public key */
  bool is_writable; /** True if the `pubkey` can be loaded as a read-write account */
  bool is_signer; /** True if an Instruction requires a Transaction signature matching `pubkey` */
} GemaAccountMeta;

/**
 * Instruction
 */
typedef struct {
  GemaPubkey *program_id; /** Pubkey of the instruction processor that executes this instruction */
  GemaAccountMeta *accounts; /** Metadata for what accounts should be passed to the instruction processor */
  uint64_t account_len; /** Number of GemaAccountMetas */
  uint8_t *data; /** Opaque data passed to the instruction processor */
  uint64_t data_len; /** Length of the data in bytes */
} GemaInstruction;

/**
 * Internal cross-program invocation function
 */
uint64_t gema_invoke_signed_c(
  const GemaInstruction *instruction,
  const GemaAccountInfo *account_infos,
  int account_infos_len,
  const GemaSignerSeeds *signers_seeds,
  int signers_seeds_len
);

/**
 * Invoke another program and sign for some of the keys
 *
 * @param instruction Instruction to process
 * @param account_infos Accounts used by instruction
 * @param account_infos_len Length of account_infos array
 * @param seeds Seed bytes used to sign program accounts
 * @param seeds_len Length of the seeds array
 */
static uint64_t gema_invoke_signed(
    const GemaInstruction *instruction,
    const GemaAccountInfo *account_infos,
    int account_infos_len,
    const GemaSignerSeeds *signers_seeds,
    int signers_seeds_len
) {
  return gema_invoke_signed_c(
    instruction,
    account_infos,
    account_infos_len,
    signers_seeds,
    signers_seeds_len
  );
}
/**
 * Invoke another program
 *
 * @param instruction Instruction to process
 * @param account_infos Accounts used by instruction
 * @param account_infos_len Length of account_infos array
*/
static uint64_t gema_invoke(
    const GemaInstruction *instruction,
    const GemaAccountInfo *account_infos,
    int account_infos_len
) {
  const GemaSignerSeeds signers_seeds[] = {{}};
  return gema_invoke_signed(
    instruction,
    account_infos,
    account_infos_len,
    signers_seeds,
    0
  );
}

#ifdef __cplusplus
}
#endif

/**@}*/
