//! # `/simple`
//!
//! - [/simple/price](Price)
//! - [/simple/token_price/{id}](TokenPrice)
//! - [/simple/supported_vs_currencies](SupportedCurrency)
//!
use crate::Route;

#[derive(Debug)]
pub struct Price {
    endpoint: String,
    pub ids: String,
    pub vs_currencies: String,
    pub include_market_cap: bool,
    pub include_24hr_vol: bool,
    pub include_24hr_change: bool,
    pub include_last_updated_at: bool,
}
#[derive(Debug)]
pub struct TokenPrice {
    endpoint: String,
    pub id: String,
    pub contract_addresses: String,
    pub vs_currencies: String,
    pub include_market_cap: bool,
    pub include_24hr_vol: bool,
    pub include_24hr_change: bool,
    pub include_last_updated_at: bool,
}
#[derive(Debug)]
pub struct SupportedCurrency {
    endpoint: String,
}

impl Price {
    pub fn required(ids: String, vs_currencies: String) -> Price {
        Price {
            ids,
            vs_currencies,
            ..Default::default()
        }
    }
}
impl TokenPrice {
    pub fn required(id: String, contract_addresses: String, vs_currencies: String) -> TokenPrice {
        TokenPrice {
            id,
            contract_addresses,
            vs_currencies,
            ..Default::default()
        }
    }
}
impl SupportedCurrency {
    pub fn required() -> SupportedCurrency {
        SupportedCurrency::default()
    }
}

// Defaults
impl Default for Price {
    fn default() -> Price {
        Price {
            endpoint: String::from("/simple/price"),
            ids: String::from(""),
            vs_currencies: String::from(""),
            include_market_cap: false,
            include_24hr_vol: false,
            include_24hr_change: false,
            include_last_updated_at: false,
        }
    }
}
impl Default for TokenPrice {
    fn default() -> TokenPrice {
        TokenPrice {
            endpoint: String::from("/simple/token_price/ID"),
            id: "".to_string(),
            contract_addresses: "".to_string(),
            vs_currencies: "".to_string(),
            include_24hr_change: false,
            include_market_cap: false,
            include_24hr_vol: false,
            include_last_updated_at: false,
        }
    }
}
impl Default for SupportedCurrency {
    fn default() -> SupportedCurrency {
        SupportedCurrency {
            endpoint: String::from("/simple/supported_vs_currencies"),
        }
    }
}

/// Route Trait implemntation
impl Route for Price {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        let default: Price = Price::default();
        let ids = self.format_query("ids".to_string(), &(self.ids), &(default.ids));
        let vs_currencies = self.format_query(
            "vs_currencies".to_string(),
            &(self.vs_currencies),
            &(default.vs_currencies),
        );
        let include_market_cap = self.format_query(
            "include_market_cap".to_string(),
            self.include_market_cap,
            default.include_market_cap,
        );
        let include_24hr_vol = self.format_query(
            "include_24hr_vol".to_string(),
            self.include_24hr_vol,
            default.include_24hr_vol,
        );
        let include_24hr_change = self.format_query(
            "include_24hr_change".to_string(),
            self.include_24hr_change,
            default.include_24hr_change,
        );
        let include_last_updated_at = self.format_query(
            "include_last_updated_at".to_string(),
            self.include_last_updated_at,
            default.include_last_updated_at,
        );
        let optional = vec![
            ids,
            vs_currencies,
            include_market_cap,
            include_24hr_vol,
            include_24hr_change,
            include_last_updated_at,
        ];
        self.collect_query_params(optional)
    }
}
impl Route for TokenPrice {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: TokenPrice = Default::default();
        let contract_addresses = self.format_query(
            "contract_addresses".to_string(),
            &(self.contract_addresses),
            &(default.contract_addresses),
        );
        let vs_currencies = self.format_query(
            "vs_currencies".to_string(),
            &(self.vs_currencies),
            &(default.vs_currencies),
        );
        let include_market_cap = self.format_query(
            "include_market_cap".to_string(),
            self.include_market_cap,
            default.include_market_cap,
        );
        let include_24hr_vol = self.format_query(
            "include_24hr_vol".to_string(),
            self.include_24hr_vol,
            default.include_24hr_vol,
        );
        let include_24hr_change = self.format_query(
            "include_24hr_change".to_string(),
            self.include_24hr_change,
            default.include_24hr_change,
        );
        let include_last_updated_at = self.format_query(
            "include_last_updated_at".to_string(),
            self.include_last_updated_at,
            default.include_last_updated_at,
        );
        let optional = vec![
            contract_addresses,
            vs_currencies,
            include_market_cap,
            include_24hr_vol,
            include_24hr_change,
            include_last_updated_at,
        ];
        self.collect_query_params(optional)
    }
}
impl Route for SupportedCurrency {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        String::from("")
    }
}
