//! # gecko 
//! 
//! The `gecko` crate provides convenient access coingeckos api. 
//! The focus of this crate is on access/retreival of data from coingecko api endpoints and not on convenience or manipulating reponses/data.
//! Although, if raw text reponses are not desired, the crate has funcitonality to provide coingeckos api response in [json objects](CoinGeckoAPI::get_json). 
//! 
//! **Note:** Coingeckos api has a limit of 100 requests per minute.
//! 
//! ## Examples
//! #### Default
//! ```
//! use gecko;
//! #[tokio::main]
//! async fn main() -> Result<(), Box<dyn std::error::Error>> {
//!    let gecko_client = gecko::CoinGeckoAPI::default();
//!    let coins_list = gecko::coins::List::required();
//!    let response = gecko_client(&coins_list).await.unwrap();
//!    println!("{:?}", response):
//! }
//! ```
//!
//! #### Custom
//! ```
//! use gecko;
//! #[tokio::main]
//! async fn main() -> Result<(), Box<dyn std::error::Error>> {
//!    let gecko_client = gecko::CoinGeckoAPI::default();
//!    let mut coins_list = gecko::coins::Info::required("premia".to_string());
//!    coins_list.tickers = false;
//!    let response = gecko_client(&coins_list).await.unwrap();
//!    println!("{:?}", response["id"]):
//! }
//! ```
//! Or if you do not want the request object to be a mutable variable...  
//! ```
//! use gecko;
//! #[tokio::main]
//! async fn main() -> Result<(), Box<dyn std::error::Error>> {
//!    let gecko_client = gecko::CoinGeckoAPI::default();
//!    let coins_list = {
//!       let mut _tmp = gecko::coins::Info::required("premia".to_string());
//!       _tmp.tickers = false;
//!       _tmp
//!    };
//!    let response = gecko_client(&coins_list).await.unwrap();
//!    println!("{:?}", response["id"]):
//! }
//! ``` 
//! 
//! For more information about CoinGecko API can be found [here](https://coingecko.com/en/api) 

use reqwest;
use serde_json;
use serde_json::Value;
use std::fmt;

pub struct CoinGeckoAPI {
    version: String,
    base_url: String,
}

impl Default for CoinGeckoAPI {
    fn default() -> CoinGeckoAPI {
        CoinGeckoAPI {
            version: String::from("v3"),
            base_url: String::from("https://api.coingecko.com"),
        }
    }
}

impl CoinGeckoAPI {
    /// Returns the base uri for coingecko api
    pub fn build_base_url(&self) -> String {
        format!("{}/api/{}", self.base_url, self.version)
    }

    /// Returns a json Value built from the response_text arguement
    pub fn to_json(response_text: &String) -> Value {
        let v: Value = serde_json::from_str(response_text).unwrap();
        v
    }

    /// Returns the get method response text as a json Value  
    pub async fn get_json<T: Route>(
        &self,
        resource: &T,
    ) -> Result<Value, Box<dyn std::error::Error>> {
        let resp = self.get(resource).await?;
        if resp == "" || resp == "{}" {
            return Ok(Value::Null);
        }
        let json = CoinGeckoAPI::to_json(&resp);
        Ok(json)
    }

    /// Fetches the resource for coin gecko api and returns the repsonse's text 
    pub async fn get<T: Route>(&self, resource: &T) -> Result<String, Box<dyn std::error::Error>> {
        let base_url = self.build_base_url();
        let api_endpoint = resource.route();
        let url = format!("{}{}", base_url, api_endpoint);
        let client = reqwest::Client::new();
        let resp = client
            .get(url)
            .header("accept", "application/json")
            .send()
            .await?;
        if !resp.status().is_success() {
            panic!("Error response status code {:?}", resp.status())
        }
        let text = resp.text().await?;
        Ok(text)
    }
}

pub trait Route {
//!  The Route trait can be implented to give a struct the ability to be used by
//!  [CoinGeckoAPI] 

    /// Return the query string containing required and non-default values
    fn query_string(&self) -> String;
    /// Return the api endpoint with required values
    fn api_endpoint(&self) -> String;

    /// Return the formatted route endpoint 
    fn route(&self) -> String {
        format!("{}{}", self.api_endpoint(), self.query_string())
    }

    /// Return formated query string parameter used in url
    fn format_query<T: fmt::Display + Copy + std::cmp::PartialEq>(
        &self,
        name: String,
        value: T,
        default: T,
    ) -> String {
        if value != default {
            return format!("{}={}", name, value);
        }
        "".to_string()
    }

    /// Return a string of joined formatted url query parameters 
    fn collect_query_params(&self, query_params: Vec<String>) -> String {
        let mut query = String::from("?");
        let collected: String = query_params
            .into_iter()
            .filter(|x| **x != String::from(""))
            .collect::<Vec<String>>()
            .join("&");
        query.push_str(&collected);
        query
    }
}

pub mod asset_platforms;
pub mod categories;
pub mod coins;
pub mod companies;
pub mod contract;
pub mod derivatives;
pub mod events;
pub mod exchange_rates;
pub mod exchanges;
pub mod finance;
pub mod global;
pub mod indexes;
pub mod simple;
pub mod status_updates;
pub mod trending;
