//! # `/exchange`
//!
//! - [/exchanges](Exchanges)
//! - [/exchanges/list](List)
//! - [/exchanges/{id}](Info)
//! - [/exchanges/{id}/tickers](Tickers)
//! - [/exchanges/{id}/status_updates](StatusUpdates)
//! - [/exchanges/{id}/volume_chart](VolumeChart)
use crate::Route;

pub struct Exchanges {
    endpoint: String,
    pub per_page: i32,
    pub page: i32,
}
pub struct List {
    endpoint: String,
}
pub struct Info {
    endpoint: String,
    pub id: String,
}
pub struct Tickers {
    endpoint: String,
    pub id: String,
    pub coin_ids: String,
    pub include_exchange_logo: bool,
    pub page: i32,
    pub depth: String,
    pub order: String,
}
pub struct StatusUpdates {
    endpoint: String,
    pub id: String,
    pub per_page: i32,
    pub page: i32,
}
pub struct VolumeChart {
    endpoint: String,
    pub id: String,
    pub days: String,
}

impl Exchanges {
    pub fn required() -> Exchanges {
        Exchanges::default()
    }
}
impl List {
    pub fn required() -> List {
        List::default()
    }
}
impl Info {
    pub fn required(id: String) -> Info {
        Info {
            id,
            ..Default::default()
        }
    }
}
impl Tickers {
    pub fn required(id: String) -> Tickers {
        Tickers {
            id,
            ..Default::default()
        }
    }
}
impl StatusUpdates {
    pub fn required(id: String) -> StatusUpdates {
        StatusUpdates {
            id,
            ..Default::default()
        }
    }
}
impl VolumeChart {
    pub fn required(id: String, days: String) -> VolumeChart {
        VolumeChart {
            id,
            days,
            ..Default::default()
        }
    }
}

impl Default for Exchanges {
    fn default() -> Exchanges {
        Exchanges {
            endpoint: String::from("/exchanges"),
            per_page: 100,
            page: 1,
        }
    }
}
impl Default for List {
    fn default() -> List {
        List {
            endpoint: String::from("/exchanges/list"),
        }
    }
}
impl Default for Info {
    fn default() -> Info {
        Info {
            endpoint: String::from("/exchanges/ID"),
            id: String::from(""),
        }
    }
}
impl Default for Tickers {
    fn default() -> Tickers {
        Tickers {
            endpoint: String::from("/exchanges/ID/tickers"),
            id: String::from(""),
            coin_ids: String::from(""),
            include_exchange_logo: false,
            page: 1,
            depth: String::from(""),
            order: String::from("trust_score_desc"),
        }
    }
}
impl Default for StatusUpdates {
    fn default() -> StatusUpdates {
        StatusUpdates {
            endpoint: String::from("/exchanges/ID/status_updates"),
            id: String::from(""),
            per_page: 100,
            page: 1,
        }
    }
}
impl Default for VolumeChart {
    fn default() -> VolumeChart {
        VolumeChart {
            endpoint: String::from("/exchanges/ID/volume_chart"),
            id: String::from(""),
            days: String::from(""),
        }
    }
}

impl Route for Exchanges {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        let default: Exchanges = Default::default();
        let per_page = self.format_query("per_page".to_string(), self.per_page, default.per_page);
        let page = self.format_query("page".to_string(), self.page, default.page);
        let optional = vec![per_page, page];
        self.collect_query_params(optional)
    }
}
impl Route for List {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        String::from("")
    }
}
impl Route for Info {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        String::from("")
    }
}

impl Route for Tickers {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: Tickers = Default::default();
        let coins_id = self.format_query(
            "coin_ids".to_string(),
            &(self.coin_ids),
            &(default.coin_ids),
        );
        let include_exchange_logo = self.format_query(
            "include_exchange_logo".to_string(),
            self.include_exchange_logo,
            default.include_exchange_logo,
        );
        let page = self.format_query("page".to_string(), self.page, default.page);
        let depth = self.format_query("depth".to_string(), &(self.depth), &(default.depth));
        let order = self.format_query("order".to_string(), &(self.order), &(default.order));
        let optional = vec![coins_id, include_exchange_logo, page, depth, order];
        self.collect_query_params(optional)
    }
}
impl Route for StatusUpdates {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: StatusUpdates = Default::default();
        let per_page = self.format_query("per_page".to_string(), self.per_page, default.per_page);
        let page = self.format_query("page".to_string(), self.page, default.page);
        let optional = vec![per_page, page];
        self.collect_query_params(optional)
    }
}
impl Route for VolumeChart {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: VolumeChart = Default::default();
        let days = self.format_query("days".to_string(), &(self.days), &(default.days));
        let optional = vec![days];
        self.collect_query_params(optional)
    }
}
