use reqwest;
use serde_json;
use serde_json::Value;
use std::fmt;

pub struct CoinGeckoAPI {
    version: String,
    base_url: String,
}

impl Default for CoinGeckoAPI {
    fn default() -> CoinGeckoAPI {
        CoinGeckoAPI {
            version: String::from("v3"),
            base_url: String::from("https://api.coingecko.com"),
        }
    }
}

impl CoinGeckoAPI {
    pub fn build_base_url(&self) -> String {
        format!("{}/api/{}", self.base_url, self.version)
    }

    pub fn to_json(response_text: &String) -> Value {
        let v: Value = serde_json::from_str(response_text).unwrap();
        v
    }

    pub async fn get_json<T: Route>(
        &self,
        resource: &T,
    ) -> Result<Value, Box<dyn std::error::Error>> {
        let resp = self.get(resource).await?;
        if resp == "" || resp == "{}" {
            return Ok(Value::Null);
        }
        let json = CoinGeckoAPI::to_json(&resp);
        Ok(json)
    }

    pub async fn get<T: Route>(&self, resource: &T) -> Result<String, Box<dyn std::error::Error>> {
        let base_url = self.build_base_url();
        let api_endpoint = resource.route();
        let url = format!("{}{}", base_url, api_endpoint);
        let client = reqwest::Client::new();
        let resp = client
            .get(url)
            .header("accept", "application/json")
            .send()
            .await?;
        if !resp.status().is_success() {
            panic!("Error response status code {:?}", resp.status())
        }
        let text = resp.text().await?;
        Ok(text)
    }
}

pub trait Route {
    fn query_string(&self) -> String;
    fn api_endpoint(&self) -> String;

    fn route(&self) -> String {
        format!("{}{}", self.api_endpoint(), self.query_string())
    }

    /// format query string parameter in url
    fn format_query<T: fmt::Display + Copy + std::cmp::PartialEq>(
        &self,
        name: String,
        value: T,
        default: T,
    ) -> String {
        if value != default {
            return format!("{}={}", name, value);
        }
        "".to_string()
    }

    fn collect_query_params(&self, query_params: Vec<String>) -> String {
        let mut query = String::from("?");
        let collected: String = query_params
            .into_iter()
            .filter(|x| **x != String::from(""))
            .collect::<Vec<String>>()
            .join("&");
        query.push_str(&collected);
        query
    }
}

pub mod asset_platforms;
pub mod categories;
pub mod coins;
pub mod companies;
pub mod contract;
pub mod derivatives;
pub mod events;
pub mod exchange_rates;
pub mod exchanges;
pub mod finance;
pub mod global;
pub mod indexes;
pub mod simple;
pub mod status_updates;
pub mod trending;
