
use crate::Route;

pub struct List {
    endpoint: String,           /// /coins/list
    pub include_platform: bool, // default not specified in docs assume False
}
pub struct Markets {
    endpoint: String,           /// /coins/markets
    pub vs_currency: String,                // required
    pub ids: String, 
    pub category: String,
    pub order: String,          // valid calues: market_cap_desc, gecko_desc, gecko_asc, market_cap_asc, market_cap_desc, volume_asc, volume_desc, id_asc, id_desc
    pub per_page: i32,          // valid values: 1..250
    pub page: i32,
    pub sparkline: bool,
    pub price_change_percentage: String,
        // valid format: '1h' , '14d', '1y', multi values comma separated
}
pub struct Info {
    endpoint: String,           /// /coins/{id}
    pub id: String,
    pub localization: bool,     // true
    pub tickers: bool,          // true
    pub market_data: bool,      // true
    pub community_data: bool,   // true
    pub developer_data: bool,   // true
    pub sparkline: bool,        // false
}
pub struct Tickers {
    endpoint: String,           /// /coins/{id}/tickers
    pub id: String,
    pub exchange_ids: String,
    pub include_exchange_logo: bool,
    pub page: i32,            
    pub order: String,          // trust_score_desc (default), trust_score_asc and volume
    pub depth: bool,

}
pub struct History {
    endpoint: String,           /// /coins/{id}/history
    pub id: String, 
    pub date: String,            // dd-mm-yyyy format
    pub localization: bool,    // set to false to exlude localized languages in reponse ---- maybe just a bool or 

}
pub struct MarketChart {
    endpoint: String,           /// /coins/{id}/market_chart
    pub id: String,
    pub vs_currency: String, 
    pub days: String,            // data up to days ago 1,14,30,max 
    pub interval: String,           
}
pub struct MarketChartRange {
    endpoint: String,           /// /coins/{id}/market_chart/range 
    pub id: String,
    pub vs_currency: String,
    pub from: i64,
    pub to: i64,                // unix timestamp
}
pub struct StatusUpdates {
    endpoint: String,           /// /coins/{id}/status_updates
    pub id: String,
    pub per_page: i32,
    pub page: i32,
}
pub struct OHLC {
    endpoint: String,           /// /coins/{id}/ohlc
    pub id: String,
    pub vs_currency: String,
    pub days: String,           // 1/7/14/30/max --  only success is 1 and max
}


impl Default for List {
    fn default() -> List {
        List {
            endpoint: String::from("/coins/list"),
            include_platform: false, 
        }
    }
}
impl Default for Markets {
    fn default() -> Markets {
        Markets {
            endpoint: String::from("/coins/markets"),           /// /coins/markets
            vs_currency: String::from(""),                // required
            ids: String::from(""), 
            category: String::from(""),
            order: String::from("market_cap_desc"),
        // valid calues: market_cap_desc, gecko_desc, gecko_asc, market_cap_asc, market_cap_desc, volume_asc, volume_desc, id_asc, id_desc
            per_page: 100,
                // valid values: 1..250
            page: 1,
            sparkline: false,
            price_change_percentage: String::from(""),
        }
    }
} 
impl Default for Info {
    fn default() -> Info {
        Info {
            endpoint: String::from("/coins/ID"),
            id: String::from(""),
            localization: true,
            tickers: true,
            market_data: true,
            community_data: true,
            developer_data: true,
            sparkline: false
        }
    }
}
impl Default for Tickers {
    fn default() -> Tickers {
        Tickers {
            endpoint: String::from("/coins/ID/tickers"),
            id: String::from(""),
            exchange_ids: String::from(""),
            include_exchange_logo: true,
            page: 1,
            order: String::from("trust_score_desc"),
            depth: false,
        }
    }
}
impl Default for History {
    fn default() -> History {
        History {
            endpoint: String::from("/coins/ID/history"),
            id: String::from(""),
            date: String::from(""),
            localization: false, 
        }
    }
}
impl Default for MarketChart {
    fn default() -> MarketChart {
        MarketChart {
            endpoint: String::from("/coins/ID/market_chart"),
            id: String::from(""),
            vs_currency: String::from(""),
            days: String::from(""),
            interval: String::from(""),
        }
    }
}
impl Default for MarketChartRange {
    fn default() -> MarketChartRange {
        MarketChartRange {
            endpoint: String::from("/coins/ID/market_chart/range"),
            id: String::from(""),
            vs_currency: String::from(""),
            from: 0,
            to: 0      // prolly bad default values
        }
    }
}
impl Default for StatusUpdates {
    fn default() -> StatusUpdates {
        StatusUpdates {
            endpoint: String::from("/coins/ID/status_updates"),
            id: String::from(""),
            per_page: 100,
            page: 1,
        }
    }
}
impl Default for OHLC {
    fn default() -> OHLC {
        OHLC {
            endpoint: String::from("/coins/ID/ohlc"),
            id: String::from(""),
            vs_currency: String::from(""),
            days: String::from(""),
        }
    }
}

impl Route for List {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        let default: List = Default::default();
        let include_platform = self.format_query("include_platform".to_string(), self.include_platform, default.include_platform);
        let optional = vec![include_platform];
        self.collect_query_params(optional)
    }
}
impl Route for Markets {
    fn api_endpoint(&self) -> String {
        format!("{}", self.endpoint)
    }
    fn query_string(&self) -> String {
        let default: Markets = Default::default();
        let vs_currency = self.format_query("vs_currency".to_string(), &(self.vs_currency), &(default.vs_currency));
        let ids = self.format_query("ids".to_string(), &(self.ids), &(default.ids));
        let category = self.format_query("category".to_string(), &(self.category), &(default.category));
        let order = self.format_query("order".to_string(), &(self.order), &(default.order));
        let per_page = self.format_query("per_page".to_string(), self.per_page, default.per_page);
        let page = self.format_query("page".to_string(), self.page, default.page);
        let sparkline = self.format_query("sparkline".to_string(), self.sparkline, default.sparkline);
        let price_change_percentage = self.format_query("price_change_percentage".to_string(), &(self.price_change_percentage), &(default.price_change_percentage));
        let optional = vec![vs_currency, ids, category, order, per_page, page, sparkline, price_change_percentage];
        self.collect_query_params(optional)
    }
}
impl Route for Info {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: Info = Default::default();
        let localization = self.format_query("localization".to_string(), &(self.localization), &(default.localization));
        let tickers = self.format_query("tickers".to_string(), self.tickers, default.tickers);
        let market_data = self.format_query("market_data".to_string(), self.market_data, default.market_data);
        let community_data = self.format_query("community_data".to_string(), self.community_data, default.community_data);
        let developer_data = self.format_query("developer_data".to_string(), self.developer_data, default.developer_data);
        let sparkline = self.format_query("sparkline".to_string(), self.sparkline, default.sparkline);
        let optional = vec![localization, tickers, market_data, community_data, developer_data, sparkline];
        self.collect_query_params(optional)
    }
}
impl Route for Tickers {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: Tickers = Default::default();
        let exchange_ids = self.format_query("exchange_ids".to_string(), &(self.exchange_ids), &(default.exchange_ids));
        let include_exchange_logo = self.format_query("include_exchange_logo".to_string(), self.include_exchange_logo, default.include_exchange_logo);
        let page = self.format_query("page".to_string(), self.page, default.page);
        let order = self.format_query("order".to_string(), &(self.order), &(default.order));
        let depth = self.format_query("depth".to_string(), self.depth, default.depth);
        let optional = vec![exchange_ids, include_exchange_logo, page, order, depth];
        self.collect_query_params(optional)
    }
}
impl Route for History {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: History = Default::default();
        let date = self.format_query("date".to_string(), &(self.date), &(default.date));
        let localization = self.format_query("localization".to_string(), self.localization, default.localization);
        let optional = vec![date, localization];
        self.collect_query_params(optional)
    }
}
impl Route for MarketChart {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: MarketChart = Default::default();
        let vs_currency = self.format_query("vs_currency".to_string(), &(self.vs_currency), &(default.vs_currency));
        let days = self.format_query("days".to_string(), &(self.days), &(default.days));
        let interval = self.format_query("interval".to_string(), &(self.interval), &(default.interval));
        let optional = vec![vs_currency, days, interval];
        self.collect_query_params(optional)
    }
}
impl Route for MarketChartRange {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: MarketChartRange = Default::default();
        let vs_currency = self.format_query("vs_currency".to_string(), &(self.vs_currency), &(default.vs_currency));
        let from = self.format_query("from".to_string(), self.from, default.from);
        let to = self.format_query("to".to_string(), self.to, default.to);
        let optional = vec![vs_currency, from, to];
        self.collect_query_params(optional)
    }
}
impl Route for StatusUpdates {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: StatusUpdates = Default::default();
        let per_page = self.format_query("per_page".to_string(), self.per_page, default.per_page);
        let page = self.format_query("page".to_string(), self.page, default.page);
        let optional = vec![per_page, page];
        self.collect_query_params(optional)
    }
}
impl Route for OHLC {
    fn api_endpoint(&self) -> String {
        let endpoint = self.endpoint.replace("ID", &(self.id));
        format!("{}", endpoint)
    }
    fn query_string(&self) -> String {
        let default: OHLC = Default::default();
        let vs_currency = self.format_query("vs_currency".to_string(), &(self.vs_currency), &(default.vs_currency));
        let days = self.format_query("days".to_string(), &(self.days), &(default.days));
        let optional = vec![vs_currency, days];
        self.collect_query_params(optional)
    }
}
